package csbase.client.algorithms.tasks;

import java.awt.Window;

import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.desktop.RemoteTask;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import tecgraf.javautils.core.lng.LNG;

/**
 * Tarefa de validao de uma viso de um algoritmo ou parmetro.
 */
public class ViewValidationTask extends RemoteTask<ViewValidationResult> {

  /**
   * A viso a ser validada.
   */
  final private ViewValidator viewValidator;

  /**
   * O modo de validao ({@link ValidationMode#FULL} ou
   * {@link ValidationMode#ALLOW_EMPY_VALUES}).
   */
  final private ValidationMode mode;

  /**
   * Indica se deve exibir dilogo de erro se a validao falhar.
   */
  final private boolean complain;

  /**
   * Construtor.
   *
   * @param view a viso a ser validada.
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   * @param complain Indica se deve exibir dilogo de erro se a validao
   *        falhar.
   */
  public ViewValidationTask(ViewValidator view, ValidationMode mode,
    boolean complain) {
    super();
    this.viewValidator = view;
    this.mode = mode;
    this.complain = complain;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    ViewValidationResult validation = viewValidator.validate(mode);
    setResult(validation);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void afterTaskUI() {
    ViewValidationResult result = getResult();

    if (result != null && complain) {
      viewValidator.highlightValidationResult(result);
    }
  }

  /**
   * Mtodo de convenincia para chamada automtica da task.
   *
   * @param window janela dona da task.
   * @param view a viso a ser validada.
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   * @param complain Indica se deve exibir dilogo de erro se a validao
   *        falhar.
   *
   * @return o resultado da validao.
   */
  static final public ViewValidationResult runTask(Window window,
    ViewValidator view, ValidationMode mode, boolean complain) {

    ViewValidationTask task = new ViewValidationTask(view, mode, complain);

    final String className = ViewValidationTask.class.getSimpleName();
    final String title = LNG.get(className + ".title");
    final String msg = LNG.get(className + ".msg");
    if (task.execute(window, title, msg)) {
      return task.getResult();
    }
    else if (task.wasCancelled()) {
      LocalizedMessage message = new LocalizedMessage(ViewValidationTask.class,
        "cancelled");
      return new ViewValidationResult(new ValidationError(message), view);
    }
    else {
      Exception error = task.getError();
      error.printStackTrace();
      LocalizedMessage message = new LocalizedMessage(ViewValidationTask.class,
        "error", new Object[] { error.getMessage() });
      return new ViewValidationResult(new ValidationError(message), view);
    }
  }
}
