package csbase.client.applications.flowapplication.graph.actions;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.geom.Point2D;
import java.util.Set;

import csbase.client.applications.ApplicationImages;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.filters.AddNodePopupActionFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.ParametersTransferable;
import csbase.client.applications.flowapplication.messages.CopyMessage;
import csbase.exception.algorithms.ParameterNotFoundException;

/**
 * Ao de menu popup que copia os valores dos parmetros de um n de fluxo de
 * algoritmos.
 *
 * @author lmoreira
 */
public final class CopyParameterValuesAction extends GraphElementAction {
  /**
   * Cria a ao.
   *
   * @param node O n (No aceita {@code null}).
   */
  public CopyParameterValuesAction(final GraphNode node) {
    super(node, CopyParameterValuesAction.class.getName() + ".name",
      ApplicationImages.ICON_COPY_16);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(final ActionEvent e) {
    final GraphNode node = (GraphNode) getElement();
    /* Para possibilitar um CTRL V no notepad, por exemplo */
    saveToClipboard(node);
    /* Para copiar para outro node */
    new CopyMessage(new ParametersTransferable(node)).sendVS(node);
  }

  /**
   * Cria o filtro que cadastra a ao no menu popup.
   *
   * @param workspace A rea de trabalho (No aceita {@code null}).
   *
   * @return O filtro.
   */
  public static WorkspaceFilter createFilter(final Workspace workspace) {
    if (workspace == null) {
      throw new IllegalArgumentException("O parmetro workspace est nulo.");
    }
    return new AddNodePopupActionFilter(workspace) {

      @Override
      protected Action createAction(final GraphNode graphNode,
        final Point2D point) {
        return new CopyParameterValuesAction(graphNode);
      }

    };
  }

  /**
   * Salvo os valores dos parmetros no clipboard APENAS para o caso do usurio
   * precisar "colar" como texto, manualmente.
   *
   * @param node O n.
   */
  private void saveToClipboard(final GraphNode node) {
    final Set<String> parameterNames = node.getParameterNames();
    Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    StringBuilder builder = new StringBuilder();
    for (final String parameterName : parameterNames) {
      String parameterLabel;
      String parameterValue;
      String parameterType;
      try {
        parameterLabel = node.getParameterLabel(parameterName);
        parameterValue = node.getParameterValue(parameterName);
        parameterType = node.getParameterType(parameterName);
      }
      catch (final ParameterNotFoundException e) {
        /* Apenas no faz a cpia - o erro vai surgir na prxima etapa */
        return;
      }
      builder.append(parameterLabel);
      builder.append(" = ");
      builder.append(parameterValue);
      builder.append(" (");
      builder.append(parameterType);
      builder.append(" )");
      builder.append("\n");
    }
    clipboard.setContents(new StringSelection(builder.toString()), null);
  }

}
