package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.RetrieveUsersTask;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.client.util.user.UserDialog;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Ao que remove usurios compartilhados em projetos.
 *
 * @author jnlopes
 *
 */
public class RemoveUsersAction extends ProjectsManagerAction {

  /**
   * Construtor
   *
   * @param projectsManager A aplicao
   */
  public RemoveUsersAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() == 0) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final List<UserOutline> usersInProjects = getAllUsersInProjects(projects);

    final String title = getString("RemoveUsersAction.selection.title");
    final Set<Object> ids =
      UserDialog.showUserSelectionDialog(projectsManager.getApplicationFrame(),
        usersInProjects, true, title);

    if (ids == null) {
      return;
    }

    runTask(projects, ids);

    projectsManager.refreshProjectsTable();
    projectsManager.refreshInfoPanel();
  }

  /**
   * Roda e trata a execuo da task.
   *
   * @param projects Lista de projetos vlidos para a operao.
   * @param ids Conjunto de ids dos usurios a serem adicionados aos projetos
   *        vlidos.
   * @throws Exception em caso de erro.
   */
  private void runTask(List<ProjectsManagerData> projects, Set<Object> ids)
    throws Exception {

    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    final RemoveUsersTask rut =
      new RemoveUsersTask(projectsManager, projects, ids);
    final ApplicationFrame frame = appFrame;
    rut.execute(frame, projectsManager.getName(),
      getString("RemoveUsersAction.message"));

    if (rut.wasCancelled()) {
      final String err = getString("RemoveUsersAction.cancelled.message");
      rut.showError(err);
      return;
    }

    if (rut.getStatus() != true) {
      final Exception exception = rut.getError();
      throw exception;
    }
  }

  /**
   * Retorna uma lista com todos os usurios que esto em um ou mais dos
   * projetos selecionados.
   *
   * @param projectList Lista de projetos.
   * @return Lista com todos os usurios que esto em um ou mais dos projetos
   *         selecionados.
   * @throws Exception em caso de erro.
   */
  private List<UserOutline> getAllUsersInProjects(
    List<ProjectsManagerData> projectList) throws Exception {
    final Set<Object> usersInProjects = new HashSet<Object>();
    for (ProjectsManagerData pmd : projectList) {
      final Set<Object> usersRO = pmd.getUsersRO();
      final Set<Object> usersRW = pmd.getUsersRW();
      if (usersRO != null) {
        usersInProjects.addAll(usersRO);
      }
      if (usersRW != null) {
        usersInProjects.addAll(usersRW);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame frame = projectsManager.getApplicationFrame();

    final RetrieveUsersTask rut =
      new RetrieveUsersTask(getProjectsManager(), usersInProjects);
    rut.execute(frame, "", "");

    if (rut.wasCancelled()) {
      final String err = getString("RemoveUsersAction.cancelled.message");
      rut.showError(err);
      return null;
    }

    if (rut.getStatus() != true) {
      final Exception exception = rut.getError();
      throw exception;
    }

    final List<UserOutline> users = rut.getResult();
    return users;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ProjectsManagerUI.REMOVE_SHARED;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();
      final ProjectsManagerScope scope = pmd.getScope();

      final boolean isAdmin = loggedUser.isAdmin();
      final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

      /*
       * Para que um usurio possa tornar um projeto pblico  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio seja o administrador ou o dono do projeto. -
       * O projeto tenha seu escopo compartilhado.
       *
       * TODO - Avaliar o que  melhor: Permitir que a ao seja realizada sobre
       * projetos privados / pblicos tornando-os imediatamente compartilhados
       * (o que pode pegar o usurio de surpresa) ou excluir estes projetos da
       * lista que ser processada (obrigando o usurio a torn-los
       * compartilhados antes de executar a ao). Atualmente est em vigor a
       * segunda estratgia, visto que  simples tornar compartilhado o escopo
       * de diversos projetos (basta tornar um projeto compartilhado, e copiar o
       * esquema de permisses aos outros).
       */
      if (spaceAllocation.isOpenable() && (isAdmin || isOwner)
        && scope == ProjectsManagerScope.SHARED) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("RemoveUsersAction.project.selection.denied.message")
        + getString("RemoveUsersAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("RemoveUsersAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("RemoveUsersAction.project.selection.some.denied.message")
        + getString("RemoveUsersAction.project.requirements.message");
      final int res =
        StatusDialog
        .showSomeOkDialog(
          projectsManager,
          someDeniedMessage,
          prjList,
          getString("RemoveUsersAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }
    return prjList;
  }
}

/**
 * Task que adiciona usurios a um conjunto de projetos (transformando o escopo
 * do projeto em "compartilhado" se este ainda no for).
 *
 * @author Tecgraf
 */
class RemoveUsersTask extends ProjectsManagerTask<Void> {

  /**
   * Lista de projetos aos quais os usurios sero adicionados.
   */
  final private List<ProjectsManagerData> projects;

  /**
   * Lista de usurios que sero adicionados aos projetos.
   */
  final private Set<Object> userIds;

  /**
   * Construtor.
   *
   * @param projectsManager A aplicao.
   * @param projects Lista de projetos aos quais os usurios sero adicionados.
   * @param userIds Lista de usurios que sero adicionados aos projetos.
   */
  public RemoveUsersTask(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projects, final Set<Object> userIds) {
    super(projectsManager);
    this.projects = projects;
    this.userIds = userIds;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    for (ProjectsManagerData pmd : projects) {
      CommonClientProject ccp = psi.openProject(pmd.getProjectId(), false);
      Set<Object> usersRO = pmd.getUsersRO();
      Set<Object> usersRW = pmd.getUsersRW();
      usersRO.removeAll(userIds);
      usersRW.removeAll(userIds);
      ccp.updateUsersRO(usersRO);
      ccp.updateUsersRW(usersRW);
    }
  }
}
