package csbase.client.applications.projectsmanager.models;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import csbase.logic.FileInfoSearchResult;

/**
 * Modelo de tabela para exibio do resultado da busca de arquivos.
 */
public class SearchResultTableModel extends AbstractTableModel {

  /**
   * ndice da coluna com o nmero da linha.
   */
  private static final int ROW_INDEX_COL = 0;

  /**
   * ndice da coluna com o nome do arquivo.
   */
  private static final int FILENAME_COL = 1;

  /**
   * ndice da coluna com o nome do projeto.
   */
  private static final int PRJNAME_COL = 2;

  /**
   * ndice da coluna com o path.
   */
  private static final int PATH_COL = 3;

  /**
   * ndice da coluna com o nome do dono do arquivo
   */
  private static final int OWNER_COL = 4;

  /**
   * Texto para o ttulo das colunas.
   */
  final private String[] colNames;

  /**
   * Lista com as informaes dos arquivos que foram obtidos na busca;
   */
  private List<FileInfoSearchResult> resultsList;

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return colNames.length;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    if (resultsList == null) {
      return 0;
    }
    return resultsList.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(final int col) {
    return colNames[col];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<String> getColumnClass(final int col) {
    return String.class;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(final int row, final int col) {
    return false;
  }

  /**
   * Limpa a lista com os resultados da busca.
   */
  public void resetList() {
    resultsList = new ArrayList<FileInfoSearchResult>();
    fireTableDataChanged();
  }

  /**
   * Inclui uma lista de valores ao final do resultado j existente.
   * 
   * @param newValues novos valores que sero includos na lista
   */
  public void addAll(final List<FileInfoSearchResult> newValues) {
    this.resultsList.addAll(newValues);
    fireTableDataChanged();
  }

  /**
   * Construtor.
   * 
   * @param colNames texto das colunas.
   */
  public SearchResultTableModel(final String[] colNames) {
    this.colNames = colNames;
    this.resultsList = new ArrayList<FileInfoSearchResult>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {

    if (columnIndex == ROW_INDEX_COL) {
      return "" + (rowIndex + 1);
    }

    // obtm o elemento associado a uma linha da tabela
    final FileInfoSearchResult resultLine = resultsList.get(rowIndex);

    if (columnIndex == FILENAME_COL) {
      return resultLine.getFileName();
    }

    if (columnIndex == PRJNAME_COL) {
      return resultLine.getProjectName();
    }

    if (columnIndex == PATH_COL) {
      return resultLine.getFilePath();
    }

    if (columnIndex == OWNER_COL) {
      return resultLine.getOwnerName();
    }

    return null;
  }

  /**
   * Retorna o id do projeto da referida linha
   * 
   * @param rowIndex linha referida
   * @return project id do arquivo relacionado quela linha
   */
  public Object getProjectId(int rowIndex) {
    FileInfoSearchResult resultLine = resultsList.get(rowIndex);
    return resultLine.getProjectId();
  }

  /**
   * Verifica se o usurio tem permisso de escrita no projeto associado a uma
   * determinada linha.
   * 
   * @param rowIndex ndice da linha associada ao projeto
   * @return <code>true</code> se o usurio tem permisso de acesso
   */
  public boolean isWritable(int rowIndex) {
    return resultsList.get(rowIndex).isWritable();
  }
}
