/*
 * RoleProxy.java $Author: tatimf $ $Revision: 175519 $ - $Date: 2008-02-11
 * 18:35:52 -0200 (Mon, 11 Feb 2008) $
 */
package csbase.client.remote.srvproxies;

import java.awt.Window;
import java.util.List;
import java.util.Vector;

import javax.swing.JOptionPane;

import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.logic.Utilities;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Classe que modela um proxy para Role.
 * 
 * @author Julia
 */
public class RoleProxy {
  /**
   * Obtem a lista de perfis cadastradas. TODO - Mudar tipo de retorno para
   * {@code List<Role>}
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * 
   * @return o vetor de perfis.
   */
  public static Vector<Role> getAllRoles(Window owner, String title,
    String message) {
    RemoteTask<Vector<Role>> task = new RemoteTask<Vector<Role>>() {
      @Override
      public void performTask() throws Exception {
        setResult(new Vector<Role>(Role.getAllRoles()));
      }
    };
    if (task.execute(owner, title, message)) {
      Vector<Role> rolesVector = task.getResult();
      if ((rolesVector == null) || (rolesVector.size() <= 0)) {
        return new Vector<Role>();
      }
      return rolesVector;
    }
    return new Vector<Role>();
  }

  /**
   * Obtm um perfil com uma dada identificao.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param roleId o identificador do perfil.
   * 
   * @return o perfil requisitado.
   */
  public static Role getRole(Window owner, String title, String message,
    final Object roleId) {
    RemoteTask<Role> task = new RemoteTask<Role>() {
      @Override
      public void performTask() throws Exception {
        setResult(Role.getRole(roleId));
      }
    };
    if (task.execute(owner, title, message)) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Cria um perfil.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param info As informaes do novo perfil.
   * 
   * @return O perfil criado.
   */
  public static Role createRole(Window owner, String title, String message,
    final RoleInfo info) {
    RemoteTask<Role> task = new RemoteTask<Role>() {
      @Override
      public void performTask() throws Exception {
        setResult(Role.createRole(info));
      }
    };
    if (task.execute(owner, title, message)) {
      Role role = task.getResult();
      if (role == null) {
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_ROLE_ALREADY_EXISTS_ERROR"));
      }
      return role;
    }
    return null;
  }

  /**
   * Modifica um perfil.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param roleId o identificador do perfil a ser alterado.
   * @param info As informaes de modificao do perfil.
   * 
   * @return O perfil modificado.
   */
  public static Role modifyRole(Window owner, String title, String message,
    final Object roleId, final RoleInfo info) {
    RemoteTask<Role> task = new RemoteTask<Role>() {
      @Override
      public void performTask() throws Exception {
        setResult(Role.modifyRole(roleId, info));
      }
    };
    if (task.execute(owner, title, message)) {
      Role role = task.getResult();
      if (role == null) {
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_ROLE_ALREADY_EXISTS_ERROR"));
      }
      return role;
    }
    return null;
  }

  /**
   * Remove um perfil.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param roleId o identificador do perfil a ser removido.
   */
  public static void deleteRole(final Window owner, final String title,
    String message, final Object roleId) {
    class DeleteRoleRemoteTask extends RemoteTask<Void> {
      boolean removeRole = false;

      @Override
      public void performTask() throws Exception {
        Role.deleteRole(roleId);
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof AdministrationDeleteException) {
          String errorMsg = LNG.get("IAS_CONFIRM_ROLE_REMOVAL");
          int res = StandardDialogs.showYesNoDialog(owner, title, errorMsg);
          if (res == JOptionPane.YES_OPTION) {
            removeRole = true;
          }
        }
        else {
          super.handleError(error);
        }
      }
    }
    DeleteRoleRemoteTask deleteTask = new DeleteRoleRemoteTask();

    if (!deleteTask.execute(owner, title, message) && deleteTask.removeRole) {
      // Remoo no autorizada: existem usurios cadastrados com esse
      // perfil. Oferece-se ento a opo de retirar esse perfil da
      // configurao de todos os usurios que o estejam usando.
      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        public void performTask() throws Exception {
          // Procura por usurios que tenham esse perfil
          List<User> users = User.getAllUsers();
          for (int i = 0; i < users.size(); i++) {
            User user = users.get(i);
            Object[] roleIds = user.getRoleIds();
            for (int j = 0; j < roleIds.length; j++) {
              if (roleIds[j].equals(roleId)) {
                // Remove esse perfil do conjunto de perfis do usurio
                Object[] remainingRoleIds = Utilities.removeItemFromArray(
                  roleIds, j);
                UserInfo userInfo = user.getUserInfo();
                userInfo.setAttribute(User.ROLE_IDS, remainingRoleIds);
                User.modifyUser(user.getId(), userInfo);
                break;
              }
            }
          }

          // Como no existem mais usurios com esse perfil, pode-se
          // remov-lo
          Role.deleteRole(roleId);
        }
      };
      task.execute(owner, title, message);
    }
  }
}
