package csbase.client.util.csvpanel.columns;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.ClientLocalFile;
import csbase.client.applications.ApplicationImages;
import csbase.client.util.filechooser.ClientFileChooserSelectionMode;
import csbase.client.util.filechooser.ClientFileChooserType;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil
  .OperationResult;
import csbase.logic.ClientFile;

/**
 * Painel de escolha de arquivos locais e de projeto.
 */
public class ClientFileChooserTextPanel extends JPanel {

  /**
   * Arquivo selecionado.
   */
  private ClientFile selectedFile;

  /**
   * A janela dona do painel.
   */
  private Window owner;

  /**
   * O tipo do arquivo a ser selecionado. Vlido somente para o modo
   * {@link ClientFileChooserSelectionMode#FILES_ONLY}.
   */
  private String fileType;

  /**
   * O tipo da seleo (para abrir ou para salvar).
   */
  private ClientFileChooserType chooserType;

  /**
   * O modo de seleo (arquivo ou diretrio).
   */
  private ClientFileChooserSelectionMode mode;

  /**
   * O campo de texto com o nome do arquivo selecionado.
   */
  private JTextField configurationFileField;

  /**
   * Localizao dos arquivos selecionveis.
   */
  private ClientFileLocation location;

  /**
   * Observadores de seleo de arquivos.
   */
  private final List<FileChooserListener> listeners;

  /**
   * Define as possveis localizaes dos arquivos que podem ser selecionados.
   */
  public enum ClientFileLocation {
    /**
     * Somente arquivos locais.
     */
    LOCAL_ONLY, /**
     * Somente arquivos de projeto.
     */
    PROJECT_ONLY, /**
     * Qualquer arquivo.
     */
    ANY
  }

  /**
   * Construtor.
   *
   * @param initialValue seleo inicial.
   * @param owner a janela dona do painel.
   * @param fileType o tipo do arquivo ou nulo, se for uma seleo de
   * diretrios.
   * @param chooserType o tipo da seleo (para abrir ou para salvar).
   * @param mode modo de seleo (arquivo ou diretrio).
   * @param location localizao dos arquivos selecionveis.
   */
  public ClientFileChooserTextPanel(ClientFile initialValue, Window owner,
    String fileType, ClientFileChooserType chooserType,
    ClientFileChooserSelectionMode mode, ClientFileLocation location) {
    super();
    this.owner = owner;
    this.fileType = fileType;
    this.chooserType = chooserType;
    this.mode = mode;
    this.location = location;
    this.configurationFileField = new JTextField(30);
    this.listeners = new ArrayList<FileChooserListener>();
    createContents();
    setSelectedFile(initialValue);
  }

  /**
   * Construtor.
   *
   * @param owner a janela dona do painel.
   * @param fileType o tipo do arquivo ou nulo, se for uma seleo de
   * diretrios.
   * @param chooserType o tipo da seleo (para abrir ou para salvar).
   * @param mode modo de seleo (arquivo ou diretrio).
   * @param location localizao dos arquivos selecionveis.
   */
  public ClientFileChooserTextPanel(Window owner, String fileType,
    ClientFileChooserType chooserType, ClientFileChooserSelectionMode mode,
    ClientFileLocation location) {
    this(null, owner, fileType, chooserType, mode, location);
  }

  /**
   * Cria o contedo do painel.
   */
  private void createContents() {
    GBC gbc = new GBC().insets(5, 5, 5, 5).west().weighty(1);
    configurationFileField.setEditable(false);
    int gridx = 1;
    add(configurationFileField,
      new GBC(gbc).right(0).gridy(1).gridx(gridx).horizontal());
    gbc = gbc.right(1).left(0).gridy(1);
    if (location != ClientFileLocation.LOCAL_ONLY) {
      JButton browseButton = createBrowseProjectButton();
      add(browseButton, new GBC(gbc).gridx(++gridx));
    }
    if (location != ClientFileLocation.PROJECT_ONLY) {
      JButton browseLocalButton = createBrowseLocalButton();
      add(browseLocalButton, new GBC(gbc).gridx(++gridx));
    }
  }

  /**
   * Cria o boto de seleo de arquivos locais.
   *
   * @return o boto.
   */
  private JButton createBrowseLocalButton() {
    JButton browseLocalButton =
      GUIUtils.createImageButton(ApplicationImages.ICON_BROWSELOCALFILE_16);
    browseLocalButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        List<String> fileCodes = new ArrayList<String>();
        fileCodes.add(fileType);
        ClientLocalFile file = null;
        switch (chooserType) {
          case SAVE:
            if (mode == ClientFileChooserSelectionMode.FILES_ONLY) {
              file = ClientLocalFileChooserUtil
                .browseSingleFileInSaveMode(owner, fileCodes, fileType, null,
                  false, null);
            }
            break;
          case OPEN:
            if (mode == ClientFileChooserSelectionMode.FILES_ONLY) {
              file = ClientLocalFileChooserUtil
                .browseSingleFileInOpenMode(owner, fileCodes, null, false,
                  null);
            }
            else {
              file = ClientLocalFileChooserUtil
                .browseSingleDirectoryInOpenMode(owner, fileCodes, null, false,
                  null);
            }
        }
        if (file != null) {
          setSelectedFile(file);
          notifySelection(selectedFile);
        }
      }
    });
    return browseLocalButton;
  }

  /**
   * Cria o boto de seleo de arquivos de projeto.
   *
   * @return o boto.
   */
  private JButton createBrowseProjectButton() {
    JButton browseButton =
      GUIUtils.createImageButton(ApplicationImages.ICON_BROWSEFILE_16);
    browseButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        OperationResult result = null;
        switch (chooserType) {
          case SAVE:
            if (mode == ClientFileChooserSelectionMode.FILES_ONLY) {
              result = ClientProjectFileChooserUtil
                .browseSingleFileInSaveMode(owner, fileType, null, false, null,
                  true);
            }
            break;
          case OPEN:
            if (mode == ClientFileChooserSelectionMode.FILES_ONLY) {
              result = ClientProjectFileChooserUtil
                .browseSingleFileInOpenMode(owner, fileType, null, false);
            }
            else {
              result = ClientProjectFileChooserUtil
                .browseSingleDirectoryInOpenMode(owner, fileType, null, false,
                  null);
            }
            break;
        }
        if (result != null) {
          setSelectedFile(result.getClientProjectFile());
          notifySelection(selectedFile);
        }
      }
    });
    return browseButton;
  }

  /**
   * Obtm o arquivo selecionado pelo usurio ou nulo, caso o usurio no tenha
   * feito seleo ou cancelado a operao.
   *
   * @return selected file o arquivo seleciondo.
   */
  public ClientFile getSelectedFile() {
    return selectedFile;
  }

  /**
   * Atribui o arquivo selecionado.
   *
   * @param selectedFile o arquivo selecionado.
   */
  protected void setSelectedFile(ClientFile selectedFile) {
    this.selectedFile = selectedFile;
    String path = null;
    if (selectedFile != null) {
      path = selectedFile.getStringPath();
    }
    configurationFileField.setText(path);
  }

  /**
   * Notifica os observadores sobre a seleo do arquivo.
   *
   * @param file a seleo.
   */
  private void notifySelection(ClientFile file) {
    for (FileChooserListener listener : listeners) {
      listener.fileSelected(file);
    }
  }

  /**
   * Adiciona um observador de seleo ao painel.
   *
   * @param listener o observador.
   */
  public void addFileSelectionListener(FileChooserListener listener) {
    this.listeners.add(listener);
  }

  /**
   * Remove um observador de seleo ao painel.
   *
   * @param listener o observador.
   */
  public void removeFileSelectionListener(FileChooserListener listener) {
    this.listeners.remove(listener);
  }
}
