package csbase.client.applicationmanager;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JPopupMenu;

import csbase.logic.applicationservice.ApplicationCategory;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Classe utilitria para criao de menus com bas na configurao do gerente de
 * aplicaes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ApplicationManagerMenuUtilities {

  /**
   * Constri o menu de aplicaes para a barra de menu no topo da janela.
   * 
   * @return o menu de aplicaes
   */
  final static public JPopupMenu buildPopupMenu() {
    final JPopupMenu menu = new JPopupMenu();
    final List<JMenu> menuList = mountCategoriesMenuList();
    for (JMenu mn : menuList) {
      menu.add(mn);
    }
    return menu;
  }

  /**
   * Constri o menu de aplicaes para a barra de menu no topo da janela.
   * 
   * @param title Ttulo do menu
   * 
   * @return o menu de aplicaes
   */
  final static public JMenu buildMenu(String title) {
    final JMenu menu = new JMenu();
    menu.setText(title);

    final List<JMenu> menuList = mountCategoriesMenuList();
    for (JMenu mn : menuList) {
      menu.add(mn);
    }
    return menu;
  }

  /**
   * Monta uma lista de menus das categorias de aplicaes.
   * 
   * @return a lista.
   */
  static private List<JMenu> mountCategoriesMenuList() {
    final List<JMenu> menuList = new ArrayList<JMenu>();
    final ApplicationManager mgr = ApplicationManager.getInstance();
    if (mgr == null) {
      return menuList;
    }

    final Hashtable<String, ApplicationCategory> cts =
      mgr.getAllApplicationCategories();
    final Collection<ApplicationCategory> categories = cts.values();

    for (ApplicationCategory cat : categories) {
      if (cat.isShownAtApplicationMenu()) {
        final JMenu mn = buildCategoryMenu(cat);
        menuList.add(mn);
      }
    }

    // Ordenao de menus conforme seu ttulo (j internacionalizado)
    Collections.sort(menuList, new Comparator<JMenu>() {
      @Override
      public int compare(JMenu o1, JMenu o2) {
        final String title1 = o1.getText();
        final String title2 = o2.getText();
        return title1.compareToIgnoreCase(title2);
      }
    });
    return menuList;
  }

  /**
   * Constroi um menu com applicaes de uma categoria.
   * 
   * @param cat categoria
   * @return menu
   */
  static final private JMenu buildCategoryMenu(ApplicationCategory cat) {
    final JMenu menu = new JMenu();
    final ApplicationManager mgr = ApplicationManager.getInstance();
    if (mgr == null) {
      return menu;
    }
    final ArrayList<String> ids = cat.getApplicationIds();
    for (String id : ids) {
      final ApplicationRegistry reg = mgr.getApplicationRegistry(id);
      if (reg != null && reg.isShownAtApplicationMenu()) {
        final AbstractAction action = mgr.getApplicationAction(reg);
        menu.add(action);
      }
    }
    menu.setText(mgr.getCategoryName(cat));
    final byte[] ic = cat.getIcon();
    if (ic != null) {
      menu.setIcon(new ImageIcon(ic));
    }
    return menu;
  }
}
