/*
 * $Id: TreeUtil.java 173413 2016-05-10 19:56:56Z isabella $
 */
package csbase.client.applications.flowapplication;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.swing.JTree;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import tecgraf.javautils.gui.tree.Node;

/**
 * Rotina utilitrias para uso com <code>JTree</code>
 * 
 * @author Tecgraf/PUC-Rio
 * @see JTree
 */
public class TreeUtil {

  /**
   * Expande todos os ns da rvore
   * 
   * @param tree a rvore
   */
  static public void _collapseAll(final JTree tree) {
    final TreeNode root = (TreeNode) tree.getModel().getRoot();
    TreeUtil.doAll(tree, new TreePath(root), false);
  }

  /**
   * Expande/Collapsa todos os elementos de rvore.
   * 
   * @param tree rvore
   * @param parentPath path do pai
   * @param expand indicativo de expanso.
   */
  static private void doAll(final JTree tree, final TreePath parentPath,
    final boolean expand) {
    // Caminha nos filhos
    final TreeNode node = (TreeNode) parentPath.getLastPathComponent();
    if (node.getChildCount() >= 0) {
      for (final Enumeration<?> e = node.children(); e.hasMoreElements();) {
        final TreeNode n = (TreeNode) e.nextElement();
        final TreePath path = parentPath.pathByAddingChild(n);
        TreeUtil.doAll(tree, path, expand);
      }
    }
    // Expansion or collapse de baixo para cima.
    if (expand) {
      tree.expandPath(parentPath);
    }
    else {
      tree.collapsePath(parentPath);
    }
  }

  /**
   * Colapsa todos os ns da rvore
   * 
   * @param tree a rvore
   */
  static public void collapsedAll(final JTree tree) {
    TreeUtil.processNodes(tree, TreeUtil.collapseCallback);
  }

  /**
   * Expansion callback
   */
  final private static NodeCallback collapseCallback = new NodeCallback() {
    @Override
    public void postProcess(final JTree t, final TreePath parentPath,
      final Node node) {
      final TreePath path;
      if (parentPath == null) {
        path = new TreePath(node);
      }
      else {
        path = parentPath.pathByAddingChild(node);
      }
      t.collapsePath(path);
    }
  };

  /**
   * Consulta um path associado a um objeto
   * 
   * @param tree rvore
   * @param parentPath path do pai
   * @param node n
   * @param object objeto.
   * @return o path
   */
  private static TreePath hasObject(final JTree tree, final TreePath parentPath,
    final Node node, final Object object) {

    if (node.equals(object) && node.getParent().equals(((Node) object)
      .getParent())) {
      return parentPath.pathByAddingChild(node);
    }

    final TreePath path = parentPath.pathByAddingChild(node);
    for (Node childNode : node.getChildren()) {
      final TreePath tpath = TreeUtil.hasObject(tree, path, childNode, object);
      if (tpath != null) {
        return tpath;
      }
    }
    return null;
  }

  /**
   * Consulta um path associado a um objeto
   * 
   * @param tree rvore
   * @param object objeto.
   * @return o path (se existir)
   */
  private static TreePath hasRootObject(final JTree tree, final Object object) {
    final TreeModel model = tree.getModel();
    final Node root = (Node) model.getRoot();
    final TreePath rootPath = new TreePath(root);

    if (root.equals(object)) {
      return new TreePath(root);
    }

    for (Node node : root.getChildren()) {
      final TreePath tpath = TreeUtil.hasObject(tree, rootPath, node, object);
      if (tpath != null) {
        return tpath;
      }
    }
    return null;
  }

  /**
   * Faz a carga de ns expandidos
   * 
   * @param tree rvore
   * @param nodes lista de objetos associados.
   */
  public static void loadExpanded(final JTree tree, final List<Node> nodes) {
    for (Node node : nodes) {
      final TreePath path = TreeUtil.hasRootObject(tree, node);
      if (path != null) {
        tree.expandPath(path);
      }
    }
  }

  /**
   * Faz a carga de ns colapsados
   * 
   * @param tree rvore
   * @param nodes lista de objetos associados.
   */
  public static void loadCollapsed(final JTree tree, final List<Node> nodes) {
    for (Node node : nodes) {
      final TreePath path = TreeUtil.hasRootObject(tree, node);
      if (path != null) {
        tree.collapsePath(path);
      }
    }
  }

  /**
   * Abre o n raiz da rvore.
   * 
   * @param tree a rvore.
   */
  public static void openRootNode(final JTree tree) {
    final NodeCallback callback = new NodeCallback() {
      @Override
      final public void postProcess(final JTree t, final TreePath parentPath,
        final Node node) {
        final Node root = (Node) tree.getModel().getRoot();
        if (parentPath == null) {
          final TreePath rootPath = new TreePath(root);
          tree.expandPath(rootPath);
        }
      }
    };
    TreeUtil.processNodes(tree, callback);
  }

  /**
   * Abre todos os ns da rvore que forem instncia de {@link CategoryTreeNode}
   * 
   * @param tree a rvore.
   */
  public static void openCategoryTreeNode(final JTree tree) {
    final NodeCallback callback = new NodeCallback() {
      @Override
      final public void postProcess(final JTree t, final TreePath parentPath,
        final Node node) {
        final Node root = (Node) tree.getModel().getRoot();
        final TreePath path;
        if (node instanceof CategoryTreeNode) {
          if (parentPath == null) {
            final TreePath rootPath = new TreePath(root);
            tree.expandPath(rootPath);
          }
          else {
            path = parentPath.pathByAddingChild(node);
            tree.expandPath(path);
          }
        }
      }
    };

    TreeUtil.processNodes(tree, callback);
  }

  /**
   * Aplica callback a todos os ns de rvore.
   * 
   * @param tree rvore
   * @param callback callback
   */
  static public void processNodes(final JTree tree,
    final NodeCallback callback) {
    final Node root = (Node) tree.getModel().getRoot();
    final TreePath rootPath = new TreePath(root);
    callback.preProcess(tree, null, root);
    if (root.getChildren().size() >= 0) {
      for (Node e : root.getChildren()) {
        final Node node = e;
        TreeUtil.visitNodes(tree, rootPath, node, callback);
      }
    }
    callback.postProcess(tree, null, root);
  }

  /**
   * Salva em uma lista os ns expandidos de uma rvore.
   * 
   * @param tree a rvore
   * @return lista com os ns expandidos.
   */
  static List<Node> saveExpanded(final JTree tree) {
    final List<Node> list = new ArrayList<Node>();
    final NodeCallback callback = new NodeCallback() {
      @Override
      final public void postProcess(final JTree t, final TreePath parentPath,
        final Node node) {
        final TreePath path;
        if (parentPath == null) {
          path = new TreePath(node);
        }
        else {
          path = parentPath.pathByAddingChild(node);
        }

        if (t.isExpanded(path)) {
          list.add(node);
        }
      }
    };
    TreeUtil.processNodes(tree, callback);
    return list;
  }

  /**
   * Salva em uma lista os ns colapsados de uma rvore.
   * 
   * @param tree a rvore
   * @return lista com os ns colapsados.
   */
  public static List<Node> saveCollapsed(final JTree tree) {
    final List<Node> list = new ArrayList<Node>();
    final NodeCallback callback = new NodeCallback() {
      @Override
      final public void postProcess(final JTree t, final TreePath parentPath,
        final Node node) {
        final TreePath path;
        if (parentPath == null) {
          path = new TreePath(node);
        }
        else {
          path = parentPath.pathByAddingChild(node);
        }

        if (t.isCollapsed(path)) {
          list.add(node);
        }
      }
    };
    TreeUtil.processNodes(tree, callback);
    return list;
  }

  /**
   * Faz visita recursiva aos ns de rvore
   * 
   * @param tree rvore
   * @param parentPath path do pai
   * @param node n
   * @param callback callback a ser aplicada aos ns.
   */
  static private void visitNodes(final JTree tree, final TreePath parentPath,
    final Node node, final NodeCallback callback) {
    callback.preProcess(tree, parentPath, node);
    final TreePath path = parentPath.pathByAddingChild(node);
    if (node.getChildren().size() >= 0) {
      for (Node e : node.getChildren()) {
        final Node childNode = e;
        TreeUtil.visitNodes(tree, path, childNode, callback);
      }
    }
    callback.postProcess(tree, parentPath, node);
  }

}
