/*
 * $Id: ImageViewerEffectDialog.java 146656 2013-11-07 17:13:39Z mjulia $
 */
package csbase.client.applications.imageviewer.actions.effects;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.image.BufferedImage;

import javax.swing.AbstractAction;
import javax.swing.BoundedRangeModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationComponentDialog;
import csbase.client.applications.imageviewer.ImageViewer;
import csbase.client.applications.imageviewer.effects.AbstractEffect;
import csbase.client.util.ClientUtilities;

/**
 * Dilogo para efeitos.
 * 
 * @author Tecgraf/PUC-Rio
 * @param <T> efeito
 */
public class ImageViewerEffectDialog<T extends AbstractEffect> extends
  ApplicationComponentDialog<ImageViewer> {

  /**
   * Ao
   */
  final private T effect;

  /**
   * Preview de imagem original
   */
  final JLabel srcLabel = new JLabel();

  /**
   * Preview de imagem preview
   */
  final JLabel dstLabel = new JLabel();

  /**
   * Boto de Ok
   */
  final private JButton okButton = new JButton();

  /**
   * Boto de Reset
   */
  final private JButton resetButton = new JButton();

  /**
   * Construtor
   * 
   * @param application aplicao
   * @param effect efeito
   */
  ImageViewerEffectDialog(ImageViewer application, T effect) {
    super(application);
    this.effect = effect;
    this.effect.setPreviewLabel(dstLabel);

    setLayout(new BorderLayout());
    final JPanel paramPanel = effect.getParameterPanel();
    if (paramPanel != null) {
      add(paramPanel, BorderLayout.NORTH);
      BorderUtil.setLinedBorder(paramPanel);
    }
    add(createPreviewPanel(), BorderLayout.CENTER);
    add(createButtonsPanel(), BorderLayout.SOUTH);
    setMinimumSize(new Dimension(450, 400));
    setSize(new Dimension(700, 500));
  }

  /**
   * Monta o painel de preview.
   * 
   * @return o painel
   */
  private Component createPreviewPanel() {
    dstLabel.setVerticalTextPosition(JLabel.BOTTOM);
    dstLabel.setHorizontalTextPosition(JLabel.CENTER);
    dstLabel.setHorizontalAlignment(SwingConstants.CENTER);
    dstLabel.setVerticalAlignment(SwingConstants.CENTER);

    srcLabel.setVerticalTextPosition(JLabel.BOTTOM);
    srcLabel.setHorizontalTextPosition(JLabel.CENTER);
    srcLabel.setHorizontalAlignment(SwingConstants.CENTER);
    srcLabel.setVerticalAlignment(SwingConstants.CENTER);

    final JScrollPane srcPane = new JScrollPane(srcLabel);
    BorderUtil.setLowBorder(srcPane);

    final JScrollPane dstPane = new JScrollPane(dstLabel);
    BorderUtil.setLowBorder(dstPane);

    final JScrollBar vBar = srcPane.getVerticalScrollBar();
    final BoundedRangeModel vModel = vBar.getModel();
    dstPane.getVerticalScrollBar().setModel(vModel);

    final JScrollBar hBar = srcPane.getHorizontalScrollBar();
    final BoundedRangeModel hModel = hBar.getModel();
    dstPane.getHorizontalScrollBar().setModel(hModel);

    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    panel.add(srcPane, new GBC(0, 0).both().weights(10.0, 10.0));
    panel.add(new JLabel(), new GBC(1, 0).both().weights(1.0, 1.0));
    panel.add(dstPane, new GBC(2, 0).both().weights(10.0, 10.0));

    BorderUtil.setLinedBorder(panel);
    return panel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean b) {
    final ImageViewer application = getImageViewer();
    final BufferedImage srcImg = application.getCurrentBufferedImage();
    if (srcImg == null) {
      return;
    }
    final ImageIcon imageIcon = new ImageIcon(srcImg);
    srcLabel.setIcon(imageIcon);
    okButton.setVisible(true);
    resetButton.setVisible(true);
    effect.resetParameters();
    super.setVisible(b);
  }

  /**
   * Consulta a aplicao.
   * 
   * @return a aplicao
   */
  private ImageViewer getImageViewer() {
    final ImageViewer application = getApplication();
    return application;
  }

  /**
   * Monta o painel de botes.
   * 
   * @return o painel
   */
  private JPanel createButtonsPanel() {
    final ImageViewer application = getImageViewer();
    final ImageViewerEffectDialog<T> dialog = ImageViewerEffectDialog.this;
    final AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        dialog.setVisible(false);
      }
    };
    final JButton closeButton = new JButton(cancelAction);
    closeButton.setText(getString("ImageViewerEffectDialog.cancel.button"));

    final AbstractAction okAction = new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        try {
          effect.applyImageTransformation();
        }
        catch (Exception ex) {
          final String title = dialog.getTitle();
          final String message = ex.getMessage();
          StandardDialogs.showErrorDialog(dialog, title, message);
        }
        dialog.setVisible(false);
        application.setSaveNeeded(true);
      }
    };
    okButton.setAction(okAction);
    okButton.setText(getString("ImageViewerEffectDialog.ok.button"));

    final AbstractAction resetAction = new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        effect.resetParameters();
      }
    };
    resetButton.setAction(resetAction);
    resetButton.setText(getString("ImageViewerEffectDialog.reset.button"));

    // Montagem do painel.
    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.setLayout(new FlowLayout(FlowLayout.CENTER));
    buttonsPanel.add(closeButton);
    buttonsPanel.add(resetButton);
    buttonsPanel.add(new JPanel());
    buttonsPanel.add(new JPanel());
    buttonsPanel.add(new JPanel());
    buttonsPanel.add(okButton);
    final JButton[] buttons =
      new JButton[] { closeButton, resetButton, okButton };
    ClientUtilities.adjustEqualSizes(buttons);
    BorderUtil.setLinedBorder(buttonsPanel);
    return buttonsPanel;
  }

  /**
   * Atualizao do preview.
   */
  final protected void initPreview() {
    final boolean updated = effect.updatePreview();
    if (updated) {
      okButton.setEnabled(true);
      resetButton.setEnabled(true);
    }
    else {
      okButton.setEnabled(false);
      resetButton.setEnabled(false);
    }
  }
}
