package csbase.client.applications.projectsmanager.dialogs;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applicationmanager.resourcehelpers.ApplicationUtils;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.SearchResultTableModel;
import csbase.client.applications.projectsmanager.panels.ProjectSharingInfoPanel;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.FileInfoSearchResult;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Dilogo para pesquisa de arquivos em projetos.
 */
public class SearchFilesDialog extends ProjectsManagerDialog {

  /**
   * Cor de fundo para linhas correspondentes a projetos sem permisso de
   * escrita.
   */
  private static final Color READONLY_BG = UIManager
    .getColor("Panel.background");

  /**
   * Cor de fundo para linhas correspondentes a projetos com permisso de
   * escrita.
   */
  private static final Color READWRITE_BG = UIManager
    .getColor("Table.background");

  /**
   * Cor de fundo para linhas selecionadas.
   */
  private static final Color SELECTION_BG = UIManager
    .getColor("Table.selectionBackground");

  /**
   * Cor de fundo para linhas selecionadas quando o usurio no tem permisso de
   * escrita no projeto correspondente.
   */
  private static final Color SELECTION_BG_DARK = new Color(154, 177, 199);

  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> selectedProjectList;

  /**
   * Lista de TODOS os projetos do usurio.
   */
  private final List<ProjectsManagerData> allProjectList;

  /**
   * Radio button para considerar TODOS os projetos ao fazer a busca.
   */
  private JRadioButton allProjectsButton;

  /**
   * Radio button para considerar os projetos selecionados ao fazer a busca.
   */
  private JRadioButton selectedProjectsButton;

  /**
   * Caixa de texto para nome do arquivo.
   */
  private JTextField nameTextField;

  /**
   * Label para exibir a informao de auxlio ao usurio, associado a caixa de
   * texto 'nameTextField' para entrada do nome do arquivo.
   */
  private JLabel infoNameTextField;

  /**
   * Mensagem de texto default para o label infoNameTextField
   */
  private final String infoDefaultMessage =
    getString("SearchFilesDialog.info.default");

  /**
   * Sufixo para tooltips das linhas correspondentes a projetos sem permisso de
   * escrita.
   */
  private final String READONLY_TOOLTIP =
    getString("SearchFilesDialog.tooltip.suffix.readOnly");

  /**
   * Mensagem de texto de alerta para o label infoNameTextField no caso de
   * entrada de uma expresso regular invlida.
   */
  private final String infoRegexErrorMessage =
    getString("SearchFilesDialog.info.regex.error");

  /**
   * Check-box ignorar maiculas e minsculas.
   */
  private JCheckBox ignoreCheckBox;

  /**
   * Check-box utilizar expresso regular.
   */
  private JCheckBox regexCheckBox;

  /**
   * Tabela para exibio do resultado da busca.
   */
  private SortableTable resultTable;

  /**
   * Boto para iniciar a busca de arquivos
   */
  private JButton searchButton;

  /**
   * Boto fechar
   */
  private JButton closeButton;

  /**
   * Indica qual  a opo default para o escopo da busca, considerando a
   * seleo de projetos que foi feita pelo usurio. Indica que o default 
   * buscar os arquivos na lista de TODOS os projetos filtrados. Se FALSE, o
   * default  buscar na lista dos projetos selecionados pelo usurio.
   */
  private final boolean useAllProjectsList;

  /**
   * Barra de status
   */
  private StatusBar statusBar;

  /**
   * Construtor.
   * 
   * @param projectsManager - A aplicao.
   * @param selectedProjectList - Lista de projetos selecionados.
   * @param allProjectList - Lista de todos os projetos do usurio.
   * @param useAllProjectsList - indica que a busca ser feita em todos os
   *        projetos.
   */
  public SearchFilesDialog(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> selectedProjectList,
    final List<ProjectsManagerData> allProjectList,
    final boolean useAllProjectsList) {
    super(projectsManager);
    this.selectedProjectList = selectedProjectList;
    this.allProjectList = allProjectList;
    this.useAllProjectsList = useAllProjectsList;
    this.statusBar = new StatusBar();
    buildPanel();
    statusBar.showStatusBar();
    this.setTitle(getString("SearchFilesDialog.title"));
    nameTextField.requestFocusInWindow();
  }

  /**
   * Monta o painel principal.
   */
  private void buildPanel() {
    this.setLayout(new GridBagLayout());
    int y = 0;
    this.add(buildProjectsPanel(), new GBC(0, y++).horizontal());
    add(new JSeparator(SwingConstants.HORIZONTAL), new GBC(0, y++).fillx()
      .insets(10));
    this.add(buildInputPanel(), new GBC(0, y++).horizontal().insets(0, 10, 0,
      10));
    this.add(buildResultScrollPane(), new GBC(0, y++).both().insets(10, 10, 0,
      10));
    this.add(statusBar, new GBC(0, y++).horizontal().insets(0, 10, 10, 10)
      .bottom(10).top(5));
    this.add(buildButtonPanel(), new GBC(0, y++).horizontal().bottom(5));
  }

  /**
   * Monta o painel para seleo do escopo onde ser feita a busca.
   * 
   * @return o painel
   */
  private JPanel buildProjectsPanel() {

    final JPanel panel = new JPanel(new GridBagLayout());
    String label;
    if (getProjectsManager().isFilterOn()) {
      label = getString("SearchFilesDialog.radio.filteredProjects");
    }
    else {
      label = getString("SearchFilesDialog.radio.allProjects");
    }
    allProjectsButton = new JRadioButton(label, useAllProjectsList);
    selectedProjectsButton =
      new JRadioButton(getString("SearchFilesDialog.radio.selectedProjects"),
        !useAllProjectsList);
    selectedProjectsButton.setEnabled(!useAllProjectsList);

    final ButtonGroup group = new ButtonGroup();
    group.add(allProjectsButton);
    group.add(selectedProjectsButton);

    GBC gbc = new GBC(0, 0).northwest().insets(10, 10, 0, 10);
    panel.add(allProjectsButton, gbc);

    gbc = new GBC(0, 1).northwest().insets(0, 10, 0, 10);
    panel.add(selectedProjectsButton, gbc);

    final ProjectsManager projectsManager = getProjectsManager();
    final ProjectSharingInfoPanel projectsPanel =
      new ProjectSharingInfoPanel(projectsManager, selectedProjectList, true,
        false);
    gbc =
      new GBC(0, 2).northwest().insets(5, 30, 10, 10).horizontal().weightx(1.0);
    panel.add(projectsPanel, gbc);

    return panel;

  }

  /**
   * Cria o painel de entrada dos dados para realizao da busca.
   * 
   * @return o painel.
   */
  private JPanel buildInputPanel() {
    final JPanel panel = new JPanel(new GridBagLayout());

    // label
    GBC gbc = new GBC(0, 0).west();
    panel.add(new JLabel(getString("SearchFilesDialog.fileName")), gbc);

    // campo
    gbc = new GBC(1, 0).west().insets(5, 5, 0, 0).horizontal();
    nameTextField = new JTextField(10);
    nameTextField.addKeyListener(new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        if (regexCheckBox.isSelected()) {
          checkRegex();
        }
      }
    });
    nameTextField.addFocusListener(new FocusListener() {
      @Override
      public void focusGained(FocusEvent e) {
        nameTextField.selectAll();
      }

      @Override
      public void focusLost(FocusEvent e) {
        nameTextField.select(0, 0);
      }
    });

    panel.add(nameTextField, gbc);
    /*
     * faz com que o foco comece no campo de texto
     */
    SwingThreadDispatcher.invokeLater(new Runnable() {

      @Override
      public void run() {
        nameTextField.requestFocusInWindow();
      }
    });

    // dica
    gbc = new GBC(1, 1).northwest().insets(0, 5, 0, 0);
    infoNameTextField = new JLabel(infoDefaultMessage);
    Font tipFont =
      infoNameTextField.getFont().deriveFont(Font.PLAIN,
        infoNameTextField.getFont().getSize() - 1);
    infoNameTextField.setFont(tipFont);
    panel.add(infoNameTextField, gbc);

    // checkbox maisculas/minsculas
    gbc = new GBC(2, 0).northwest().insets(5, 10, 0, 0);
    ignoreCheckBox =
      new JCheckBox(getString("SearchFilesDialog.check.ignore"), true);
    panel.add(ignoreCheckBox, gbc);

    // checkbox regex
    gbc = new GBC(2, 1).northwest().insets(0, 10, 0, 0);
    regexCheckBox = new JCheckBox(getString("SearchFilesDialog.check.regex"));
    regexCheckBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        checkInput();
      }
    });
    panel.add(regexCheckBox, gbc);

    // boto "buscar"
    gbc = new GBC(3, 0).east().insets(5, 10, 0, 0);
    searchButton =
      new JButton(new AbstractAction(
        getString("SearchFilesDialog.searchButton")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          searchFiles();
        }
      });
    panel.add(searchButton, gbc);

    /*
     * configuramos ENTER no campo com a regex como atalho para a busca
     */
    KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
    nameTextField.getInputMap().put(keyStroke, "doSearch");
    nameTextField.getActionMap().put("doSearch", searchButton.getAction());

    return panel;
  }

  /**
   * Valida a sintaxe do texto fornecido como entrada, no caso de expresso
   * regular. De acordo o status deste texto, habilita ou desabilita o boto de
   * busca. Alm disso, modifica o texto associado ao label informativo para
   * sinalizar: expresso regular invlida (mensagem de erro) ou expresso
   * regular vlida (texto vazio).
   */
  private void checkRegex() {
    final String text = nameTextField.getText();
    boolean validPattern = true;

    try {
      Pattern.compile(text);
    }
    catch (PatternSyntaxException pse) {
      validPattern = false;
    }

    if (validPattern) {
      // expresso regular vlida, nenhuma mensagem
      infoNameTextField.setText("");
    }
    else {
      // expresso regular invlida, mensagem de alerta
      infoNameTextField.setText(infoRegexErrorMessage);
    }
    searchButton.setEnabled(validPattern);

  }

  /**
   * Atualiza a mensagem informativa sobre o texto e o status do boto de busca
   * de acordo com a nova opo que foi selecionada no check-box indicativo de
   * 'expresso regular'.
   */
  private void checkInput() {
    if (regexCheckBox.isSelected()) {
      checkRegex();
    }
    else {
      // neste caso no  necessrio validar a entrada (sempre vlida)
      infoNameTextField.setText(infoDefaultMessage);
      searchButton.setEnabled(true);
    }
  }

  /**
   * Cria o painel de exibio do resultado da busca.
   * 
   * @return o painel.
   */
  private JScrollPane buildResultScrollPane() {
    /*
     * Trata-se de uma SortableTable que mostra como tooltip, os valores clula,
     * pois em muitos casos eles so maiores que a largura da coluna
     */
    resultTable = new SortableTable() {
      @Override
      public String getToolTipText(MouseEvent e) {
        String ret = null;
        Point p = e.getPoint();
        int rowIndex = rowAtPoint(p);
        int colIndex = columnAtPoint(p);
        ret = (String) getValueAt(rowIndex, colIndex);
        SearchResultTableModel model =
          (SearchResultTableModel) resultTable.getModel();
        boolean writable =
          model.isWritable(resultTable.convertRowIndexToModel(rowIndex));
        if (!writable) {
          ret += READONLY_TOOLTIP;
        }
        return ret;
      }
    };
    resultTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          handleDoubleClick();
        }
      }
    });
    final String[] colNames =
      new String[] { "#", getString("SearchFilesDialog.column.file"),
          getString("SearchFilesDialog.column.project"),
          getString("SearchFilesDialog.column.path"),
          getString("SearchFilesDialog.column.owner") };

    resultTable.setModel(new SearchResultTableModel(colNames));
    resultTable.setNoSortStateEnabled(true);
    resultTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);

    final TableColumnModel columnModel = resultTable.getColumnModel();
    final TableColumn column0 = columnModel.getColumn(0);
    final TableColumn column1 = columnModel.getColumn(1);
    final TableColumn column2 = columnModel.getColumn(2);
    final TableColumn column3 = columnModel.getColumn(3);
    final TableColumn column4 = columnModel.getColumn(4);
    column0.setPreferredWidth(10);
    column1.setPreferredWidth(225);
    column2.setPreferredWidth(50);
    column3.setPreferredWidth(125);
    column4.setPreferredWidth(50);

    configureDefaultRenderer();

    final JScrollPane scrollTable = new JScrollPane(resultTable);
    return scrollTable;
  }

  /**
   * Configura o renderizador default para mudar a cor de fundo das linhas
   * associadas a projetos read-only.
   */
  private void configureDefaultRenderer() {
    resultTable.setDefaultRenderer(String.class,
      new DefaultTableCellRenderer() {

        /**
         * {@inheritDoc}
         */
        @Override
        public Component getTableCellRendererComponent(JTable table,
          Object value, boolean isSelected, boolean hasFocus, int row,
          int column) {
          Component component =
            super.getTableCellRendererComponent(table, value, isSelected,
              hasFocus, row, column);
          SearchResultTableModel model =
            (SearchResultTableModel) resultTable.getModel();
          boolean writable =
            model.isWritable(resultTable.convertRowIndexToModel(row));
          if (isSelected) {
            component
              .setBackground(writable ? SELECTION_BG : SELECTION_BG_DARK);
          }
          else {
            component.setBackground(writable ? READWRITE_BG : READONLY_BG);
          }
          return component;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isOpaque() {
          return true;
        }
      });
  }

  /**
   * Cria o painel inferior com o boto fechar.
   * 
   * @return o painel inferior.
   */
  private JPanel buildButtonPanel() {

    final JPanel panel = new JPanel();

    closeButton = new JButton(getString("SearchFilesDialog.closeButton"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(java.awt.event.ActionEvent e) {
        SearchFilesDialog.this.dispose();
      }
    });
    panel.add(closeButton);

    return panel;
  }

  /**
   * Realiza a pesquisa de arquivos nos projetos selecionados.
   */
  private void searchFiles() {

    final String text = nameTextField.getText();
    final boolean isCaseInsensitive = ignoreCheckBox.isSelected();
    final boolean isRegex = regexCheckBox.isSelected();

    // limpa o modelo para retirar os valores da busca anterior
    final SearchResultTableModel model =
      (SearchResultTableModel) resultTable.getModel();
    model.resetList();

    final ApplicationFrame appFrame =
      getProjectsManager().getApplicationFrame();
    final String taskTitle = getString("SearchFilesDialog.running.title");
    final String taskMessage = getString("SearchFilesDialog.running.message");

    // indica o sub-conjunto de projetos onde ser realizada a busca
    final List<ProjectsManagerData> projects;
    if (allProjectsButton.isSelected()) {
      projects = allProjectList;
    }
    else {
      projects = selectedProjectList;
    }

    // realiza uma busca em profundidade em todos os projetos selecionados
    int count = 0;
    for (final ProjectsManagerData project : projects) {

      /*
       * Projetos que esto aguardando alocao ou desalocao de espao no
       * podem ser abertos.
       */
      final ProjectSpaceAllocation projectSpaceAllocation =
        project.getProjectSpaceAllocation();
      if (projectSpaceAllocation == ProjectSpaceAllocation.WAITING_ALLOCATION
        || projectSpaceAllocation == ProjectSpaceAllocation.WAITING_DEALLOCATION) {
        continue;
      }

      final RemoteTask<Collection<FileInfoSearchResult>> task =
        new RemoteTask<Collection<FileInfoSearchResult>>() {
          @Override
          protected void performTask() throws Exception {
            final ProjectServiceInterface psi =
              ClientRemoteLocator.projectService;
            final Collection<FileInfoSearchResult> data =
              psi.getAllFileInfoSearchResult(project.getProjectId(), text,
                isCaseInsensitive, isRegex);
            setResult(data);
          }
        };

      final List<FileInfoSearchResult> filesList;
      if (task.execute(appFrame, taskTitle, String.format(taskMessage, project
        .getProjectName()), 0, Task.CANCEL_BUTTON)) {
        filesList = (List<FileInfoSearchResult>) task.getResult();
      }
      else {
        filesList = null;
      }

      if (task.wasCancelled()) {
        break;
      }

      // adiciona o resultado parcial no modelo
      if (filesList != null && !filesList.isEmpty()) {
        model.addAll(filesList);
        count = count + filesList.size();
      }

    }
    final String msg = getString("SearchFilesDialog.statusBar.message");
    statusBar.setStatus(String.format(msg, count));
  }

  /**
   * Trata o double click na tabela de itens buscados, abrindo o arquivo com a
   * aplicao adequada.
   */
  public void handleDoubleClick() {
    /*
     * Obtm o nome do arquivo, diretrio e id do projeto a partir da linha
     * selecionada
     */
    int selectedRow = resultTable.getSelectedRow();
    String fileName = (String) resultTable.getValueAt(selectedRow, 1);
    /*
     * O project id no est mostrado visualmente na tabela, mas  um resultado
     * j guardado no modelo. Usa-se a converso abaixo, pois o ndice da tabela
     * pode no se o ndice do modelo, aps, por exemplo, uma ordenao da
     * tabela
     */
    Object projectId =
      ((SearchResultTableModel) resultTable.getModel())
        .getProjectId(resultTable.convertRowIndexToModel(selectedRow));
    String dirPath = (String) resultTable.getValueAt(selectedRow, 3);

    /* Obtm o caminho do arquivo com seu nome em formato de array de strings */
    String[] filePath = ClientProjectFile.splitPath(dirPath + "/" + fileName);

    // Obtm o projeto corrente
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    /*
     * Se no houver projeto corrente aberto, abre o do arquivo escolhido. Se
     * for um projeto diferente do corrente, mostra dilogo perguntando se pode
     * fechar todas as janelas, e se sim, troca o projeto no desktop. Se no,
     * retorna e no manda abrir a aplicao.
     */
    if (project == null) {
      project = DesktopFrame.getInstance().openProject(projectId);
      DesktopFrame.getInstance().setCurrentProject(project);
    }
    else if (!project.getId().equals(projectId)) {
      int answer =
        StandardDialogs.showYesNoDialog(getProjectsManager()
          .getApplicationFrame(),
          getString("SearchFilesDialog.openProject.dialog.title"),
          getString("SearchFilesDialog.openProject.dialog.message"));
      if (answer != JOptionPane.YES_OPTION) {
        return;
      }
      project = DesktopFrame.getInstance().openProject(projectId);
      DesktopFrame.getInstance().setCurrentProject(project);
    }

    // Obtm o arquivo e manda rodar a aplicao adequada
    try {
      ClientProjectFile file = project.getFile(filePath);
      ApplicationUtils.runPreferredApp(this, file);
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(getProjectsManager()
        .getApplicationFrame(), e);
    }
  }
}
