/*
 * $Id: BasicDesktopFrame.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.client.desktop;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationManagerMenuUtilities;
import csbase.client.applicationmanager.ApplicationPanel;
import csbase.client.applicationmanager.ClearApplicationCacheAction;
import csbase.client.applications.ApplicationImages;
import csbase.client.ias.AdminFrame;
import csbase.client.ias.AdminModifyUserInfoDialog;
import csbase.client.ias.ModifyUserInfoDialog;
import csbase.client.ias.UserInfoDialog;
import csbase.client.kernel.ClientException;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.project.BasicProjectTreeFactory;
import csbase.client.project.ProjectLocationBar;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.ProjectTreeFactory;
import csbase.client.project.action.CommonProjectCreationAction;
import csbase.client.project.action.CommonProjectOpenAction;
import csbase.client.project.action.CommonProjectUpdateAction;
import csbase.client.project.action.ProjectCloseAction;
import csbase.client.project.action.ProjectDeleteAction;
import csbase.client.project.action.ProjectHistoryShowAction;
import csbase.client.project.action.ProjectTreeUpdateAction;
import csbase.client.project.action.ProjectUsersManagementAction;
import csbase.client.project.action.ShowHiddenFilesAction;
import csbase.client.project.action.ShowTreeFilterAction;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.ProjectAdminObserver;
import csbase.client.remote.srvproxies.ApplicationManagementProxy;
import csbase.logic.AlgorithmAdminPermission;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectUserEvent;
import csbase.logic.SGAAdminPermission;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationRegistry;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * A classe <code>BasicDesktopFrame</code> implementa a interface principal do
 * sistema no lado cliente. Ela prov a funcionalidade mnima comum a qualquer
 * sistema desenvolvido com base no CSBase. A configurao deste componente
 * (especfica para cada sistema) compreende: - o ttulo do desktop e a imagem a
 * ele associada - a localizao dos arquivos de propriedades que configuram as
 * aplicaes e os grupos de aplicaes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class BasicDesktopFrame extends DesktopFrame implements Observer {
  /**
   * Caminho do diretrio selecionado na rvore de projetos
   */
  private ProjectLocationBar locationBar;

  /**
   * Boto para selecionar projeto corrente (toolbar)
   */
  private JButton openProjButton;

  /**
   * Item de menu para abertura de projetos (do prprio usurio, compartilhados
   * com ele e, no caso do admin, de todos os usurios).
   */
  private JMenuItem openProjMenuItem;

  /**
   * Item de Menu para Trasnferncia de Arquivos
   */
  private JMenuItem fileTransferMenuItem;

  /**
   * Item de Menu para Remoo de projeto
   */
  private JMenuItem removeProjMenuItem;

  /**
   * Painel de aplicaes.
   */
  private ApplicationPanel applicationPanel;

  /** Id da aplicao Gerenciador de Algoritmos */
  private final static String ALGORITHMS_MANAGER_APPLICATION_ID =
    "algorithmsmanager";

  /**
   * Constri a interface principal do cliente.
   * 
   * @throws ClientException em caso de falha.
   */
  public BasicDesktopFrame() throws ClientException {
    this.createProjectTreeFactory();
    setTitle(Client.getInstance().getSystemName());
    /* Constri os painis do desktop, a barra de ferramentas e o menu */
    JFrame mainFrame = getDesktopFrame();

    final JComponent projectPanel = createProjectPanel();

    applicationPanel = new ApplicationPanel();
    applicationPanel.addProjectTreeListernerForLayoutAutoPersistency(
      projectTree);
    applicationPanel.addMouseListenerForLayoutMenu();
    applicationPanel.addMouseListenerForApplicationsMenu();

    JComponent mainPanel = composeMainPanel(projectPanel,
      createDirectoryContentsPanel(), createTreeFilterPanel(), applicationPanel,
      createNotificationPanel());
    JPanel firstPanel = new JPanel(new BorderLayout());
    firstPanel.add(createToolBar(), BorderLayout.NORTH);
    firstPanel.add(mainPanel, BorderLayout.CENTER);
    mainFrame.getContentPane().add(firstPanel);
    this.createDesktopMenuBar();
    /* Cadastra o desktop como observador de projetos */
    ProjectAdminObserver.getInstance().addObserver(this);

    /* Salva as configuraes do projeto */
    mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        shutdownDesktop();
      }
    });
  }

  /**
   * Obtm uma propriedade opcional da classe.
   * 
   * @param name nome da propriedade
   * @return uma propriedade opcional da classe
   */
  protected String getProperty(String name) {
    return this.getConfiguration().getOptionalProperty(name);
  }

  /**
   * Consulta o valor de applicationPanel
   * 
   * @return o valor
   */
  public final ApplicationPanel getApplicationPanel() {
    return applicationPanel;
  }

  /**
   * Cria o painel que contm a rvore de projetos.
   * 
   * @return o painel com a rvore de projetos
   * @throws ClientException caso ocorra algum problema na criao da rvore
   */
  private JComponent createProjectPanel() throws ClientException {
    final ProjectTreeFactory projectTreeFactory = ProjectTreeFactory
      .getInstance();
    final String title = getTitle();
    final DesktopComponentFrame desktopFrame = getDesktopFrame();
    projectTree = projectTreeFactory.makeProjectTree(desktopFrame, title);
    projectTree.addMouseListener();

    /* Listener para reao  mudana de projeto e seleo na rvore */
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      /* Mudana de projeto: atualiza caminho selecionado */
      @Override
      public void projectChanged(CommonClientProject proj) {
        applicationPanel.setProjectFlag(proj != null);
        treeFilterPanel.setVisible(false);
      }
    });
    JComponent scrollPane = new JScrollPane(projectTree.getTree());
    projectTreePanel = new JPanel(new BorderLayout());
    projectTreePanel.add(scrollPane, BorderLayout.CENTER);
    return projectTreePanel;
  }

  /**
   * Cria uma fbrica de rvores de projeto.
   */
  protected void createProjectTreeFactory() {
    new BasicProjectTreeFactory();
  }

  /**
   * Compe o painel principal, separando os diferentes subpainis com
   * SplitPanes.
   * 
   * @param proTreePanel painel com a rvore de projeto
   * @param dirContPanel painel com detalhes do diretrio selecionado
   * @param filterPanel painel de filtro da rvore de projeto
   * @param appPanel painel de aplicaes
   * @param notificationPanel painel de notificaes
   * @return o painel principal do desktop
   */
  private JComponent composeMainPanel(JComponent proTreePanel,
    JComponent dirContPanel, JComponent filterPanel, JComponent appPanel,
    JComponent notificationPanel) {
    /*
     * Compe um painel com a rvore de diretrios e os detalhes do diretrio
     * corrente, com "arrumao" vertical
     */
    locationBar = new ProjectLocationBar(projectTree);
    proTreePanel.add(locationBar, BorderLayout.NORTH);
    splitPaneBetweenTreeAndTable = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
      proTreePanel, dirContPanel);
    splitPaneBetweenTreeAndTable.setOneTouchExpandable(true);
    splitPaneBetweenTreeAndTable.setResizeWeight(0.75);
    JPanel panel = new JPanel(new BorderLayout());
    panel.add(treeFilterPanel, BorderLayout.NORTH);
    panel.add(splitPaneBetweenTreeAndTable, BorderLayout.CENTER);

    /* Compe este painel com o de aplicaes, com "arrumao" horizontal */
    splitPaneBetweenTreeAndApp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
      panel, appPanel);
    splitPaneBetweenTreeAndApp.setOneTouchExpandable(true);
    splitPaneBetweenTreeAndApp.setResizeWeight(0.0);

    /* Acrescenta o painel de notificaes, com "arrumao" vertical */
    splitPaneBetweenAppAndNotif = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
      splitPaneBetweenTreeAndApp, notificationPanel);
    splitPaneBetweenAppAndNotif.setOneTouchExpandable(true);
    splitPaneBetweenAppAndNotif.setResizeWeight(1.0);
    getDesktopFrame().setResizable(true);
    return splitPaneBetweenAppAndNotif;
  }

  /**
   * Cria a barra de ferramentas do desktop. Esse elemento  composto por botes
   * para a seleo (open) de um projeto, e para a configurao das janelas que
   * contm grupos de aplicaes. (opes: em cascata, vertical ou horizontal)
   * 
   * @return a barra de ferramentas
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    /* Abrir projeto */
    openProjButton = new JButton(new CommonProjectOpenAction(projectTree));
    openProjButton.setIcon(ApplicationImages.ICON_OPEN_16);
    openProjButton.setText("");
    openProjButton.setEnabled(User.getLoggedUser().isAdmin() || userHasProject()
      || userHasOthersProject());
    toolBar.add(openProjButton);
    toolBar.addSeparator();

    /* Cascatear janelas */
    JButton cascadeButton = new JButton();
    cascadeButton.setIcon(ApplicationImages.ICON_WINCASCADE_16);
    cascadeButton.setToolTipText(LNG.get("desktop.groups.cascade"));
    cascadeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        applicationPanel.cascadeGroups();
      }
    });

    /* Janelas na vertical */
    JButton tileVertButton = new JButton();
    tileVertButton.setIcon(ApplicationImages.ICON_WINTILEVERTICAL_16);
    tileVertButton.setToolTipText(LNG.get("desktop.groups.tile_vertical"));
    tileVertButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        applicationPanel.tileGroupsVertical();
      }
    });

    /* Janelas na horizontal */
    JButton tileHorButton = new JButton();
    tileHorButton.setIcon(ApplicationImages.ICON_WINTILEHORIZONTAL_16);
    tileHorButton.setToolTipText(LNG.get("desktop.groups.tile_horizontal"));
    tileHorButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        applicationPanel.tileGroupsHorizontal();
      }
    });
    final JToggleButton showTreeFilterButton = new JToggleButton(
      new ShowTreeFilterAction(treeFilterPanel));
    showTreeFilterButton.setIcon(ApplicationImages.ICON_FILTER_16);
    showTreeFilterButton.setText("");
    showTreeFilterButton.setToolTipText(LNG.get(
      "BasicDesktopFrame.tooltip.showTreeFilterButton"));
    showTreeFilterButton.setEnabled(false);
    enableIfProjectOpened(showTreeFilterButton);
    treeFilterPanel.addListener(new TreeFilterPanelListener() {
      @Override
      public void visibilityChanged(boolean panelVisible) {
        showTreeFilterButton.setSelected(panelVisible);
      }
    });
    toolBar.add(cascadeButton);
    toolBar.add(tileVertButton);
    toolBar.add(tileHorButton);
    toolBar.addSeparator();
    toolBar.add(showTreeFilterButton);
    toolBar.addSeparator();

    JButton srvMonitorButton = toolBar.add(applicationPanel
      .getApplicationAction("sgamonitor"));
    srvMonitorButton.setToolTipText(LNG.get(
      "BasicDesktopFrame.tooltip.sgamonitor"));
    JButton cmdMonitorButton = toolBar.add(applicationPanel
      .getApplicationAction("commandmonitor"));
    cmdMonitorButton.setToolTipText(LNG.get(
      "BasicDesktopFrame.tooltip.commandmonitor"));
    toolBar.addSeparator();

    return toolBar;
  }

  /**
   * Constri a barra de menu do desktop.
   */
  private void createDesktopMenuBar() {
    boolean includeAlgorithmMenu = this.getConfiguration()
      .getOptionalBooleanProperty("algorithm.menu", true);
    boolean leadingConfigMenu = this.getConfiguration()
      .getOptionalBooleanProperty("leading.config.menu", false);
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(createProjectMenu());
    JMenu configMenu = createConfigurationMenu();
    if (leadingConfigMenu) {
      menuBar.add(configMenu);
    }
    JMenu appMenu = ApplicationManagerMenuUtilities.buildMenu(LNG.get(
      "desktop.applications"));
    if (appMenu != null) {
      menuBar.add(appMenu);
    }
    if (includeAlgorithmMenu) {
      menuBar.add(createAlgorithmMenu());
    }
    menuBar.add(createMonitoringMenu());
    if (!leadingConfigMenu) {
      menuBar.add(configMenu);
    }

    /*
     * Menu de administrao  criado apenas para usurio administrador ou com
     * permisso de administrar algoritmos e/ou sgas
     */
    User user = User.getLoggedUser();
    AlgorithmAdminPermission algoPerm = user.getPermission(
      AlgorithmAdminPermission.class);
    SGAAdminPermission sgaPerm = user.getPermission(SGAAdminPermission.class);
    if (user.isAdmin() || (algoPerm != null) || (sgaPerm != null)) {
      menuBar.add(createAdminMenu());
    }
    JMenu optionalMenu = createOptionalMenu();
    if (optionalMenu != null) {
      menuBar.add(optionalMenu);
    }
    setMenuBar(menuBar);
  }

  /**
   * Constri o menu para gerenciamente dos projetos do usurio.
   * 
   * @return o menu de gerenciamente de projetos.
   */
  private JMenu createProjectMenu() {
    JMenu menu = new JMenu(LNG.get("desktop.project"));
    boolean hasOwnProjects = userHasProject();
    boolean hasOtherProjects = User.getLoggedUser().isAdmin()
      || userHasOthersProject();

    // cone vazio para alinharmos pelo checkbox do filtro
    Icon emptyIcon = GUIUtils.createEmptyIcon(6, 6);

    /* Criao de projeto */
    JMenuItem createMenuItem = new JMenuItem(new CommonProjectCreationAction(
      projectTree));
    createMenuItem.setText(LNG.get("desktop.project.create"));
    createMenuItem.setIcon(emptyIcon);
    createMenuItem.setEnabled(true);
    menu.add(createMenuItem);

    /* Seleo de projeto */
    openProjMenuItem = new JMenuItem(new CommonProjectOpenAction(projectTree));
    openProjMenuItem.setText(LNG.get("desktop.project.open"));
    openProjMenuItem.setEnabled(User.getLoggedUser().isAdmin() || hasOwnProjects
      || hasOtherProjects);
    openProjMenuItem.setIcon(ApplicationImages.ICON_OPEN_16);
    menu.add(openProjMenuItem);

    /* Remoo do projeto */
    removeProjMenuItem = new JMenuItem(new ProjectDeleteAction(projectTree));
    removeProjMenuItem.setText(LNG.get("desktop.project.remove"));
    removeProjMenuItem.setIcon(emptyIcon);
    removeProjMenuItem.setEnabled(hasOwnProjects);
    menu.add(removeProjMenuItem);

    /* Transferncia de arquivos entre projetos */
    fileTransferMenuItem = menu.add(applicationPanel.getApplicationAction(
      "filetransfer"));
    fileTransferMenuItem.setText(LNG.get("desktop.project.file_transfer"));
    fileTransferMenuItem.setIcon(emptyIcon);
    fileTransferMenuItem.setEnabled(hasOwnProjects);
    menu.addSeparator();

    /* Obteno de Dados */
    JMenu getDataSubmenu = createDataRetrievalMenu();
    if (getDataSubmenu != null) {
      getDataSubmenu.setEnabled(false);
      getDataSubmenu.setIcon(emptyIcon);
      enableIfProjectOpened(getDataSubmenu);
      menu.add(getDataSubmenu);
    }

    /* Atualizao da rvore do projeto */
    JMenuItem updateTreeMenuItem = new JMenuItem(new ProjectTreeUpdateAction(
      projectTree));
    updateTreeMenuItem.setEnabled(false);
    updateTreeMenuItem.setIcon(emptyIcon);
    enableIfProjectOpened(updateTreeMenuItem);
    menu.add(updateTreeMenuItem);

    /* Ver histrico do projeto */
    JMenuItem historyViewMenuItem = new JMenuItem(new ProjectHistoryShowAction(
      projectTree));
    historyViewMenuItem.setText(LNG.get("desktop.project.history"));
    historyViewMenuItem.setIcon(emptyIcon);
    historyViewMenuItem.setEnabled(false);
    enableIfProjectOpened(historyViewMenuItem);
    menu.add(historyViewMenuItem);

    /* Atualizao de dados do projeto */
    JMenuItem updateMenuItem = new JMenuItem(new CommonProjectUpdateAction(
      projectTree));
    updateMenuItem.setText(LNG.get("desktop.project.modify"));
    updateMenuItem.setIcon(emptyIcon);
    updateMenuItem.setEnabled(false);
    enableIfProjectOpened(updateMenuItem);
    menu.add(updateMenuItem);

    /* Gerenciamento dos usurios de um projeto */
    JMenuItem usersManagementMenuItem = new JMenuItem(
      new ProjectUsersManagementAction(projectTree));
    usersManagementMenuItem.setText(LNG.get("desktop.project.access.control"));
    usersManagementMenuItem.setEnabled(false);
    usersManagementMenuItem.setIcon(emptyIcon);
    enableIfAdminProject(usersManagementMenuItem);
    menu.add(usersManagementMenuItem);

    /* Exibio do painel de filtro para a rvore de projetos */
    final JCheckBoxMenuItem showTreeFilterMenuItem = new JCheckBoxMenuItem(
      new ShowTreeFilterAction(treeFilterPanel));
    showTreeFilterMenuItem.setAccelerator(KeyStroke.getKeyStroke(
      "ctrl pressed F"));
    showTreeFilterMenuItem.setEnabled(false);
    enableIfProjectOpened(showTreeFilterMenuItem);
    treeFilterPanel.addListener(new TreeFilterPanelListener() {
      @Override
      public void visibilityChanged(boolean panelVisible) {
        showTreeFilterMenuItem.setSelected(panelVisible);
      }
    });
    menu.add(showTreeFilterMenuItem);

    /*
     * Controle da exibio de arquivos ocultos (que comeam com "."). O estado
     * do checkbox  obtido das preferncias do usurio.
     */
    final JCheckBoxMenuItem showHiddenFilesMenuItem = new JCheckBoxMenuItem(
      new ShowHiddenFilesAction(projectTree));

    PreferenceCategory dp = getDesktopPreferences();
    boolean showHiddenFilesPref = dp.getPreferenceAsBoolean(
      DesktopPref.SHOW_HIDDEN_FILES);
    showHiddenFilesMenuItem.setSelected(showHiddenFilesPref);

    showHiddenFilesMenuItem.setEnabled(false);
    enableIfProjectOpened(showHiddenFilesMenuItem);
    menu.add(showHiddenFilesMenuItem);

    /* Fechar projeto */
    JMenuItem closeMenuItem = new JMenuItem(new ProjectCloseAction(
      projectTree));
    closeMenuItem.setEnabled(false);
    closeMenuItem.setIcon(emptyIcon);
    enableIfProjectOpened(closeMenuItem);
    menu.add(closeMenuItem);
    menu.addSeparator();

    /* Sair do sistema */
    JMenuItem exitMenuItem = menu.add(LNG.get("desktop.exit"));
    exitMenuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        shutdownDesktop();
      }
    });
    exitMenuItem.setIcon(emptyIcon);
    menu.add(exitMenuItem);
    return menu;
  }

  /**
   * Cria o Menu para abertura rpida de projetos abertos recentemente pelo
   * usurio ("Abrir Recentes..."). Cadastra um listener em ProjectTree para que
   * seja atualizado a cada mudana de projeto.
   * 
   * @return O menu
   */
  protected JMenu createRecentProjectsMenu() {
    /* Item para abertura rpida de projetos recentes */
    final JMenu recentProjectsMenu = new JMenu(LNG.get(
      "DesktopFrame.menu.project.recents"));
    updateRecentProjectsMenu(recentProjectsMenu);

    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        updateRecentProjectsMenu(recentProjectsMenu);
      }
    });
    return recentProjectsMenu;
  }

  /**
   * Atualiza os Itens do Menu "Abrir Recentes..."
   * 
   * @param recentProjectsMenu o menu
   */
  private void updateRecentProjectsMenu(JMenu recentProjectsMenu) {
    recentProjectsMenu.removeAll();

    AbstractAction[] openProjectActions = getRecentProjectOpenActions();
    for (int i = 0; i < openProjectActions.length; i++) {
      recentProjectsMenu.add(openProjectActions[i]);
    }
    recentProjectsMenu.setEnabled(recentProjectsMenu
      .getSubElements().length > 0);
  }

  /**
   * Constri o submenu para recuperao de dados externos ao projeto. O desktop
   * bsico no oferece nenhuma fonte de obteno de dados. Em alguns sistemas,
   * dados podem ser obtidos a partir de bases de dados, ou outras fontes. Neste
   * caso, este mtodo deve ser redefinido. Caso algum sistema no queira exibir
   * este menu, este mtodo deve ser redefinido para retornar null.
   * 
   * @return o menu de recuperao de dados ou null para omitir este menu.
   */
  protected JMenu createDataRetrievalMenu() {
    JMenu menu = new JMenu(LNG.get("desktop.project.data_retrieval"));
    return menu;
  }

  /**
   * Constri o menu para algoritmos (execuo e monitorao)
   * 
   * @return o menu de algoritmos
   */
  private JMenu createAlgorithmMenu() {
    JMenu menu = new JMenu(LNG.get("desktop.algorithms"));
    JMenuItem execMenuItem = menu.add(LNG.get("desktop.algorithms.execute"));
    execMenuItem.addActionListener(applicationPanel.getApplicationAction(
      "executor"));
    execMenuItem.setEnabled(false);
    enableIfProjectOpened(execMenuItem);
    return menu;
  }

  /**
   * Constri o menu para monitorao
   * 
   * @return menu de monitorao
   */
  private JMenu createMonitoringMenu() {
    JMenu menu = new JMenu(LNG.get("desktop.monitoring"));
    JMenuItem serversMenuItem = menu.add(applicationPanel.getApplicationAction(
      "sgamonitor"));
    serversMenuItem.setText(LNG.get("desktop.monitoring.servers"));
    JMenuItem cmdsMenuItem = menu.add(applicationPanel.getApplicationAction(
      "commandmonitor"));
    cmdsMenuItem.setText(LNG.get("desktop.monitoring.commands"));

    List<JMenuItem> diskItems = createDiskMonitoringMenuItems();
    if (diskItems != null && diskItems.size() > 0) {
      menu.addSeparator();
      for (JMenuItem item : diskItems) {
        menu.add(item);
      }
    }
    addOptionalMonitoringItems(menu);
    return menu;
  }

  /**
   * Constri itens opcionais do menu de monitorao. Esses itens sero
   * adicionados no final do menu
   * 
   * @param monitoringMenu o menu de monitorao
   */
  protected void addOptionalMonitoringItems(JMenu monitoringMenu) {
  }

  /**
   * Itens para monitorao de espao em disco
   * 
   * @return uma lista com os itens de monitorao de espao em disco
   */
  protected List<JMenuItem> createDiskMonitoringMenuItems() {
    ArrayList<JMenuItem> items = new ArrayList<JMenuItem>();
    JMenuItem diskinfoMenuItem = new JMenuItem(applicationPanel
      .getApplicationAction("diskinfo"));
    diskinfoMenuItem.setText(LNG.get("desktop.monitoring.disk"));
    items.add(diskinfoMenuItem);

    JMenuItem diskAreaMenuItem = new JMenuItem(applicationPanel
      .getApplicationAction("projectdiskmonitor"));
    diskAreaMenuItem.setText(LNG.get("desktop.monitoring.project_area"));
    items.add(diskAreaMenuItem);
    return items;
  }

  /**
   * Constri o menu para configurao do sistema. Este menu contm itens para a
   * alterao dos dados de um usurio e para a configurao das janelas dos
   * grupos de aplicaes.
   * 
   * @return o menu de configurao
   */
  private JMenu createConfigurationMenu() {
    JMenu menu = new JMenu(LNG.get("desktop.config"));
    addOptionalConfigItems(menu);
    JMenuItem configUserMenuItem = menu.add(LNG.get("desktop.config.user"));
    configUserMenuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        JFrame frame = getInstance().getDesktopFrame();
        User loggedUser = User.getLoggedUser();
        UserInfoDialog dialog;
        if (loggedUser.isAdmin()) {
          dialog = new AdminModifyUserInfoDialog(frame, loggedUser.getId());
        }
        else {
          dialog = new ModifyUserInfoDialog(frame, loggedUser.getId());
        }
        dialog.setVisible(true);
      }
    });
    menu.addSeparator();
    menu.add(applicationPanel.createGroupsConfigurationMenu());
    menu.add(createProjectAutoUpdateItem());
    if (isTrayIconSupported()) {
      menu.add(createTrayIconItem());
    }
    menu.add(createRestoreDesktopItem());
    menu.addSeparator();
    menu.add(new ClearApplicationCacheAction());
    return menu;
  }

  /**
   * Constri e apresenta os componentes da interface principal do cliente.
   */
  @Override
  public void showDesktop() {
    super.showDesktop();
    applicationPanel.tileGroupsHorizontal();
  }

  /**
   * Constri itens opcionais do menu de configurao. Esses itens sero
   * adicionados no incio do menu
   * 
   * @param configMenu o menu de configurao
   */
  protected void addOptionalConfigItems(JMenu configMenu) {
  }

  /**
   * Constri o menu de administrao do sistema. Se o usurio  administrador,
   * esse menu conter itens para a gerncia de usurios, grupos de usurios,
   * perfis, permisses e plataformas de execuo, realizada atravs da
   * interface de administrao (AdminFrame). Itens disponveis ao administrador
   * e a usurios que tem permisses correspondentes: - gerncia de SGAs
   * (servidores de algoritmos) - gerncia de algoritmos
   * 
   * @return o menu de administrao do sistema
   */
  private JMenu createAdminMenu() {
    boolean first = true;
    JMenu menu = new JMenu(LNG.get("desktop.admin"));
    User user = User.getLoggedUser();

    /* Itens exclusivos do administrador */
    if (user.isAdmin()) {
      /* Gerncia de usurios */
      JMenuItem userMenuItem = menu.add(LNG.get("desktop.admin.users"));
      userMenuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (adminFrame == null) {
            adminFrame = new AdminFrame();
          }
          adminFrame.showUserManagement();
        }
      });

      /* Gerncia de grupos de usurios */
      JMenuItem unMenuItem = menu.add(LNG.get("desktop.admin.userGroups"));
      unMenuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (adminFrame == null) {
            adminFrame = new AdminFrame();
          }
          adminFrame.showUserGroupManagement();
        }
      });

      /* Gerncia de perfis */
      JMenuItem roleMenuItem = menu.add(LNG.get("desktop.admin.roles"));
      roleMenuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (adminFrame == null) {
            adminFrame = new AdminFrame();
          }
          adminFrame.showRoleManagement();
        }
      });

      /* Gerncia de permisses */
      JMenuItem permissionMenuItem = menu.add(LNG.get(
        "desktop.admin.permissions"));
      permissionMenuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (adminFrame == null) {
            adminFrame = new AdminFrame();
          }
          adminFrame.showPermissionManagement();
        }
      });

      /* Gerncia de plataformas de execuo */
      JMenuItem platformMenuItem = menu.add(LNG.get("desktop.admin.platforms"));
      platformMenuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (adminFrame == null) {
            adminFrame = new AdminFrame();
          }
          adminFrame.showPlatformManagement();
        }
      });
      first = false;
    }

    /*
     * Aplicao Gerenciador de Algoritmos: apenas para administrador ou usurio
     * com a permisso correspondente
     */
    AlgorithmAdminPermission algoPermission = user.getPermission(
      AlgorithmAdminPermission.class);
    if (user.isAdmin() || (algoPermission != null)) {
      if (!first) {
        menu.addSeparator();
      }
      JMenuItem algorithmsManagerMenuItem = createAlgorithmsManagerMenuItem();
      if (algorithmsManagerMenuItem != null) {
        menu.add(algorithmsManagerMenuItem);
      }
      first = false;
    }

    if (user.isAdmin()) {
      if (!first) {
        menu.addSeparator();
      }
      JMenuItem applicationsReloadMenuItem = createApplicationReloadMenuItem();
      if (applicationsReloadMenuItem != null) {
        menu.add(applicationsReloadMenuItem);
        first = false;
      }
    }
    return menu;
  }

  /**
   * Cria um item de menu para a aplicao Gerenciador de Algoritmos.
   * 
   * @return o item de menu criado
   */
  private JMenuItem createApplicationReloadMenuItem() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent event) {
        ApplicationManagementProxy.reloadApplications(getDesktopFrame());
      }
    };
    final String text = LNG.get("desktop.admin.applications.reload");
    final JMenuItem menuItem = new JMenuItem(action);
    menuItem.setText(text);
    return menuItem;
  }

  /**
   * Cria um item de menu para a aplicao Gerenciador de Algoritmos.
   * 
   * @return o item de menu criado
   */
  private JMenuItem createAlgorithmsManagerMenuItem() {
    JMenuItem menuItem = new JMenuItem(LNG.get("desktop.admin.algorithms"));
    ApplicationManager mgr = ApplicationManager.getInstance();
    if (mgr != null) {
      ApplicationRegistry reg = mgr.getApplicationRegistry(
        ALGORITHMS_MANAGER_APPLICATION_ID);
      if (reg != null && reg.isShownAtApplicationMenu()) {
        AbstractAction action = mgr.getApplicationAction(reg);
        menuItem = new JMenuItem(action);
      }
    }
    return menuItem;
  }

  /**
   * Cria menu opcional no final da barra de menu quando sobrescrito pelas
   * subclasses. A implementao padro retorna null.
   * 
   * @return menu opcional
   */
  protected JMenu createOptionalMenu() {
    return null;
  }

  /**
   * Fecha o projeto corrente (caso exista) antes de encerrar a aplicao
   * cliente principal.
   */
  @Override
  public boolean shutdownProject() {
    ProjectCloseAction projCloseAction = new ProjectCloseAction(projectTree);
    if (ClientRemoteMonitor.getInstance().isAlive()) {
      /*
       * S fecha o projeto se houver conexo com o servidor e se o usurio no
       * cancelar o fechamento.
       */
      if (!projCloseAction.close()) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void prepareShutdown() {
    notifPanel.setUser(null);
  }

  /**
   * Este mtodo  chamado nas seguintes situaes: 1 - Um projeto do usurio 
   * criado 2 - Um projeto do usurio  removido 3 - O usurio  adicionado ou
   * removido de um projeto 4 - Um projeto do qual o usurio participa 
   * removido Desta forma, habilita ou desabilita os itens do menu Projeto
   * "(Abrir) Do Usurio" e "(Abrir) De outros usurios".
   * 
   * @param o .
   * @param arg .
   */
  @Override
  public void update(Observable o, Object arg) {
    if (arg instanceof ProjectUserEvent) {
      final ProjectUserEvent event = (ProjectUserEvent) arg;
      final boolean hasProject = userHasProject();
      switch (event.type) {
        /*
         * Projeto do usurio foi removido ou Projeto do qual o usurio
         * participava foi removido
         */
        case ProjectUserEvent.DELETE: {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              openProjMenuItem.setEnabled(hasProject);
              removeProjMenuItem.setEnabled(hasProject);
              fileTransferMenuItem.setEnabled(hasProject);
              openProjButton.setEnabled(hasProject);

              CommonProjectInfo info = (CommonProjectInfo) event.item;

              // Limpa a rvore de projetos se o projeto removido est aberto
              if ((projectTree.getProject() != null) && (info.projectId.equals(
                projectTree.getProject().getId()))) {
                projectTree.resetProject();
              }
            }
          });
          break;
        }

        /* Projeto do usurio foi criado */
        case ProjectUserEvent.CREATE: {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              openProjMenuItem.setEnabled(true);
              removeProjMenuItem.setEnabled(true);
              fileTransferMenuItem.setEnabled(true);
              openProjButton.setEnabled(true);
            }
          });
          break;
        }
      }
    }
  }

  /**
   * Finaliza o cliente.
   */
  @Override
  public boolean confirmShutdown() {
    if (!ClientRemoteMonitor.getInstance().ping()) {
      /*
       * Caso no haja comunicao com o servidor, o usurio  questionado se
       * deseja continuar com o processo de finalizao da aplicao.
       */
      int answer = tecgraf.javautils.gui.StandardDialogs.showYesNoDialog(
        getDesktopFrame(), getTitle(), LNG.get("desktop.confirm_shutdown"));
      if (answer != JOptionPane.YES_OPTION) {
        return false;
      }
    }
    return true;
    //    shutdownDesktop(new ProjectCloseAction(projectTree));
  }
}
