/*
 * $Id: DesktopComponentFrame.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.client.desktop;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Image;
import java.awt.Window;
import java.awt.event.WindowEvent;
import java.awt.event.WindowFocusListener;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import csbase.client.Client;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StatusBar;

/**
 * Janela de componente de desktop
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DesktopComponentFrame extends JFrame implements
  DesktopWindowInterface {

  /**
   * O contedo da janela.
   */
  private final JPanel frameContents;

  /**
   * A barra de status.
   */
  private StatusBar statusBar;

  /**
   * O ndice com que essa janela  encontrada
   */
  private Object index = null;

  /**
   * Janela a partir da qual esta foi crida.
   */
  protected DesktopComponentFrame owner;

  /**
   * Indica se o nome do servidor deve ser anexo ao ttulo da janela.
   */
  private static boolean showServerName = Client.getInstance().showServerName();

  /**
   * As janelas que esto abertas. Garante que a mesma janela no  aberta mais
   * de uma vez.
   */
  private static Map<Object, DesktopComponentFrame> openedFrames =
    new HashMap<Object, DesktopComponentFrame>();

  /**
   * Identificador da janela que possui o foco.
   */
  private static Object focusedWindowIndex;

  /**
   * Construtor padro.
   */
  public DesktopComponentFrame() {
    this(null, null, "");
  }

  /**
   * Construtor.
   * 
   * @param title o ttulo da janela
   */
  public DesktopComponentFrame(String title) {
    this(null, null, title);
  }

  /**
   * Construtor.
   * 
   * @param index o ndice que identifica esse dilogo
   * @param owner janela a partir da qual esta foi criada
   * @param title o ttulo da janela
   */
  public DesktopComponentFrame(Object index, DesktopComponentFrame owner,
    String title) {
    super();
    this.owner = owner;
    setTitle(title);
    setStandardIcon();
    setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    frameContents = new JPanel(new BorderLayout());
    statusBar = new StatusBar();
    Container cp = super.getContentPane();
    cp.add(frameContents, BorderLayout.CENTER);
    cp.add(statusBar, BorderLayout.SOUTH);
    DesktopFrame dktFrame = DesktopFrame.getInstance();
    if (dktFrame != null) {
      dktFrame.addWindow(this);
    }
    this.index = index;
    if (this.index == null) {
      Set<Object> existingKeys = openedFrames.keySet();
      int i = 0;
      while (existingKeys.contains("DesktopComponentFrame" + i)) {
        i++;
      }
      this.index = "DesktopComponentFrame" + i;
    }

    // adiciona o listener guardar a janela que possui o foco
    addWindowFocusListener(new WindowFocusListener() {

      @Override
      public void windowGainedFocus(WindowEvent arg0) {
        focusedWindowIndex = getIndex();
      }

      @Override
      public void windowLostFocus(WindowEvent arg0) {
      }

    });

    putDesktopComponentFrame(this.index, this);
    /* Seta o GlassPane translucido para a janela */
    setGlassPane(new GlassPane());
  }

  /**
   * Obtm o identificador da janela
   * 
   * @return o identificador da janela
   */
  public Object getIndex() {
    return index;
  }

  /**
   * Torna esta janela visvel ou no, mesmo que esteja minimizada.
   * 
   * @param isVisible verdadeiro se a janela tiver que ser visvel.
   */
  @Override
  public void setVisible(boolean isVisible) {
    super.setVisible(isVisible);
    if (isVisible && (getExtendedState() == Frame.ICONIFIED)) {
      setExtendedState(Frame.NORMAL);
    }
  }

  /**
   * Centraliza uma janela em relao a outra.
   * 
   * @param window .
   */
  public void center(Window window) {
    // TODO passar para GUIUtils do JavaUtils
    if (window instanceof JFrame) {
      /*
       * getSize(), getX() e getY() no levam em considerao o fato da janela
       * de referncia estar maximizada; neste caso, centralizamos na tela. Caso
       * window ainda no esteja com as dimenses definidas, centraliza na tela
       * tambm.
       */
      JFrame jframe = (JFrame) window;
      final int windowState = jframe.getExtendedState();
      if (((windowState & Frame.MAXIMIZED_BOTH) != 0) || window
        .getHeight() == 0) {
        GUIUtils.centerOnScreen(this);
        return;
      }
    }
    this.center(window.getSize(), window.getX(), window.getY());
  }

  /**
   * Mtodo de centralizao
   * 
   * @param dimension o tamanho
   * @param x a coordenada x
   * @param y a coordenada y
   */
  public void center(Dimension dimension, int x, int y) {
    Dimension wSize = this.getSize();
    int newX = x + ((dimension.width - wSize.width) / 2);
    if (newX < 0) {
      newX = 0;
    }
    int newY = y + ((dimension.height - wSize.height) / 2);
    if (newY < 0) {
      newY = 0;
    }
    this.setLocation(newX, newY);
  }

  /**
   * Ajuste do cone default do sistema em um dilogo.
   */
  private void setStandardIcon() {
    DesktopFrame dktFrame = DesktopFrame.getInstance();
    Image img = null;
    if (dktFrame != null) {
      img = dktFrame.getFrameImage();
    }
    if (img != null) {
      setIconImage(img);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    DesktopFrame dktFrame = DesktopFrame.getInstance();
    if (dktFrame != null) {
      dktFrame.removeWindow(this);
    }
    // Remove a janela da lista de janelas abertas
    if (index != null) {
      removeDesktopComponentFrame(index);
    }
    statusBar.shutdownTimer();
    super.dispose();
  }

  /**
   * Deprecated. Use o mtodo <code>close</code> para que a janela seja removida
   * do mapeamento das janelas abertas.
   */
  @Override
  @Deprecated
  public final void dispose() {
    close();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Container getContentPane() {
    return frameContents;
  }

  /**
   * Sinaliza um erro na janela.
   * 
   * @param errorEvent erro a ser sinalizado.
   */
  public void signalError(DesktopErrorEvent errorEvent) {
    /* Desabilita os componentes afetados */
    JComponent[] disabledComponents = errorEvent.disabledComponents;
    if (disabledComponents != null) {
      for (int i = 0; i < disabledComponents.length; i++) {
        disabledComponents[i].setEnabled(false);
      }
    }
    /* Se indicado, sinaliza o componente causador do erro */
    if (errorEvent.signaller != null) {
      DesktopFrame dktFrame = DesktopFrame.getInstance();
      Image img = null;
      if (dktFrame != null) {
        img = dktFrame.getDesktopErrorImage();
      }
      if ((dktFrame != null) && (img != null)) {
        errorEvent.signaller.setIcon(new ImageIcon(img));
      }
    }
    /* Coloca a mensagem na barra de status */
    statusBar.setError(errorEvent.errorMessage != null,
      errorEvent.errorMessage);
  }

  /**
   * Remove uma sinalizao de erro na janela.
   * 
   * @param errorEvent erro sinalizado.
   */
  public void resetError(DesktopErrorEvent errorEvent) {
    /* Reabilita os componentes afetados */
    JComponent[] disabledComponents = errorEvent.disabledComponents;
    if (disabledComponents != null) {
      for (int i = 0; i < disabledComponents.length; i++) {
        disabledComponents[i].setEnabled(true);
      }
    }
    /* Se indicado, remove a sinalizao de erro */
    if (errorEvent.signaller != null) {
      errorEvent.signaller.setIcon(null);
    }
    /* Retira a mensagem na barra de status */
    statusBar.hideStatusBar(errorEvent.errorMessage != null);
  }

  /**
   * Mostra uma sub-janela com uma mensagem.
   * 
   * @param title O ttulo da janela de mensagem
   * @param msg A mensagem a ser exibida
   * @param type .
   */
  public void showMessage(String title, Object msg, int type) {
    JOptionPane.showMessageDialog(this, msg, title, type);
  }

  /**
   * <p>
   * Torna a barra de status visvel.
   * </p>
   *  o mesmo que:<br>
   * <code>
   * DesktopComponentFrame frame = ...;<br> 
   * frame.getStatusBar().showStatusBar();
   * </code>
   */
  public void showStatusBar() {
    statusBar.showStatusBar();
  }

  /**
   * Consulta a barra de status. <br>
   * Obs. Chamadas a este mtodo no influencia na visibilidade da barra de
   * status.
   * 
   * @return a barra de status.
   */
  public StatusBar getStatusBar() {
    return this.statusBar;
  }

  /**
   * Atribui a status bar como sendo a do atributo e atualiza o contentPane
   * (removendo a statusBar anterior)
   * 
   * @param statusBar status bar a ser atribuda.
   */
  public void setStatusBar(StatusBar statusBar) {
    Container cp = super.getContentPane();
    cp.remove(this.statusBar);
    this.statusBar = statusBar;
    cp.add(statusBar, BorderLayout.SOUTH);
    cp.validate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTitle(final String title) {
    final DesktopFrame dktFrame = DesktopFrame.getInstance();
    String newTitle = title;
    if (dktFrame != null) {
      newTitle = dktFrame.adjustTitle(title);
    }
    super.setTitle(newTitle);
  }

  /**
   * Obtm uma janela que j esteja aberta.
   * 
   * @param id um identificador dessa janela
   * 
   * @return a janela aberta que possui o identificador
   */
  public static DesktopComponentFrame getDesktopComponentFrame(Object id) {
    DesktopComponentFrame frame = openedFrames.get(id);
    return frame;
  }

  /**
   * Registra uma janela aberta.
   * 
   * @param id um identificador dessa janela
   * @param frame uma janela
   */
  protected static void putDesktopComponentFrame(Object id,
    DesktopComponentFrame frame) {
    openedFrames.put(id, frame);
  }

  /**
   * Remove uma janela da lista de janelas abertas.
   * 
   * @param id um identificador dessa janela
   */
  protected static void removeDesktopComponentFrame(Object id) {
    openedFrames.remove(id);
  }

  /**
   * Retorna a janela que possui o foco.
   * 
   * @return a janela que possui o foco
   */
  public static DesktopComponentFrame getFocusedWindow() {
    return openedFrames.get(focusedWindowIndex);
  }
}
