package csbase.client.util;

import java.awt.Component;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * Renderizador para datas em tabelas. Possui um formatador padro, no formato
 * "dd/MM/yyyy - HH:mm:ss", que pode ser substitudo por outro especificado.
 * 
 * @author Leonardo Barros
 */
public class DateTableCellRenderer extends DefaultTableCellRenderer {
  /** Formato padro para datas no CSBase */
  private static final String DEFAULT_DATE_PATTERN = "dd/MM/yyyy - HH:mm:ss";

  /** O valor padro a ser utilizado quando a data for nula. */
  private String emptyValue = "";

  /** Formatador encarregado de formatar a data */
  private DateFormat formatter;

  /**
   * Cria o renderizador no formato padro do CSBase.
   */
  public DateTableCellRenderer() {
    this.formatter = new SimpleDateFormat(DEFAULT_DATE_PATTERN);
  }

  /**
   * Cria o renderizador com o formatador especificado.
   * 
   * @param formatter formatador para datas.
   */
  public DateTableCellRenderer(DateFormat formatter) {
    this.formatter = formatter;
  }

  /**
   * Atribui o valor padro a ser utilizado quando nenhuma data for encontrada.
   * 
   * @param emptyValue O valor padro a ser utilizado quando nenhuma data for
   *        encontrada.
   */
  public void setDefaultEmptyValue(String emptyValue) {
    this.emptyValue = emptyValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    Component comp =
      super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
        row, column);
    if (value == null) {
      ((DefaultTableCellRenderer) comp).setText(emptyValue);
    }
    else if (value instanceof Date) {
      Date date = (Date) value;
      String strDate = formatter.format(date);
      ((DefaultTableCellRenderer) comp).setText(strDate);
      setToolTipText(strDate);
    }
    return comp;
  }
}
