/*
 * StandardDialogs.java $Id: StandardErrorDialogs.java 66445 2007-08-15
 * 20:07:03Z taciana $
 */
package csbase.client.util;

import java.awt.Window;
import java.util.Arrays;

import javax.swing.JOptionPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.exception.PermissionException;

/**
 * A classe <code>StandardDialogs</code> prove mtodos estticos que implementam
 * vrias funcionalidades utilizadas repetidamente pela interface do sistema.
 * 
 * @author Ana Moura e Maria Jlia Lima
 */
public class StandardErrorDialogs {

  /** Texto de fachamento */
  private static final String[] CLOSE = new String[] { LNG.get("UTIL_CLOSE") };

  /**
   * Mostra uma janela com uma mensagem de erro; O ttulo da janela  padro.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param msg A mensagem de erro,
   */
  public static void showErrorDialog(Window window, Object msg) {
    showErrorDialog(window, LNG.get("UTIL_EXECUTION_ERROR"), msg, null);
  }

  /**
   * Mostra uma janela com uma mensagem de uma exceo.
   * 
   * @param wnd a janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param t a exceo ocorrida.
   */
  public static void showExceptionDialog(final Window wnd, final String title,
    final Throwable t) {
    showExceptionDialog(wnd, title, t, (String[]) null);
  }

  /**
   * Mostra uma janela com uma mensagem de uma exceo.
   * 
   * @param wnd a janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param t a exceo ocorrida.
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email ao admin.
   */
  public static void showExceptionDialog(final Window wnd, final String title,
    final Throwable t, String[] additionalInfo) {
    final String msg = (title == null ? "" : title);
    if (t instanceof PermissionException) {
      // o usurio no tem permisso para realizar uma operao.
      // mostrar dilogo mais amigvel:
      showPermissionDeniedDialog(wnd, (PermissionException) t);
      return;
    }
    final ExceptionDialog dlg =
      ExceptionDialog.createDialog(wnd, msg, t, additionalInfo);
    dlg.setVisible(true);
  }

  /**
   * Mostra dilogo com um aviso de que uma operao no pde ser realizada pois
   * o usurio atual no tem permisso.
   * 
   * @param wnd - janela de referncia.
   * @param exception - a exceo associada.
   */
  private static void showPermissionDeniedDialog(final Window wnd,
    PermissionException exception) {
    String msg = exception.getMessage();
    if (msg == null || msg.isEmpty()) {
      msg = LNG.get("UTIL_PERMISSION_DENIED_MSG");
    }
    StandardDialogs.showErrorDialog(wnd,
      LNG.get("UTIL_PERMISSION_DENIED_TITLE"), msg);
  }

  /**
   * Mostra uma janela com uma mensagem de erro e uma exceo; A janela oferece
   * um boto que permite ao usurio ver os detalhes da exceo, como a pilha de
   * execuo.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param titleStr O ttulo da janela.
   * @param msg A mensagem a ser apresentada na janela.
   * @param throwable A exceo ocorrida.
   */
  public static void showErrorDialog(final Window window,
    final String titleStr, final Object msg, final Throwable throwable) {
    showErrorDialog(window, titleStr, msg, throwable, null);
  }

  /**
   * Mostra uma janela com uma mensagem de erro e uma exceo; A janela oferece
   * um boto que permite ao usurio ver os detalhes da exceo, como a pilha de
   * execuo.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param titleStr O ttulo da janela.
   * @param msg A mensagem a ser apresentada na janela.
   * @param throwable A exceo ocorrida.
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email ao admin.
   */
  public static void showErrorDialog(final Window window,
    final String titleStr, final Object msg, final Throwable throwable,
    String[] additionalInfo) {
    if (throwable instanceof PermissionException) {
      // o usurio no tem permisso para realizar uma operao.
      // mostrar dilogo mais amigvel:
      showPermissionDeniedDialog(window, (PermissionException) throwable);
      return;
    }
    final String title = LNG.get("UTIL_ERROR") + " - " + titleStr;
    if (throwable == null) {
      // dilogo de erro simples ...
      if (additionalInfo != null) {
        Object[] allMessages;
        // temos alguma informao sobre o contexto do erro ... adicionar na
        // mensagem (JOptionPane aceita Object[], organizando-as verticalmente)
        if (msg instanceof Object[]) {
          // j era uma lista de mensagens ... inserir msgs adicionais no final
          allMessages = joinArrays((Object[]) msg, additionalInfo);
        }
        else {
          // era apenas uma 'mensagem' ... criar lista com msgs adicionais ...
          allMessages = new Object[additionalInfo.length + 1];
          allMessages[0] = msg;
          for (int i = 1; i < allMessages.length; i++) {
            allMessages[i] = additionalInfo[i - 1];
          }
        }
        showSimpleErrorDialog(window, title, allMessages);
      }
      else {
        showSimpleErrorDialog(window, title, msg);
      }
    }
    else {
      showExceptionDialog(window, title, msg, throwable, additionalInfo);
    }
  }

  /**
   * Unifica os arrays.
   * 
   * @param array1
   * @param array2
   * @return um novo array com a unificao 'array1 + array2'.
   */
  private static Object[] joinArrays(Object[] array1, Object[] array2) {
    int newSize = array1.length + array2.length;
    Object[] newMsg = Arrays.copyOf(array1, newSize);
    for (int i = 0; i < array2.length; i++) {
      newMsg[array1.length + i] = array2[i];
    }
    return newMsg;
  }

  /**
   * Mostra dilogo de erro especial, com a opo 'detalhar' e com uma exceo
   * associada.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param msg A mensagem a ser apresentada na janela.
   * @param throwable - a exceo que causou o problema.
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email ao admin.
   */
  private static void showExceptionDialog(Window window, String title,
    Object msg, Throwable throwable, String[] additionalInfo) {
    String message = (msg != null) ? msg.toString() : null;
    final ExceptionDialog dlg =
      ExceptionDialog.createDialog(window, title, throwable, message,
        additionalInfo);
    dlg.setVisible(true);
  }

  /**
   * Mostra dilogo de erro sem uma exceo associada.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param msg A mensagem a ser apresentada na janela.
   */
  private static void showSimpleErrorDialog(Window window, String title,
    Object msg) {

    JOptionPane.showOptionDialog(window, msg, title, JOptionPane.YES_NO_OPTION,
      JOptionPane.ERROR_MESSAGE, null, CLOSE, CLOSE[0]);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param msg A mensagem a ser apresentada na janela.
   */
  public static void showErrorDialog(final Window window, final String title,
    final Object msg) {
    showErrorDialog(window, title, msg, null);
  }

  /**
   * Mostra uma janela de erro com uma exceo ocorrida; A janela oferece um
   * boto que permite ao usurio ver os detalhes da exceo, como a pilha de
   * execuo.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param title O ttulo da janela.
   * @param throwable A exceo ocorrida.
   */
  public static void showErrorDialog(final Window window, final String title,
    final Throwable throwable) {
    showErrorDialog(window, title, null, throwable);
  }

  /**
   * Mostra uma janela com uma mensagem de erro; O ttulo da janela  padro.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param msg A mensagem de erro.
   * @param throwable A causa.
   */
  public static void showErrorDialog(final Window window, final Object msg,
    final Throwable throwable) {
    showErrorDialog(window, LNG.get("UTIL_EXECUTION_ERROR"), msg, throwable);
  }

  /**
   * Mostra uma janela de erro com uma exceo ocorrida; O ttulo da janela 
   * padro e a janela oferece um boto que permite ao usurio ver os detalhes
   * da exceo, como a pilha de execuo.
   * 
   * @param window A janela principal que solicitou esse dilogo.
   * @param throwable A exceo ocorrida.
   */
  public static void showErrorDialog(final Window window,
    final Throwable throwable) {
    showErrorDialog(window, LNG.get("UTIL_EXECUTION_ERROR"),
      throwable.getLocalizedMessage(), throwable);
  }
}
