/*
 * $Id: CommandViewerTable.java 176427 2016-10-04 14:35:28Z fpina $
 */
package csbase.client.applications.flowapplication.commandviewer;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.CommandsFilter;
import csbase.client.algorithms.commands.cache.events.AbstractCacheUpdatedEventListener;
import csbase.client.algorithms.commands.newview.CommandViewFactory;
import csbase.client.algorithms.commands.newview.NewCommandViewUtils;
import csbase.client.algorithms.commands.view.AlgorithmCommandViewFactory;
import csbase.client.algorithms.commands.view.TabType;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.commandtable.CommandStatusCellData;
import csbase.client.facilities.commandtable.CommandStatusCellRenderer;
import csbase.client.facilities.commandtable.CommandStatusHelper;
import csbase.client.kernel.ClientException;
import csbase.client.util.table.ProgressCellRenderer;
import csbase.logic.CommandInfo;
import csbase.logic.ProgressData;
import csbase.logic.Utilities;
import csbase.remote.CommandPersistenceServiceInterface;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Tabela para visualizao de comandos.
 */
final class CommandViewerTable extends SortableTable {

  /**
   * A quantidade de colunas da tabela de histrico de comandos.
   */
  private static final int COLUMN_COUNT = 8;
  /**
   * O ndice da coluna em execuo da tabela de histrio de comandos.
   */
  private static final int STATUS_COLUMN_INDEX = 0;
  /**
   * Largura da coluna estado.
   */
  private static final int STATUS_COLUMN_WIDTH = 50;
  /**
   * O ndice da coluna progresso da tabela de histrico de comandos.
   */
  private static final int PROGRESS_COLUMN_INDEX = 1;
  /**
   * Largura da coluna progresso.
   */
  private static final int PROGRESS_COLUMN_WIDTH = 100;
  /**
   * O ndice da coluna usurio da tabela de histrico de comandos.
   */
  private static final int USER_COLUMN_INDEX = 2;
  /**
   * Largura da coluna descrio.
   */
  private static final int USER_COLUMN_WIDTH = 75;
  /**
   * O ndice da coluna nome do SGA da tabela de histrico de comandos.
   */
  private static final int SGA_NAME_COLUMN_INDEX = 3;
  /**
   * Largura da coluna data inicial.
   */
  private static final int SGA_NAME_COLUMN_WIDTH = 75;
  /**
   * O ndice da coluna descrio da tabela de histrico de comandos.
   */
  private static final int DESCRIPTION_COLUMN_INDEX = 4;
  /**
   * Largura da coluna descrio.
   */
  private static final int DESCRIPTION_COLUMN_WIDTH = 200;
  /**
   * O ndice da coluna comando da tabela de histrico de comandos.
   */
  private static final int COMMAND_TIP_COLUMN_INDEX = 5;
  /**
   * Largura da coluna comando.
   */
  private static final int COMMAND_TIP_COLUMN_WIDTH = 125;
  /**
   * O ndice da coluna data inicial da tabela de histrico de comandos.
   */
  private static final int START_DATE_COLUMN_INDEX = 6;
  /**
   * Largura da coluna data inicial.
   */
  private static final int START_DATE_COLUMN_WIDTH = 160;
  /**
   * O ndice da coluna tempo de execuo da tabela de histrico de comandos.
   */
  private static final int ELAPSED_TIME_INDEX = 7;
  /**
   * Largura da coluna tempo de execuo.
   */
  private static final int ELAPSED_TIME_COLUMN_WIDTH = 160;

  /**
   * O valor da clula da tabela para indicar valor desconhecido.
   */
  private static final String UNKNOWN_CELL_VALUE = "";

  /**
   * Observador de comandos.
   */
  private AbstractCacheUpdatedEventListener listener;

  /**
   * A lista de comandos.
   */
  private final List<CommandInfo> commandInfoList;

  /**
   * O filtro.
   */
  private CommandPropertiesFilter filter;

  /**
   * Filtro de comandos.
   */
  private CommandsFilter commandsCacheFilter;

  /**
   * O identificador do projeto.
   */
  private Object projectId;

  /**
   * O modelo da tabela de histrico dos comandos.
   */
  private AbstractTableModel tableModel;

  /**
   * A janela que criou esta.
   */
  private final DesktopComponentFrame owner;

  /**
   * Cria uma tabela para visualizao de comandos.
   *
   * @param owner A janela que criou esta (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   */
  public CommandViewerTable(DesktopComponentFrame owner, Object projectId) {
    this(owner, projectId, null);
  }

  /**
   * Cria uma tabela para visualizao de comandos.
   *
   * @param owner A janela que criou esta (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   * @param filter O filtro (Aceita {@code null}).
   */
  public CommandViewerTable(DesktopComponentFrame owner, Object projectId,
    CommandPropertiesFilter filter) {

    this.owner = owner;
    this.commandInfoList = new LinkedList<CommandInfo>();
    setProjectId(projectId);
    setFilter(filter);
    createGui();
    createCommandListener(projectId);
    loadCommandInfoList();
  }

  /**
   * Obtm o identificador do projeto.
   *
   * @return O identificador do projeto.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Obtm uma lista dos comandos selecionados.
   *
   * @return uma lista dos comandos selecionados.
   */
  public List<CommandInfo> getSelectedCommands() {
    /**
     * {@link JTable#getSelectedRows()} retorna o ndice das linhas selecionadas
     * no modelo. Como o modelo da tabela  um {@link SortableTableModel} esses
     * ndices no so os mesmos do {@link AbstractTableModel}. Utiliza-se ento
     * o mtodo {@link SortableTableModel#getModelRowIndex(int)} para fazer a
     * converso e ento pegarmos as propriedades do comando correto no modelo
     * do tipo {@link AbstractTableModel}.
     */
    int[] sortedRowsIndex = getSelectedRows();
    if (0 == sortedRowsIndex.length) {
      return Collections.emptyList();
    }

    List<CommandInfo> commands = new ArrayList<CommandInfo>();
    for (int inx = 0; inx < sortedRowsIndex.length; inx++) {
      int sortedRowIndex = sortedRowsIndex[inx];
      int selectedRowIndex = this.convertRowIndexToModel(sortedRowIndex);
      CommandInfo commandInfo = commandInfoList.get(selectedRowIndex);
      commands.add(commandInfo);
    }

    return commands;
  }

  /**
   * Se s um comando estiver selecionado, exibe um relatrio daquele comando.
   *
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   *
   * @return {@code true} em caso de sucesso ou {@code false} em caso de erro.
   */
  public boolean showSelectedCommand(final TabType preferredTab) {
    try {
      List<CommandInfo> commands = getSelectedCommands();
      if (1 != commands.size()) {
        return false;
      }

      final CommandInfo command = commands.get(0);

      if (NewCommandViewUtils.useNewVersion()) {
        CommandViewFactory.showCommandView(command, owner, NewCommandViewUtils
          .getNewTabType(preferredTab));
      }
      else {
        AlgorithmCommandViewFactory.showView(owner, command, preferredTab);
      }
    }
    catch (ClientException e) {
      return false;
    }
    return true;
  }

  /**
   * Carrega as propriedades do comando e comear a observar o
   * {@link CommandPersistenceServiceInterface servio de persistncia de
   * comandos}.
   */
  public void start() {
    CommandsCache.getInstance().addEventListener(listener);
  }

  /**
   * Para de observar o {@link CommandPersistenceServiceInterface servio de
   * persistncia de comandos}.
   */
  public void stop() {
    CommandsCache.getInstance().removeEventListener(listener);
  }

  /**
   * Cria o observador de comandos para um determinado projeto.
   *
   * @param prjId o identificador do projeto.
   */
  private void createCommandListener(final Object prjId) {
    this.commandsCacheFilter = new CommandsFilter() {
      @Override
      protected final boolean acceptCommand(CommandInfo command) {
        return command.getProjectId().equals(prjId);
      }
    };
    this.listener = new AbstractCacheUpdatedEventListener(commandsCacheFilter) {
      @Override
      public void eventFired(Collection<CommandInfo> commands) {
        loadCommandInfoList(commands);
      }

      @Override
      protected void eventInterrupted(Exception exception, String description) {
        owner.getStatusBar().setError(description);
      }
    };
  }

  /**
   * Ajusta as larguras da tabela e das colunas.
   */
  private void adjustWidths() {
    setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
    TableColumnModel tableColumnModel = getColumnModel();
    TableColumn startDateColumn = tableColumnModel.getColumn(
      START_DATE_COLUMN_INDEX);
    startDateColumn.setPreferredWidth(START_DATE_COLUMN_WIDTH);
    TableColumn cmdNameColumn = tableColumnModel.getColumn(
      COMMAND_TIP_COLUMN_INDEX);
    cmdNameColumn.setPreferredWidth(COMMAND_TIP_COLUMN_WIDTH);
    TableColumn sgaNameColumn = tableColumnModel.getColumn(
      SGA_NAME_COLUMN_INDEX);
    sgaNameColumn.setPreferredWidth(SGA_NAME_COLUMN_WIDTH);
    TableColumn userColumn = tableColumnModel.getColumn(USER_COLUMN_INDEX);
    userColumn.setPreferredWidth(USER_COLUMN_WIDTH);
    TableColumn descriptionColumn = tableColumnModel.getColumn(
      DESCRIPTION_COLUMN_INDEX);
    descriptionColumn.setPreferredWidth(DESCRIPTION_COLUMN_WIDTH);
    TableColumn progressColumn = tableColumnModel.getColumn(
      PROGRESS_COLUMN_INDEX);
    progressColumn.setPreferredWidth(PROGRESS_COLUMN_WIDTH);
    TableColumn statusColumn = tableColumnModel.getColumn(STATUS_COLUMN_INDEX);
    statusColumn.setPreferredWidth(STATUS_COLUMN_WIDTH);
    TableColumn elapsedColumn = tableColumnModel.getColumn(ELAPSED_TIME_INDEX);
    elapsedColumn.setPreferredWidth(ELAPSED_TIME_COLUMN_WIDTH);
    Dimension intercellSpacing = getIntercellSpacing();
    int intercellWidth = intercellSpacing.width;
    int tableWidth = 0;
    tableWidth += intercellWidth;
    int separator = 0;
    TableColumnModel colModel = getColumnModel();
    for (int columnIndex = 0; columnIndex < getColumnCount(); columnIndex++) {
      TableColumn column = colModel.getColumn(columnIndex);
      int columnWidth = column.getPreferredWidth();
      tableWidth += separator;
      tableWidth += columnWidth;
      separator = intercellWidth;
    }
    tableWidth += intercellWidth;
    Dimension tableSize = getPreferredScrollableViewportSize();
    int tableHeight = tableSize.height;
    setPreferredScrollableViewportSize(new Dimension(tableWidth, tableHeight));
  }

  /**
   * Cria o comparador de datas iniciais.
   *
   * @return .
   */
  private Comparator<CommandInfo> createComparator() {
    return new Comparator<CommandInfo>() {
      @Override
      public int compare(CommandInfo o1, CommandInfo o2) {
        if (o1.equals(o2)) {
          return 0;
        }
        Date startDate1 = o1.getSubmittedDate();
        if (startDate1 == null) {
          return 1;
        }
        Date startDate2 = o2.getSubmittedDate();
        if (startDate2 == null) {
          return -1;
        }
        return startDate2.compareTo(startDate1);
      }
    };
  }

  /**
   * Cria a interface grfica.
   */
  private void createGui() {
    createTableModel();
    setModel(tableModel);
    createDescriptionCellRenderer();
    createDateCellRenderer();
    createProgressCellRenderer();
    createElapsedTimeCellRenderer();
    setDefaultRenderer(CommandStatusCellData.class,
      new CommandStatusCellRenderer());
    setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    setCellSelectionEnabled(false);
    setColumnSelectionAllowed(false);
    setRowSelectionAllowed(true);
    addMouseListener(createMouseListener());
    adjustWidths();
    selectFirstRow();
  }

  /**
   * Cria o renderizador da columna descrio.
   */
  private void createDescriptionCellRenderer() {
    TableColumn column = getColumnModel().getColumn(DESCRIPTION_COLUMN_INDEX);
    DefaultTableCellRenderer cellRenderer = new DefaultTableCellRenderer();
    cellRenderer.setIcon(ApplicationImages.ICON_EDITABLE_TABLE_CELL_14);
    column.setCellRenderer(cellRenderer);
  }

  /**
   * Cria o renderizador da columna descrio.
   */
  private void createDateCellRenderer() {
    TableColumn column = getColumnModel().getColumn(START_DATE_COLUMN_INDEX);
    DefaultTableCellRenderer cellRenderer = new DefaultTableCellRenderer() {
      @Override
      public void setValue(Object value) {
        if (value == null) {
          return;
        }
        final Date date = (Date) value;
        String formattedDate = Utilities.getFormattedDate(date);
        setText(formattedDate);
      }

    };
    column.setCellRenderer(cellRenderer);
  }

  /**
   * Cria o renderizador da coluna progresso.
   */
  private void createProgressCellRenderer() {
    TableColumn column = getColumnModel().getColumn(PROGRESS_COLUMN_INDEX);
    ProgressCellRenderer cellRenderer = new ProgressCellRenderer(
      UNKNOWN_CELL_VALUE);
    column.setCellRenderer(cellRenderer);
  }

  /**
   * Cria o renderizador da coluna tempo de execuo.
   */
  private void createElapsedTimeCellRenderer() {
    TableColumn column = getColumnModel().getColumn(ELAPSED_TIME_INDEX);
    DefaultTableCellRenderer renderer = new DefaultTableCellRenderer() {
      @Override
      public Component getTableCellRendererComponent(JTable table, Object value,
        boolean isSelected, boolean hasFocus, int row, int column) {

        super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
          row, column);

        if (value == null) {
          setText(getMessage("column.default.value.empty"));
        }
        else {
          final int wtime = (Integer) value;
          if (wtime < 0) {
            setText(getMessage("column.default.value.error"));
          }
          else {
            final String text = FormatUtils.formatInterval(wtime);
            setText(text);
          }
        }
        return this;
      }
    };
    column.setCellRenderer(renderer);
  }

  /**
   * Cria o {@link MouseListener} da tabela.
   *
   * @return O {@link MouseListener}.
   */
  private MouseListener createMouseListener() {
    return new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          showSelectedCommand(TabType.PARAMETERS);
        }
      }
    };
  }

  /**
   * Cria o modelo da tabela de histrio de comandos.
   */
  private void createTableModel() {
    tableModel = new AbstractTableModel() {

      @Override
      public Class<?> getColumnClass(int columnIndex) {
        switch (columnIndex) {
          case START_DATE_COLUMN_INDEX:
            return Date.class;
          case STATUS_COLUMN_INDEX:
            return CommandStatusCellData.class;
          case PROGRESS_COLUMN_INDEX:
            return ProgressData.class;
          case ELAPSED_TIME_INDEX:
            return Integer.class;
          default:
            return String.class;
        }
      }

      @Override
      public int getColumnCount() {
        return COLUMN_COUNT;
      }

      @Override
      public String getColumnName(int columnIndex) {
        switch (columnIndex) {
          case START_DATE_COLUMN_INDEX:
            return getMessage("label_column_start_date");
          case COMMAND_TIP_COLUMN_INDEX:
            return getMessage("label_column_command_tip");
          case SGA_NAME_COLUMN_INDEX:
            return getMessage("label_column_sga_name");
          case USER_COLUMN_INDEX:
            return getMessage("label_column_user_login");
          case DESCRIPTION_COLUMN_INDEX:
            return getMessage("label_column_description");
          case PROGRESS_COLUMN_INDEX:
            return getMessage("label_column_progress");
          case STATUS_COLUMN_INDEX:
            return getMessage("label_column_status");
          case ELAPSED_TIME_INDEX:
            return getMessage("label_column_elapsed_time");
          default:
            String errorMessage = String.format(
              "ndice de coluna invlido.\nndice passado: %d.\n", columnIndex);
            throw new IllegalArgumentException(errorMessage);
        }
      }

      @Override
      public int getRowCount() {
        return commandInfoList.size();
      }

      @Override
      public Object getValueAt(int rowIndex, int columnIndex) {
        CommandInfo commandInfo = getCommandInfo(rowIndex);
        switch (columnIndex) {
          case START_DATE_COLUMN_INDEX:
            return commandInfo.getSubmittedDate();
          case COMMAND_TIP_COLUMN_INDEX:
            return commandInfo.getTip();
          case SGA_NAME_COLUMN_INDEX:
            String sgaName = commandInfo.getSGAName();
            if (sgaName == null) {
              return UNKNOWN_CELL_VALUE;
            }
            return sgaName;
          case USER_COLUMN_INDEX:
            Object userId = commandInfo.getUserId();
            if (userId == null) {
              return UNKNOWN_CELL_VALUE;
            }
            return userId;
          case DESCRIPTION_COLUMN_INDEX:
            String description = commandInfo.getDescription();
            if (description == null) {
              return UNKNOWN_CELL_VALUE;
            }
            return description;
          case PROGRESS_COLUMN_INDEX:
            ProgressData progressData = commandInfo.getProgressData();
            return progressData;
          case STATUS_COLUMN_INDEX:
            return CommandStatusHelper.createCellData(commandInfo);
          case ELAPSED_TIME_INDEX:
            return commandInfo.getWallTimeSec();
          default:
            String errorMessage = String.format("ndice de coluna invlido.\n"
              + "ndice passado: %d.\n", columnIndex);
            throw new IllegalArgumentException(errorMessage);
        }
      }

      @Override
      public boolean isCellEditable(int rowIndex, int columnIndex) {
        if (columnIndex == DESCRIPTION_COLUMN_INDEX) {
          return true;
        }
        return false;
      }

      @Override
      public void setValueAt(Object value, int rowIndex, int columnIndex) {
        if (columnIndex == DESCRIPTION_COLUMN_INDEX) {
          CommandInfo commandInfo = getCommandInfo(rowIndex);
          if (commandInfo != null) {
            String description = (String) value;
            if (description.length() == 0) {
              description = null;
            }
            String previousDescription = commandInfo.getDescription();
            commandInfo.setDescription(description);
            if (updateCommandInfo(commandInfo)) {
              fireTableCellUpdated(rowIndex, columnIndex);
            }
            else {
              commandInfo.setDescription(previousDescription);
            }
          }
        }
        super.setValueAt(value, rowIndex, columnIndex);
      }
    };
  }

  /**
   * Remove as informaes dos comandos que no forem aceitos pelo filtro.
   *
   * @param commandInfoSet O conjunto de comandos (No aceita {@code null}).
   */
  private void filterCommandsInfo(Collection<CommandInfo> commandInfoSet) {
    Iterator<CommandInfo> commandInfoIterator = commandInfoSet.iterator();
    while (commandInfoIterator.hasNext()) {
      CommandInfo commandInfo = commandInfoIterator.next();
      if (!filterCommandInfo(commandInfo)) {
        commandInfoIterator.remove();
      }
    }
  }

  /**
   * Indica se as informaes do comando devem ser utilizadas.
   *
   * @param commandInfo As informaes do comando.
   *
   * @return .
   */
  private boolean filterCommandInfo(CommandInfo commandInfo) {
    return filter.accept(commandInfo);
  }

  /**
   * Obtm as informaes do comando de uma determinada linha.
   *
   * @param rowIndex O ndice da linha.
   *
   * @return As propriedades do comando.
   */
  private CommandInfo getCommandInfo(int rowIndex) {
    CommandInfo commandInfo = commandInfoList.get(rowIndex);
    return commandInfo;
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   *
   * @param keySuffix O sufixo da chave (o nome da classe ser pr-fixado ao
   *        sufixo).
   *
   * @return A mensagem.
   */
  private String getMessage(String keySuffix) {
    return LNG.get(getClass().getSimpleName() + "." + keySuffix);
  }

  /**
   * Obtm a janela que  dona deste componente.
   *
   * @return A janela.
   */
  private Window getWindow() {
    return SwingUtilities.getWindowAncestor(this);
  }

  /**
   * Obtm o ndice da linha na viso a partir do ndice da linha do modelo
   * (tabela).
   *
   * @param modelRowIndex ndice de linha no modelo (tabela).
   *
   * @return ndice da linha na viso. Retorna -1 se a linha especificada
   *         estiver fora dos limites do modelo, ou se o modelo atual for nulo.
   */
  @Override
  public int convertRowIndexToView(int modelRowIndex) {
    if (modelRowIndex < 0) {
      return -1;
    }
    if (getModel() == null) {
      return -1;
    }
    if (modelRowIndex >= getModel().getRowCount()) {
      return -1;
    }
    return super.convertRowIndexToView(modelRowIndex);
  }

  /**
   * Carrega as informaes de comando.
   */
  private void loadCommandInfoList() {
    RemoteTask<Collection<CommandInfo>> task =
      new RemoteTask<Collection<CommandInfo>>() {
        @Override
        protected void performTask() throws Exception {
          Collection<CommandInfo> commands = CommandsCache.getInstance()
            .getCommands(commandsCacheFilter);
          setResult(commands);
        }
      };
    String taskTitle = LNG.get("CommandViewerTable.msg_reading_command_title");
    String taskMessage = LNG.get(
      "CommandViewerTable.msg_reading_command_title");
    if (!task.execute(getWindow(), taskTitle, taskMessage)) {
      return;
    }

    final Collection<CommandInfo> commands = task.getResult();
    loadCommandInfoList(commands);
  }

  /**
   * Carrega as informaes de comando.
   *
   * @param commands Comandos a serem visualizados.
   */
  private void loadCommandInfoList(final Collection<CommandInfo> commands) {
    filterCommandsInfo(commands);
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        /*
         * Salva a seleo corrente para recuper-la aps a atualizao dos
         * dados.
         */
        final List<CommandInfo> selecteds = getSelectedCommands();

        /*
         * Altera a lista de comandos que  a fonte de dados do modelo da tabela
         * e informar a tabela que os dados foram alterados.
         */
        commandInfoList.clear();
        commandInfoList.addAll(commands);
        sortCommandInfoList();
        tableModel.fireTableDataChanged();

        /*
         * Com a alterao dos dados da tabela, deve-se rever os comandos
         * selecionados. Para fazer isso, a viso da tabela deve antes ser
         * atualizada com os dados novos. Como a viso da tabela s muda aps o
         * evento disparado acima ser tratado pelo EDT (event-dispatching
         * thread), foi agendado o processo abaixo para ser tratado em seguida
         * pelo EDT, garantindo assim a ordem dos eventos.
         */
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            clearSelection();
            for (int inx = 0; inx < commandInfoList.size(); inx++) {
              CommandInfo cmd = commandInfoList.get(inx);
              if (selecteds.contains(cmd)) {
                int selectedInx = convertRowIndexToView(inx);
                if (-1 < selectedInx) {
                  addRowSelectionInterval(selectedInx, selectedInx);
                }
              }
            }
          }
        });
      }
    });
  }

  /**
   * Seleciona o comando da primeira linha da tabela de histrico de comandos.
   */
  private void selectFirstRow() {
    if (!commandInfoList.isEmpty()) {
      changeSelection(0, 0, false, false);
    }
  }

  /**
   * Atribui um filtro a esta tabela.
   *
   * @param filter O filtro (Aceita {@code null}).
   */
  private void setFilter(CommandPropertiesFilter filter) {
    this.filter = filter == null ? new AlwaysTrueCommandPropertiesFilter()
      : filter;
  }

  /**
   * Atribui o identificador do projeto a esta tabela.
   *
   * @param projectId O identificador do projeto (No aceita {@code null}).
   */
  private void setProjectId(Object projectId) {
    if (projectId == null) {
      throw new IllegalArgumentException("projectId == null");
    }
    this.projectId = projectId;
  }

  /**
   * Ordena a lista de propriedades de comando.
   */
  private void sortCommandInfoList() {
    Collections.sort(commandInfoList, createComparator());
  }

  /**
   * Atualiza o comando.
   *
   * @param commandInfo As informaes do comando (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} em caso de erro.
   */
  private boolean updateCommandInfo(final CommandInfo commandInfo) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws RemoteException {
        CommandsCache.getInstance().updateDescription(commandInfo);
      }
    };
    String taskTitle = getMessage("msg_saving_command_title");
    String commandId = commandInfo.getId();
    String taskMessage = String.format(getMessage("msg_saving_command_content"),
      commandId);
    return task.execute(getWindow(), taskTitle, taskMessage);
  }
}
