package csbase.client.applications.projectsynchronization;

import java.awt.Container;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import tecgraf.javautils.gui.wizard.History;
import tecgraf.javautils.gui.wizard.WizardException;
import csbase.client.desktop.LocalTask;
import csbase.logic.ProjectSynchronizationData;
import csbase.logic.ProjectSynchronizationFileInfo;
import csbase.logic.ProjectSynchronizationUnit;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.util.StringUtils;

/**
 * Passo de concluso da sincronizao do wizard. Apresenta a lista com
 * os arquivos a serem transferidos.
 */
public class StepConclusion extends StepProjectSynchronization {
  /** Componete da com o resumo da sincronizao. */
  private ContainerConclusion containerConclusion;
  /** Ddados para sincronizao entre dois projetos.*/
  private ProjectSynchronizationData data;
  /** Referncia para a aplicao. */
  private ApplicationRegistry registry;

  private String error;

  /**
   * Construtor. 
   * @param registry referncia para o registro da aplicao
   */
  StepConclusion(ApplicationRegistry registry) {
    this.registry = registry;
    containerConclusion = new ContainerConclusion();
    setConfirmOperation(ConfirmOperation.NOTHING);
  }

  /**
   * Anexa texto ao painel de concluso.
   * @param str texto a ser adicionado
   */
  private void appendConclusion(String str) {
    String text = containerConclusion.getText();
    text = text+str;
    containerConclusion.setText(text);
    containerConclusion.getEConclusion().setText(text+"<BR></BODY></HTML>");
  }

  @Override
  protected boolean canConfirm() {
    return true;
  }

  @Override
  protected Object confirm(History history)
  throws WizardException {
    return data;
  }

  @Override
  protected void enter(History history)
  throws WizardException {
    data = (ProjectSynchronizationData)history.get(null, "data");
    containerConclusion.getEConclusion().setText("");
    containerConclusion.setText("");
    String serverNameA = (String)history.get(null, "serverNameA");
    String serverNameB = (String)history.get(null, "serverNameB");
    if (!generateAbsoluteFilenames(data.getUnitA(), serverNameA,
      data.getUnitB(), serverNameB)) {
      if (error != null) {
        history.addGlobal("error", error);
      }
      return;
    }
    if (!generateAbsoluteFilenames(data.getUnitB(), serverNameB,
      data.getUnitA(), serverNameA)) {
      if (error != null) {
        history.addGlobal("error", error);
      }
      return;
    }
  }

  /**
   * Gera o caminho absoluto de cada arquivo que ser transferido e
   * exibe na interface para que o usurio confirme a transferncia.
   *
   * @param unitFrom os dados do projeto origem
   * @param serverNameFrom nome do servidor de origem
   * @param unitTo os dados do projeto destino
   * @param serverNameTo nome do servidor de destino
   * 
   * @return verdadeiro se conseguiu montar os caminhos e falso caso
   * contrrio.
   */
  private boolean generateAbsoluteFilenames(final ProjectSynchronizationUnit unitFrom,
    String serverNameFrom, final ProjectSynchronizationUnit unitTo, String serverNameTo) {
    final ArrayList<String> filenames = new ArrayList<String>();
    final StringBuffer conclusionBuffer = new StringBuffer();
    String msg =
      MessageFormat.format(
        registry.getString("step5.check"),
        new Object[] {
          unitFrom.getProjectId(), serverNameFrom, unitTo.getProjectId(),
          serverNameTo
        });
    conclusionBuffer.append(msg+"<BR><BR>");
    LocalTask<Void> task = new LocalTask<Void>() {
        @Override
        protected void performTask() throws Exception {
          // Primeiro cria os paths e coloca numa lista.
          Map map = unitFrom.getFiles();
          for (Iterator i = map.values().iterator(); i.hasNext();) {
            ProjectSynchronizationFileInfo psfi =
              (ProjectSynchronizationFileInfo)i.next();
            if (psfi.getTransfer()) {
              String fileName = psfi.generateAbsoluteFilename();
              setStepText(fileName);
              filenames.add(fileName);
            }
          }
          // Depois da lista pronta, ordena seus elementos e coloca na interface.
          setStepText(registry.getString("step5.buildingConclusionBuffer"));
          Collections.sort(filenames, StringUtils.getPlainSortComparator());
          for (String filename : filenames) {
              conclusionBuffer.append(filename+"<BR>");
          }
        }
      };
    if (!task.execute(null, getTitle(),
      registry.getString("step5.getAbsFilenames"), 0,
        LocalTask.CANCEL_BUTTON, false, true)) {
      if (!task.wasCancelled()) {
        Exception re = task.getError();
        error = re.getMessage();
      }
      return false;
    }
    conclusionBuffer.append("<BR><HR>");
    appendConclusion(conclusionBuffer.toString());
    return true;
  }

  @Override
  protected String getConfirmActionName() {
    return registry.getString("step5.sinchronize.button");
  }

  @Override
  protected String getConfirmActionTooltip() {
    return registry.getString("step5.sinchronize.button.tip");
  }

  @Override
  protected Container getContainer() {
    return containerConclusion.getRootPanel();
  }

  @Override
  protected String getInstruction() {
    if (error == null) {
      return registry.getString("instruction.step5");
    }
    return error;
  }

  @Override
  protected String getTitle() {
    return registry.getString("title.step5");
  }

  @Override
  protected Class getNext(History arg0) {
    return null;
  }

  @Override
  protected void exit(History arg0) {}

  /**
   * Retorna o componete da com o resumo da sincronizao.
   * @return o componete da com o resumo da sincronizao
   */
  public ContainerConclusion getContainerConclusion() {
    return containerConclusion;
  }
}
