package csbase.client.csdk.v1_0.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;

import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.IFileLock;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.Task;

/**
 * Representa um lock de um {@link IFile arquivo} local.
 */
public class CSDKLocalFileLock implements IFileLock {

  /**
   * Lock real.
   */
  private final FileLock lock;

  /**
   * Canal para manipulao do arquivo.
   */
  private FileChannel channel;

  /**
   * Construtor.
   * 
   * @param file o arquivo a ser bloqueado.
   * @param shared determina se o lock deve ser do tipo compartilhado.
   * @param window janela me da ao de lock.
   * @throws Exception em caso de erro ao criar o lock.
   */
  public CSDKLocalFileLock(final File file, final boolean shared,
    final Window window) throws Exception {
    if (file.isDirectory()) {
      throw new Exception("A directory cannot be locked ("
        + file.getAbsolutePath() + ")");
    }
    Task<FileLock> task = new Task<FileLock>() {
      @Override
      protected void performTask() {
        FileLock fileLock = _lock(shared, window, file);
        setResult(fileLock);
      }
    };
    String prefix = CSDKLocalFileLock.class.getSimpleName();
    String key;
    if (shared) {
      key = ".shared.lock.message";
    }
    else {
      key = ".exclusive.lock.message";
    }
    String message = LNG.get(prefix + key);
    task.execute(window, null, message);
    this.lock = task.getResult();
    if (task.wasCancelled()) {
      releaseLock(window);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LockStatus getLockStatus() {
    if (lock != null) {
      if (this.lock.isValid()) {
        if (this.lock.isShared()) {
          return LockStatus.LOCK_SHARED;
        }
        else {
          return LockStatus.LOCK_EXCLUSIVE;
        }
      }
      else {
        return LockStatus.LOCK_RELEASED;
      }
    }
    else {
      return LockStatus.LOCK_DENIED;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void releaseLock(final Window window) {
    Task<Void> task = new Task<Void>() {
      @Override
      protected void performTask() throws Exception {
        try {
          _releaseLock(lock);
        }
        finally {
          if(channel != null) {
            channel.close();
          }
        }
      }
    };
    String prefix = CSDKLocalFileLock.class.getSimpleName();
    String message = LNG.get(prefix + ".release.lock.message");
    task.execute(window, null, message);
  }

  /**
   * Obtm o lock do arquivo.
   * 
   * @param shared determina se o lock deve ser do tipo compartilhado.
   * @param window janela me da ao de lock.
   * @param file arquivo a ser bloqueado.
   * @return o lock do arquivo.
   */
  @SuppressWarnings("resource")
  private FileLock _lock(final boolean shared, final Window window,
    final File file) {
    RandomAccessFile randomAccessFile = null;
    try {
      final String mode = shared ? "r" : "rws";
      randomAccessFile = new RandomAccessFile(file, mode);
      this.channel = randomAccessFile.getChannel();
      return this.channel.tryLock(0, Long.MAX_VALUE, shared);
    }
    catch (Exception ex) {
      if(randomAccessFile != null) {
        try {
          randomAccessFile.close();
        }
        catch (IOException e) {
          e.printStackTrace();
        }
      }
      return null;
    }
  }

  /**
   * Libera o lock.
   * 
   * @param fileLock o lock.
   * @throws IOException em caso de erro ao liberar o lock;
   */
  private void _releaseLock(final FileLock fileLock) throws IOException {
    if (fileLock != null && fileLock.isValid()) {
      fileLock.release();
    }
  }

}
