/**
 * $Id$
 */

package csbase.client.project;

import java.awt.Window;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.logic.ClientFileLockListener;
import csbase.logic.ClientProjectFile;
import csbase.logic.FileLockEvent;

/**
 * Representa um arquivo com as operaes de lock no cliente. Esta classe obtm
 * o lock e caso ocorra algum problema na obteno, o remove. Tambm gerencia o
 * listener responsvel pelas notificaes de lock vindas do servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ClientFileLock {

  /**
   * Indica o tipo de lock obtido para um determinado arquivo.
   */
  public enum LockStatus {
    /**
     * Lock exclusivo
     */
    LOCK_EXCLUSIVE,
    /**
     * No foi possvel obter lock para o arquivo.
     */
    LOCK_DENIED,
    /**
     * Lock compartilhado
     */
    LOCK_SHARED,
    /**
     * Pedido do lock foi removido
     */
    LOCK_RELEASED
  }

  /** Representa o status do lock do arquivo. */
  private LockStatus lockStatus;

  /** Arquivo sobre o qual se tenta obter o lock. */
  private ClientProjectFile file;

  /** Observador do lock cadastrado no servidor. */
  private ClientFileLockListener listener;

  /** Identificador do lock. */
  private Object lockId;

  /**
   * Constri uma representao de lock de arquivo no cliente.
   * 
   * @param file arquivo sobre o qual se tenta obter o lock
   */
  private ClientFileLock(ClientProjectFile file) {
    this.file = file;
    this.lockStatus = LockStatus.LOCK_DENIED;
  }

  /**
   * Tenta obter um lock de exclusivo para o arquivo. Locks exclusivos so
   * obtidos apenas uma vez.
   * 
   * Este mtodo cadastra uma solicitao de lock numa fila priorizada pela
   * ordem de chegada dos pedidos de lock e espera pela notificao de que o
   * lock foi obtido ou seu timeout foi atingido. Quando o lock  obtido, o
   * mtodo FileLockListenerInterface.fileLocked  chamado. Quando o tempo
   * mximo de espera do lock  atingido e o lock no foi obtido o mtodo
   * FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param window janela pai para a RemoteTask
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timeout infinito.
   */
  private void acquireExclusiveLock(Window window, final long timeout) {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        lockId = file.acquireExclusiveLock(listener, timeout);
        listener = new ClientFileLockListener(lockId);
        MessageProxy.addListener(listener, FileLockEvent.class);
        setResult(listener.waitLock());
      }
    };

    if (task.execute(window, getWindowTitle(), String.format(LNG
      .get("ClientFileLock.acquire.exclusive.lock.msg"), file.getName()))) {
      if (task.getResult()) {
        lockStatus = LockStatus.LOCK_EXCLUSIVE;
        MessageProxy.removeListener(listener);
      }
    }
    else {
      if (task.wasCancelled()) {
        cancelLock(window);
        MessageProxy.removeListener(listener);
      }
    }
  }

  /**
   * Tenta obter um lock de exclusivo para o arquivo. Locks exclusivos so
   * obtidos apenas uma vez.
   * 
   * @param window janela pai para a RemoteTask
   */
  private void acquireExclusiveLock(Window window) {
    RemoteTask<Object> task = new RemoteTask<Object>() {
      @Override
      protected void performTask() throws Exception {
        lockId = file.acquireExclusiveLock();
        setResult(lockId);
      }
    };

    if (task.execute(window, getWindowTitle(), String.format(LNG
      .get("ClientFileLock.acquire.exclusive.lock.msg"), file.getName()))) {
      if (task.getResult() != null) {
        lockStatus = LockStatus.LOCK_EXCLUSIVE;
      }
    }
    else {
      if (task.wasCancelled()) {
        cancelLock(window);
      }
    }
  }

  /**
   * Tenta obter lock compartilhado para o arquivo. Locks compartilhados podem
   * ser obtidos mltiplas vezes.
   * 
   * Este mtodo cadastra uma solicitao de lock compartilhado em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock e espera pela
   * notificao de que o lock foi obtido ou seu timeout foi atingido. Quando o
   * lock  obtido, o mtodo FileLockListenerInterface.fileLocked  chamado.
   * Quando o tempo mximo de espera do lock  atingido e o lock no foi obtido
   * o mtodo FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param window janela pai para a RemoteTask
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timout infinito.
   */
  private void acquireSharedLock(Window window, final long timeout) {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        lockId = file.acquireSharedLock(listener, timeout);
        listener = new ClientFileLockListener(lockId);
        MessageProxy.addListener(listener, FileLockEvent.class);
      }
    };

    if (task.execute(window, getWindowTitle(), String.format(LNG
      .get("ClientFileLock.acquire.shared.lock.msg"), file.getName()))) {
      if (task.getResult()) {
        lockStatus = LockStatus.LOCK_SHARED;
        MessageProxy.removeListener(listener);
      }
    }
    else {
      if (task.wasCancelled()) {
        cancelLock(window);
        MessageProxy.removeListener(listener);
      }
    }
  }

  /**
   * Tenta obter um lock compartilhado para o arquivo.
   * 
   * @param window janela pai para a RemoteTask
   */
  private void acquireSharedLock(Window window) {
    RemoteTask<Object> task = new RemoteTask<Object>() {
      @Override
      protected void performTask() throws Exception {
        lockId = file.acquireSharedLock();
        setResult(lockId);
      }
    };

    if (task.execute(window, getWindowTitle(), String.format(LNG
      .get("ClientFileLock.acquire.shared.lock.msg"), file.getName()))) {
      if (task.getResult() != null) {
        lockStatus = LockStatus.LOCK_SHARED;
      }
    }
    else {
      if (task.wasCancelled()) {
        cancelLock(window);
      }
    }
  }

  /**
   * Chama a task que remove o lock do arquivo.
   * 
   * @param window janela pai para a RemoteTask
   */
  private void _releaseLock(Window window) {
    if (lockId == null) {
      return;
    }
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        file.releaseLock(lockId);
      }
    };

    // Exibe a task, mas demora a aparece o boto de cancelar para que o
    // o usurio no cancele a remoo do lock.
    task.execute(window, getWindowTitle(), String.format(LNG
      .get("ClientFileLock.release.lock.msg"), file.getName()), 300,
      Task.CANCEL_BUTTON);
    lockStatus = LockStatus.LOCK_RELEASED;
  }

  /**
   * Cancela o lock quando o usurio cancelou o pedido de lock.
   * 
   * @param window janela pai para a RemoteTask
   */
  private void cancelLock(Window window) {
    if (listener != null) {
      MessageProxy.removeListener(listener);
    }
    _releaseLock(window);
  }

  /**
   * Retorna o ttulo da janela
   * 
   * @return ttulo da janela
   */
  private String getWindowTitle() {
    return LNG.get("ClientFileLock.window.title");
  }

  /**
   * Remove o lock do arquivo se tem o lock compartilhado ou exclusivo.
   * 
   * @param window janela pai para a RemoteTask
   */
  public void releaseLock(Window window) {
    if (lockStatus == LockStatus.LOCK_EXCLUSIVE
      || lockStatus == LockStatus.LOCK_SHARED) {
      _releaseLock(window);
    }
  }

  /**
   * Obtm o status do lock do arquivo
   * 
   * @return o status do lock do arquivo
   */
  public LockStatus getLockStatus() {
    return lockStatus;
  }

  /**
   * Tenta obter lock exclusivo para o arquivo. Locks exclusivo pode ser obtido
   * apenas um por vez, por arquivo.
   * 
   * Este mtodo cadastra uma solicitao de lock exclusivo em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock e espera pela
   * notificao de que o lock foi obtido ou seu timeout foi atingido.
   * 
   * Caso a RemoteTask de obteno de lock seja cancelada, chama o mtodo para
   * remover o pedido de lock no servidor
   * 
   * @param window janela pai para a RemoteTask
   * @param file arquivo sobre o qual se tenta obter o lock
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timout infinito.
   * @return ClientFileLock que possui o status do lock
   */
  public static ClientFileLock acquireExclusiveLock(Window window,
    ClientProjectFile file, long timeout) {
    ClientFileLock clientFileLock = new ClientFileLock(file);
    clientFileLock.acquireExclusiveLock(window, timeout);
    return clientFileLock;
  }

  /**
   * Tenta obter lock exclusivo para o arquivo. Locks exclusivo pode ser obtido
   * apenas um por vez, por arquivo.
   * 
   * Caso a RemoteTask seja cancelada, chama o mtodo para remover o lock no
   * servidor
   * 
   * @param window janela pai para a RemoteTask
   * @param file arquivo sobre o qual se tenta obter o lock
   * @return ClientFileLock que possui o status do lock
   */
  public static ClientFileLock acquireExclusiveLock(Window window,
    ClientProjectFile file) {
    ClientFileLock clientFileLock = new ClientFileLock(file);
    clientFileLock.acquireExclusiveLock(window);
    return clientFileLock;
  }

  /**
   * Tenta obter lock compartilhado para o arquivo. Locks compartilhados podem
   * ser obtidos mltiplas vezes.
   * 
   * Este mtodo cadastra uma solicitao de lock compartilhado em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock e espera pela
   * notificao de que o lock foi obtido ou seu timeout foi atingido.
   * 
   * Caso a RemoteTask de obteno de lock seja cancelada, chama o mtodo para
   * remover o pedido de lock no servidor
   * 
   * @param window janela pai para a RemoteTask
   * @param file arquivo sobre o qual se tenta obter o lock
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timout infinito.
   * @return ClientFileLock que possui o status do lock
   */
  public static ClientFileLock acquireSharedLock(Window window,
    ClientProjectFile file, long timeout) {
    ClientFileLock clientFileLock = new ClientFileLock(file);
    clientFileLock.acquireSharedLock(window, timeout);
    return clientFileLock;
  }

  /**
   * Tenta obter lock compartilhado para o arquivo. Locks compartilhados podem
   * ser obtidos mltiplas vezes.
   * 
   * Caso a RemoteTask seja cancelada, chama o mtodo para remover o lock no
   * servidor
   * 
   * @param window janela pai para a RemoteTask
   * @param file arquivo sobre o qual se tenta obter o lock
   * @return ClientFileLock que possui o status do lock
   */
  public static ClientFileLock acquireSharedLock(Window window,
    ClientProjectFile file) {
    ClientFileLock clientFileLock = new ClientFileLock(file);
    clientFileLock.acquireSharedLock(window);
    return clientFileLock;
  }
}
