package csbase.client.project;

import java.awt.Window;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.action.ProjectCloseAction;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.CommonClientProject;
import csbase.logic.UserProjectInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.CommandPersistenceServiceInterface;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * A classe <code>CommonProjectDeleteDialog</code> usa o dilogo com todos os
 * projetos de um usuario para remocao de um.
 */
public class CommonProjectDeleteDialog extends CommonProjectSelectDialog {

  /**
   * Executa a ao de remover o projeto. Chama a ao de fechar o projeto caso
   * ele esteja aberto.
   */
  @Override
  protected void handleAction() {
    if (!confirmDeletion()) {
      return;
    }
    CommonClientProject openProject = DesktopFrame.getInstance().getProject();
    ProjectTree tree = DesktopFrame.getInstance().getTree();

    // Consulta a lista de projetos selecionados.
    final List<UserProjectInfo> selectedProjects = getSelectedProjects();

    // Verifica se algum dos projetos selecionados est aberto.
    // Se tiver, fecha
    for (UserProjectInfo userProjectInfo : selectedProjects) {
      if (openProject != null
        && openProject.getId().equals(userProjectInfo.getProjectId())) {
        ProjectCloseAction pca = new ProjectCloseAction(tree);
        pca.close();
      }
    }

    // Remove os projetos
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        for (UserProjectInfo userProjectInfo : selectedProjects) {
          CommonClientProject.remove(userProjectInfo.getProjectId());
        }
      }
    };

    String msg = LNG.get("CommonProjectSelectDialog.info.remove.project");
    if (task.execute(getOwner(), getTitle(), msg)) {
      close();
    }
  }

  /**
   * Exibe a janela de confirmao.
   *
   * @return confirmao.
   */
  private boolean confirmDeletion() {
    final String title = getTitle();
    final Window owner = getOwner();
    String msg;
    Object[] options = { LNG.get("CommonProjectDeleteDialog.delete.button"),
        LNG.get("CommonProjectDeleteDialog.cancel.button") };

    // Consulta a lista de projetos selecionados.
    final List<UserProjectInfo> selectedProjects = getSelectedProjects();

    // Verifica se algum dos projetos contm comandos em execuo ou na fila.
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        CommandPersistenceServiceInterface cpsi =
          ClientRemoteLocator.commandPersistenceService;
        for (UserProjectInfo project : selectedProjects) {
          Set<CommandInfo> runningCommands =
            cpsi.getStatusCommandInfos(project.getProjectId(),
              Arrays.asList(CommandStatus.FINISHED), false);
          if (runningCommands.size() > 0) {
            setResult(false);
            return;
          }
        }
        setResult(true);
      }
    };

    msg = LNG.get("CommonProjectSelectDialog.info.remove.project");
    if (!task.execute(getOwner(), getTitle(), msg) || !task.getResult()) {
      msg = LNG.get("CommonProjectSelectDialog.error.running.commands");
      StandardDialogs.showWarningDialog(owner, title, msg);
      return false;
    }

    // Faz mensagem de acordo com a quantidade selecionada.
    if (selectedProjects.size() > 1) {
      msg = LNG.get("CommonProjectSelectDialog.info.confirm.removal.n");
    }
    else {
      UserProjectInfo userProjectInfo = selectedProjects.get(0);
      String fmt = LNG.get("CommonProjectSelectDialog.info.confirm.removal");
      msg = String.format(fmt, userProjectInfo.getProjectName());
    }

    int opt = StandardDialogs.showOptionDialog(owner, title, msg, options);
    if (opt != 0) {
      return false;
    }
    return true;
  }

  /**
   * Constri uma viso de dilogo para remoo de projetos de um um usurio.
   *
   * @param owner A janela a partir da qual esse dilogo foi chamado.
   */
  public CommonProjectDeleteDialog(Window owner) {
    super(owner, false);
    setTitle(LNG.get("CommonProjectSelectDialog.title.delete"));
    setDefaultTextButtonText(
      LNG.get("CommonProjectDeleteDialog.delete.button"));
  }
}