package csbase.client.project;

import java.awt.Component;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationImages;
import csbase.client.project.ProjectTableModel.RowData;
import csbase.client.project.ProjectTableModel.VisibleColumns;

/**
 * Tabela de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectTableSortableTable extends SortableTable {

  /**
   * Construtor.
   * 
   * @param projects Lista com os pares (usurio, nome do projeto) que sero
   *        listados pela tabela.
   * @param visibleColumns colunas visveis
   * 
   * @see ProjectTableModel.RowData
   */
  public ProjectTableSortableTable(List<RowData> projects,
    VisibleColumns visibleColumns) {
    super(new ProjectTableModel(projects, visibleColumns));
    setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    setRowSelectionAllowed(true);
    setFillsViewportHeight(true);
    setDefaultRenderer(Boolean.class, createBooleanRenderer());
  }

  /**
   * Obtm o par (usurio, nome do projeto) associado a linha que est
   * selecionada na tabela.
   * 
   * @return Par (usurio, nome do projeto) associado  linha.
   * @see ProjectTableModel.RowData
   */
  public RowData getSelectedRowData() {
    ProjectTableModel model = (ProjectTableModel) getModel();
    return model.getRow(convertRowIndexToModel(getSelectedRow()));
  }

  /**
   * Troca as os projetos atuais da tabela por novos projetos.
   * 
   * @param projectList Lista com os pares (usurio, nome do projeto) que sero
   *        listados pela tabela.
   * @see ProjectTableModel.RowData
   */
  public void setNewValues(List<RowData> projectList) {
    ProjectTableModel model = (ProjectTableModel) getModel();
    model.replaceRows(projectList);
  }

  /**
   * Redefine as colunas visveis.
   * 
   * @param visibleColumns colunas visveis
   */
  public void setVisibleColumns(VisibleColumns visibleColumns) {
    ProjectTableModel model = (ProjectTableModel) getModel();
    model.setVisibleColumns(visibleColumns);
    if (visibleColumns == VisibleColumns.OWNER_NAME_PERMISSION) {
      TableColumn column = getColumnModel().getColumn(getColumnCount() - 1);
      column.setMaxWidth(50);
    }
  }

  /**
   * Seleciona a 1a. linha da tabela, se a tabela no for vazia.
   */
  public void setSelectedFirstRow() {
    if (getRowCount() > 0) {
      setRowSelectionInterval(0, 0);
    }
  }

  /**
   * Cria renderizador customizado para coluna que exibe se o usurio tem
   * permisso para escrita no projeto.
   * 
   * @return renderizador.
   */
  private TableCellRenderer createBooleanRenderer() {
    return new DefaultTableCellRenderer() {
      @Override
      public Component getTableCellRendererComponent(JTable table,
        Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        setHorizontalAlignment(JLabel.CENTER);
        if (value == null) {
          setText("ERROR");
          return this;
        }

        if ((Boolean) value) {
          setIcon(ApplicationImages.ICON_OK_16);
        }
        else {
          setIcon(null);
        }
        return this;
      }
    };
  }
}
