package csbase.client.algorithms.tasks;

import java.awt.Window;

import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.Validation;
import tecgraf.javautils.core.lng.LNG;

/**
 * Tarefa de validao de um configurador de algoritmo.
 */
public class ConfiguratorValidationTask extends RemoteTask<Validation> {

  /**
   * Configurador do algoritmo a ser validado.
   */
  protected AlgorithmConfigurator configurator;

  /**
   * Modo de validao.
   */
  private ValidationMode mode;

  /**
   * Construtor.
   * 
   * @param configurator o configurador do algoritmo a ser validado.
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   */
  public ConfiguratorValidationTask(AlgorithmConfigurator configurator,
    ValidationMode mode) {
    this.configurator = configurator;
    this.mode = mode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    CommonClientProject project = DesktopFrame.getInstance().getProject();

    Object projectId = project.getId();
    Object userId = User.getLoggedUser().getId();
    ValidationContext context = new ValidationContext(mode, projectId, userId);

    Validation result = configurator.validate(context);
    setResult(result);
  }

  /**
   * Mtodo de convenincia para chamada automtica da task.
   * 
   * @param window janela dona da task.
   * @param configurator a configurador a ser validado.
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   * 
   * @return o resultado da validao.
   */
  final static public Validation runTask(Window window,
    AlgorithmConfigurator configurator, ValidationMode mode) {
    ConfiguratorValidationTask task =
      new ConfiguratorValidationTask(configurator, mode);
    final String className = ConfiguratorValidationTask.class.getSimpleName();
    final String title = LNG.get(className + ".title");
    final String msg = LNG.get(className + ".msg");
    if (task.execute(window, title, msg)) {
      return task.getResult();
    }
    else if (task.wasCancelled()) {
      LocalizedMessage message =
        new LocalizedMessage(ConfiguratorValidationTask.class, "cancelled");
      return new ValidationError(message);
    }
    else {
      Exception error = task.getError();
      error.printStackTrace();
      LocalizedMessage errorMessage =
        new LocalizedMessage(ConfiguratorValidationTask.class, "error",
          new Object[] { error.getMessage() });
      return new ValidationError(errorMessage);
    }
  }
}
