package csbase.client.algorithms.commands.newview;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Tarefa que busca os logs de um comando a partir de padres (pattern-matching)
 * de nomes de arquivo configurados no {@link ConfigurationManager}.
 */
class MatchLogTabConfigurationsTask extends
  RemoteTask<Map<LogTabConfiguration, Set<ClientProjectFile>>> {

  /**
   * Caminho para o diretrio onde os logs so gravados.
   */
  protected String[] logsDir;

  /**
   * Configuraes das abas de log.
   */
  private List<LogTabConfiguration> configuration;

  /**
   * Construtor.
   * 
   * @param logsDir Caminho para o diretrio onde os logs so gravados (No
   *        aceita {@code null}).
   * @param configuration Configuraes das abas de log.
   */
  public MatchLogTabConfigurationsTask(String[] logsDir,
    List<LogTabConfiguration> configuration) {
    if (logsDir == null) {
      throw new IllegalArgumentException("O parmetro logsDir est nulo.");
    }
    this.logsDir = logsDir;
    this.configuration = configuration;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws IOException,
    ConfigurationManagerException {
    setResult(matchFilesToConfiguration());
  }

  /**
   * Monta a lista de configuraes das abas de log, determinando quais arquivos
   * devem aparecer em cada aba, de acordo com o que foi configurado via
   * {@link ConfigurationManager}.
   * 
   * @return A lista de configuraes.
   * @throws ConfigurationManagerException casa ocorra erro na leitura das
   *         configuraes.
   * @throws RemoteException caso ocorra erra na leitura dos diretrios de log.
   */
  private Map<LogTabConfiguration, Set<ClientProjectFile>> matchFilesToConfiguration()
    throws ConfigurationManagerException, RemoteException {

    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      return Collections.emptyMap();
    }

    ClientProjectFile logsBaseDir = project.getFile(logsDir);
    if (logsBaseDir == null) {
      return Collections.emptyMap();
    }

    ClientProjectFile[] children = logsBaseDir.getChildren();
    if (children == null) {
      return Collections.emptyMap();
    }

    Map<LogTabConfiguration, Set<ClientProjectFile>> configurations =
      new HashMap<LogTabConfiguration, Set<ClientProjectFile>>();

    for (LogTabConfiguration cnf : configuration) {
      Set<ClientProjectFile> matchingFiles =
        findLogFiles(children, cnf.getFilePatterns());
      configurations.put(cnf, matchingFiles);
    }
    return configurations;
  }

  /**
   * Filtra um array de arquivos e retorna um subconjunto com os arquivos de
   * log. A filtragem  feita de acordo com o padres de nomeclatura
   * configurados para arquivos que devem ser mostrados em cada aba do sistema.
   * 
   * @param files O array de arquivos a ser filtrado.
   * @param patterns A lista de padres de nomenclatura de arquivos
   *        configurados.
   * @return O conjunto de arquivos de log.
   * @throws RemoteException Se no for possvel obter os arquivos.
   */
  private Set<ClientProjectFile> findLogFiles(final ClientProjectFile[] files,
    final List<String> patterns) throws RemoteException {
    final Set<ClientProjectFile> logFiles = new TreeSet<ClientProjectFile>();
    for (final ClientProjectFile file : files) {
      if (file.isDirectory()) {
        // segue a busca recursivamente pelos diretrios filhos.
        Set<ClientProjectFile> matchedFilesInDir =
          findLogFiles(file.getChildren(), patterns);
        logFiles.addAll(matchedFilesInDir);
      }
      else if (matchesPatterns(file, patterns)) {
        logFiles.add(file);
      }
    }
    return logFiles;
  }

  /**
   * Testa se um determinado arquivo satisfaz algum dos padres de nomenclatura
   * para arquivos de log definidos para uma aba deste sistema. Caso positivo, o
   * arquivo deve ser mostrado na aba.
   * 
   * @param file O arquivo a ser testado.
   * @param logFilePatterns A lista de padres de nomenclatura para arquivos de
   *        log de uma aba.
   * @return Verdadeiro se o arquivo pode ser considerado um log de acordo com o
   *         padro de nomenclatura definido ou falso, caso contrrio.
   */
  private boolean matchesPatterns(final ClientProjectFile file,
    final List<String> logFilePatterns) {
    if (logFilePatterns != null) {
      for (String pattern : logFilePatterns) {
        if (file.getName().matches(pattern.trim())) {
          return true;
        }
      }
    }
    return false;
  }

}
