/*
 * $Id$
 */
package csbase.client.desktop;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.dircontents.DirectoryContentsPanel;
import csbase.client.project.ProjectFileTypeComboBox;
import csbase.client.project.ProjectTree;
import csbase.client.util.CountDown;
import csbase.logic.ProjectFileFilter;
import csbase.logic.filters.ProjectFileCompositeAndFilter;
import csbase.logic.filters.ProjectFileNameFilter;
import csbase.logic.filters.ProjectFileRecursiveFilter;
import csbase.logic.filters.ProjectFileTypeFilter;

/**
 * Painel de filtro para a rvore de projetos. Esta classe  um
 * <i>singleton</i>.
 * 
 * @author Tecgraf
 */
public class TreeFilterPanel extends JPanel {
  /**
   * Tempo em que aps o ltimo caracter digitado, o campo de texto deve ficar
   * ocioso para que comece a filtragem.
   */
  private static final long TEXTFIELD_COUNT_DOWN = 1;
  /**
   * Unidade de tempo do {@link #TEXTFIELD_COUNT_DOWN}.
   */
  private static final TimeUnit TEXTFIELD_COUNT_DOWN_UNIT = TimeUnit.SECONDS;

  /** Instncia nica da classe. */
  private static TreeFilterPanel instance;

  /** Caixa de texto para o filtro por nome do arquivo. */
  private JTextField filterTextField;
  /** Caixa de seleo para o filtro por tipo de arquivo. */
  private ProjectFileTypeComboBox typeComboBox;
  /** Referncia para o painel de detalhes da rvore de projetos. */
  private DirectoryContentsPanel dirContentsPanel;
  /** Referncia para a rvore de projetos. */
  private ProjectTree projectTree;
  /** Boto para limpar a caixa de texto contendo o critrio de filtro. */
  private JButton clearFieldButton;

  /**
   * Conjunto de objetos interessados em mudanas no estado deste
   * <i>singleton</i>.
   */
  private List<TreeFilterPanelListener> listeners =
    new ArrayList<TreeFilterPanelListener>();

  /** Dispatcher para eventos de teclado que atua como filtro de input. */
  private KeyEventDispatcher filterInputDispatcher;

  /**
   * Cria o painel de filtro.
   * 
   * @param tree referncia para a rvore de projetos.
   * @param panel referncia para a janela de detalhes da rvore de projetos.
   */
  private TreeFilterPanel(ProjectTree tree, DirectoryContentsPanel panel) {
    super(new GridBagLayout());
    this.projectTree = tree;
    this.dirContentsPanel = panel;
    filterTextField = new JTextField(10);
    createfilterInputDipatcher(filterTextField);

    filterTextField.getDocument().addDocumentListener(new DocumentListener() {
      public void changedUpdate(DocumentEvent e) {
        update();
      }

      public void removeUpdate(DocumentEvent e) {
        update();
      }

      public void insertUpdate(DocumentEvent e) {
        update();
      }

      public void update() {
        clearFieldButton.setEnabled(filterTextField.getText().length() > 0);
      }
    });
    filterTextField.setToolTipText(LNG
      .get("TreeFilterPanel.filterTextField.tooltip"));
    filterTextField.addKeyListener(new KeyAdapter() {
      final CountDown countDown = new CountDown(TEXTFIELD_COUNT_DOWN,
        TEXTFIELD_COUNT_DOWN_UNIT, new Runnable() {
          @Override
          public void run() {
            SwingThreadDispatcher.invokeLater(new Runnable() {
              public void run() {
                filterTree();
              }
            });
          }
        });

      @Override
      public void keyReleased(KeyEvent e) {
        countDown.restart();
      }
    });
    clearFieldButton = new JButton(ApplicationImages.ICON_CLEARMSG_16);
    clearFieldButton.setMargin(new Insets(0, 0, 0, 0));
    clearFieldButton.setEnabled(false);
    clearFieldButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterTextField.setText("");
        clearFieldButton.setEnabled(false);
        filterTextField.requestFocusInWindow();
        filterTree();
      }
    });
    typeComboBox = new ProjectFileTypeComboBox(
      ProjectFileTypeComboBox.Mode.FILE_AND_DIRECTORY, true);
    typeComboBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterTree();
      }
    });
    JPanel entryPanel = new JPanel(new GridBagLayout());
    Insets margins = new Insets(5, 5, 5, 5);
    Border border =
      BorderFactory.createTitledBorder(LNG.get("TreeFilterPanel.border.title"));
    entryPanel.setBorder(border);
    entryPanel.add(new JLabel(LNG.get("TreeFilterPanel.label.name")), new GBC(
      0, 0).insets(margins).west());
    entryPanel.add(filterTextField, new GBC(1, 0).insets(margins).west()
      .weightx(1).horizontal());
    entryPanel.add(clearFieldButton, new GBC(2, 0).insets(margins).west());
    entryPanel.add(new JLabel(LNG.get("TreeFilterPanel.label.type")), new GBC(
      0, 1).insets(margins).west());
    entryPanel.add(typeComboBox, new GBC(1, 1).insets(margins).west()
      .weightx(1).width(2).horizontal());
    this.add(entryPanel, new GBC(0, 0).both());
  }

  /**
   * Adiciona o dispatcher de filtro de input ao KeyboardFocusManager.
   */
  private void addFilterInputDispatcher() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager()
      .addKeyEventDispatcher(filterInputDispatcher);
  }

  /**
   * Remove o dispatcher de filtro de input do KeyboardFocusManager.
   */
  private void removeFilterInputDispatcher() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager()
      .removeKeyEventDispatcher(filterInputDispatcher);
  }

  /**
   * Cria um dispatcher para eventos de teclado que atue como um filtro de
   * entrada para o componente especificado. O usurio no conseguir digitar
   * nada alm dos caracteres permitidos.
   * 
   * @param cmp componente cuja entrada ser filtrada.
   */
  private void createfilterInputDipatcher(final Component cmp) {
    filterInputDispatcher = new KeyEventDispatcher() {
      /**
       * Desconsidera os eventos de teclado que forem dirigidos para um
       * determinado componente e no forem considerados "vlidos".
       * 
       * @param e informaes sobre o evento ocorrido.
       * 
       * @return true se o evento puder ser repassado, false se ele tiver que
       *         ser descartado.
       */
      @Override
      public boolean dispatchKeyEvent(KeyEvent e) {
        boolean discardEvent = false;
        if (e.getID() == KeyEvent.KEY_TYPED) {
          if (cmp.isFocusOwner() && !isValid(e.getKeyChar())) {
            discardEvent = true;
          }
        }
        return discardEvent;
      }

      /**
       * Verifica se o caracter especificado  vlido. So considerados
       * caracteres vlidos dgitos, letras, ".", "_", "?", "*" e "$".
       * 
       * @param keyChar caracter a ser verificado.
       * 
       * @return true se o caracter for vlido.
       */
      private boolean isValid(char keyChar) {
        if (Character.isLetterOrDigit(keyChar)) {
          return true;
        }
        else if ("?*$._".indexOf(keyChar) >= 0) {
          return true;
        }
        return false;
      }
    };
  }

  /**
   * Filtra a rvore de projetos, considerando o filtro para nome digitado pelo
   * usurio na caixa de texto e a seleo de tipo de arquivo.
   */
  private void filterTree() {
    final int MAX_LENGTH = 10;
    String nameFilter = filterTextField.getText();
    StringBuilder appliedFilter = new StringBuilder();
    if (nameFilter.length() > MAX_LENGTH) {
      int endIndex = Math.min(nameFilter.length(), MAX_LENGTH);
      appliedFilter.append(nameFilter.substring(0, endIndex));
      appliedFilter.append("(...)");
    }
    else {
      appliedFilter.append(nameFilter);
    }
    String typeFilter;
    if (typeComboBox.isAllItemSelected()) {
      typeFilter = null;
    }
    else {
      typeFilter = typeComboBox.getSelectedTypeCode();
    }
    appliedFilter.append(" " + LNG.get("UTIL_AND").toUpperCase() + " ");
    appliedFilter.append(typeFilter);
    ProjectFileFilter filter;
    if (nameFilter.isEmpty() && typeFilter == null) {
      filter = null;
    }
    else {
      ProjectFileCompositeAndFilter andFilter = new ProjectFileCompositeAndFilter();
      andFilter.addChild(new ProjectFileNameFilter(nameFilter));
      andFilter.addChild(new ProjectFileTypeFilter(typeFilter));
      filter = new ProjectFileRecursiveFilter(andFilter);
    }
    dirContentsPanel.setFilter(filter);
    projectTree.setVisualFilter(filter);
  }

  /**
   * Adiciona um objeto interesado em mudanas no estado deste <i>singleton</i>.
   * 
   * @param l objeto interessado em mudanas neste objeto.
   */
  public void addListener(TreeFilterPanelListener l) {
    listeners.add(l);
  }

  /**
   * Remove um objeto interesado em mudanas no estado deste <i>singleton</i>.
   * 
   * @param l objeto interessado em mudanas neste objeto.
   * 
   * @return true se o objeto tiver sido removido, false caso ele no tenha sido
   *         encontrado na lista.
   */
  public boolean removeListener(TreeFilterPanelListener l) {
    return listeners.remove(l);
  }

  /**
   * <p>
   * Esconde ou exibe o painel, de acordo com o parmetro especificado.
   * </p>
   * <p>
   * Quando exibe, transfere o foco para a caixa de texto. Quando esconde,
   * remove o filtro.
   * </p>
   * <p>
   * Notifica os observadores sobre a mudana no seu estado.
   * </p>
   */
  @Override
  public void setVisible(boolean visible) {
    if (visible == isVisible()) {
      return;
    }
    super.setVisible(visible);
    if (visible) {
      filterTextField.requestFocusInWindow();
      addFilterInputDispatcher();
    }
    else {
      /*
       * como estamos escondendo o painel, devemos desativar o filtro
       * (retornando-o para ALL_FILES caso esteja com outro valor)
       */
      reset();
      removeFilterInputDispatcher();
    }
    for (int i = 0; i < listeners.size(); i++) {
      listeners.get(i).visibilityChanged(isVisible());
    }
  }

  /**
   * Pe o filtro no estado inicial, no qual aceita todos os arquivos.
   */
  public void reset() {
    String filterText = filterTextField.getText();
    //    clearFieldButton.setEnabled(false);
    filterTextField.setText(null);
    if (!filterText.isEmpty()
      || !typeComboBox.isAllItemSelected()) {
      typeComboBox.selectAllItem();
    }
  }

  /**
   * Retorna a instncia nica desta classe.
   * 
   * @param tree referncia para rvore de projetos.
   * @param panel referncia para a janela de detalhes da rvore de projetos.
   * 
   * @return instncia nica da classe.
   */
  public static TreeFilterPanel getInstance(ProjectTree tree,
    DirectoryContentsPanel panel) {
    if (instance == null) {
      instance = new TreeFilterPanel(tree, panel);
    }
    return instance;
  }

}
