package csbase.logic;

import java.io.Serializable;
import java.util.Arrays;

/**
 * <p>
 * Esta classe  responsvel por transmitir dados de um arquivo ou diretrio de
 * projeto, de um objeto para outro ou mesmo do cliente para o servidor. Ela no
 * representa o arquivo/diretrio em si (a interface que implementa esse
 * conceito  a <code>ProjectFile</code>, com implementaes concretas no
 * cliente e no servidor), mas to somente as <i>propriedades</i> de um arquivo,
 * para criao e/ou modificao de um <code>ProjectFile</code>).
 * </p>
 * A propriedade {@link #path} (caminho para o arquivo)  fornecida em relao a
 * um referencial, que pode ser a raiz do sistema de arquivos, a raiz do projeto
 * ou mesmo um diretrio.
 * 
 * @author Leonardo Barros
 */
public class ProjectFileInfo implements Serializable {
  /** Caminho do arquivo/diretrio. */
  private String[] path;
  /** Tipo do arquivo, ou "DIRECTORY_TYPE" se for diretrio. */
  private String type;
  /** Tamanho em bytes */
  private long size;

  /**
   * Cria um objeto que armazena as propriedades de um arquivo/diretrio de
   * projeto.
   * 
   * @param path caminho para o arquivo/diretrio.
   * @param type tipo do arquivo (no so aceitos tipos nulos; se o tipo for
   *        desconhecido, use {@link ProjectFileType#UNKNOWN}).
   */
  public ProjectFileInfo(String[] path, String type) {
    this(path);
    if (type == null) {
      throw new IllegalArgumentException("type == null");
    }
    this.type = type;
  }

  /**
   * Cria um objeto que armazena as propriedades de um arquivo/diretrio de
   * projeto.
   * 
   * @param path caminho para o arquivo/diretrio.
   */
  public ProjectFileInfo(String[] path) {
    if (path == null) {
      throw new IllegalArgumentException("path == null");
    }
    if (path.length == 0) {
      throw new IllegalArgumentException("path.length == 0");
    }
    this.path = path;
  }

  /**
   * Obtm o caminho para o arquivo/diretrio, sem incluir o nome do projeto.
   * 
   * @return path caminho para o arquivo/diretrio, sem incluir o nome do
   *         projeto.
   */
  public String[] getPath() {
    return path;
  }

  /**
   * Substitui o caminho para o arquivo/diretrio.
   * 
   * @param path novo caminho para o arquivo/diretrio.
   */
  public void setPath(String[] path) {
    if (path == null) {
      throw new IllegalArgumentException("path == null");
    }
    if (path.length == 0) {
      throw new IllegalArgumentException("path.length == 0");
    }
    this.path = path;
  }

  /**
   * Obtm o tipo do arquivo/diretrio.
   * 
   * @return type tipo do arquivo/diretrio.
   */
  public String getType() {
    return type;
  }

  /**
   * Substitui o tipo do arquivo/diretrio.
   * 
   * @param type novo tipo do arquivo (no so aceitos tipos nulos; se o tipo
   *        for desconhecido, use {@link ProjectFileType#UNKNOWN}).
   */
  public void setType(String type) {
    if (type == null) {
      throw new IllegalArgumentException("type == null");
    }
    this.type = type;
  }

  /**
   * Obtm o nome do arquivo/diretrio.
   * 
   * @return nome do arquivo/diretrio.
   */
  public String getName() {
    return path[path.length - 1];
  }

  /**
   * Substitui o nome do arquivo/diretrio.
   * 
   * @param name novo nome para o arquivo/diretrio.
   */
  public void setName(String name) {
    if (name == null) {
      throw new IllegalArgumentException("name == null");
    }
    path[path.length - 1] = name;
  }

  /**
   * Obtm o tamanho do arquivo/diretrio.
   * 
   * @return tamanho do arquivo/diretrio.
   */
  public long getSize() {
    return size;
  }

  /**
   * Substitui o tamanho do arquivo/diretrio.
   * 
   * @param size novo tamanho do arquivo/diretrio.
   */
  public void setSize(long size) {
    this.size = size;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (this == o) {
      return true;
    }
    if (!(o instanceof ProjectFileInfo)) {
      return false;
    }
    ProjectFileInfo other = (ProjectFileInfo) o;
    return (Arrays.equals(this.path, other.path) && this.type
      .equals(other.type));
  }

}
