package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;
import java.util.List;

import csbase.logic.algorithms.parameters.validators.TextParameterValidator;

/**
 * Parmetro do tipo Lista de Texto.
 *
 * @author lmoreira
 */
public final class TextListParameter extends ListParameter<String> {

  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE = "LIST_OF_TEXTS";

  /**
   * O nmero mximo de caracteres.
   */
  private Integer maxCharacters;

  /**
   * Cria um parmetro do tipo lista de texto.
   *
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maxCharacters O nmero mximo de caracteres para que o valor de uma
   *        clula seja considerado vlido (Aceita {@code null}).
   */
  public TextListParameter(String name, String label, String description,
    List<String> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maxCharacters) {
    super(name, label, description, defaultValue, isOptional, isVisible, false,
      true, commandLinePattern);
    setMaxCharacters(maxCharacters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getItemValueFromText(String itemValue) {
    return itemValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Obtm o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   *
   * @return O nmero mximo de caracteres ou {@code null} caso no haja um
   *         nmero mximo de caracteres definido.
   */
  public Integer getMaxCharacters() {
    return this.maxCharacters;
  }

  /**
   * Atribui o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   *
   * @param maxCharacters O nmero mximo de caracteres ou {@code null} caso no
   *        haja um nmero mximo de caracteres definido.
   */
  private void setMaxCharacters(Integer maxCharacters) {
    if (maxCharacters != null && maxCharacters.intValue() <= 0) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro maxCharacters deveria ser positivo.\n"
          + "Valor encontrado: {0}.", new Object[] { maxCharacters }));
    }
    this.maxCharacters = maxCharacters;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected TextParameterValidator createItemValidator() {
    return new TextParameterValidator(isOptional(), maxCharacters);
  }
}
