package csbase.logic.algorithms.parameters;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.parameters.validators.URLParameterValidator;

/**
 * Classe base para parmetro do tipo URL.
 * 
 * @author Tecgraf
 */
public abstract class URLParameter extends SimpleParameter<URLValue> {

  /**
   * Chave utilizada para informar o <b>caminho</b> da URL no mapa de parmetros
   * dos mtodos de importao ({@link #importValue(Map)}) e exportao (
   * {@link #exportValue()}).
   */
  private static final String PATH = ".path";

  /**
   * Chave utilizada para informar o <b>tipo</b> da URL no mapa de parmetros
   * dos mtodos de importao ({@link #importValue(Map)}) e exportao (
   * {@link #exportValue()}).
   */
  private static final String TYPE = ".type";

  /**
   * Chave utilizada para informar o <b>protocolo</b> da URL no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static final String PROTOCOL = ".protocol";

  /**
   * Chave utilizada para informar o <b>host</b> da URL no mapa de parmetros
   * dos mtodos de importao ({@link #importValue(Map)}) e exportao (
   * {@link #exportValue()}).
   */
  private static final String HOST = ".host";

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   */
  private String localization;

  /**
   * O tipo de arquivo que este parmetro filtra, {@code null} caso o parmetro
   * desconsidere o tipo do arquivo.
   */
  private String fileType;

  /**
   * O modo de funcionamento atual (apenas arquivos, apenas diretrios ou
   * ambos).
   */
  private FileParameterMode mode;

  /** Indica se a URL deve existir. */
  private boolean mustExist;

  /** Conjunto de protocolos aceitos na URL. */
  private EnumSet<URLProtocol> allowedProtocols;

  /**
   * Construtor.
   * 
   * @param name nome do parmetro (No aceita {@code null}).
   * @param label rtulo do parmetro (No aceita {@code null}).
   * @param description descrio do parmetro (No aceita {@code null}).
   * @param defaultValue valor-padro (Aceita {@code null}).
   * @param isOptional indica se o valor do parmetro  opcional.
   * @param isVisible indica se o parmetro deve ficar visvel.
   * @param commandLinePattern padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param localization define o argumento na linha de comando informando o
   *        protoloco usado pelo usurio.
   * @param fileType tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode modo do parmetro (No aceita {@code null}).
   * @param mustExist indica se o arquivo deve existir.
   * @param allowedProtocols conjunto de protocolos aceitos na URL.
   */
  public URLParameter(String name, String label, String description,
    URLValue defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String localization, String fileType,
    FileParameterMode mode, boolean mustExist,
    EnumSet<URLProtocol> allowedProtocols) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    if (mode == null) {
      throw new IllegalArgumentException("mode no pode ser nulo.");
    }
    this.localization = localization;
    this.fileType = fileType;
    this.mustExist = mustExist;
    this.allowedProtocols = allowedProtocols;
    this.mode = mode;
  }

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   * 
   * @return localizao.
   */
  public String getLocalization() {
    return localization;
  }

  /**
   * Obtm o tipo de arquivo que este parmetro aceita.
   * 
   * @return tipo de arquivo ou {@code null} se ele no se importar com um tipo
   *         especfico.
   */
  public String getFileType() {
    return fileType;
  }

  /**
   * Obtm o modo de funcionamento (apenas arquivos, apenas diretrios ou ambos)
   * deste parmetro.
   * 
   * @return modo de funcionamento.
   */
  public FileParameterMode getMode() {
    return mode;
  }

  /**
   * Atribui um tipo de arquivo especfico a este parmetro.
   * 
   * @param fileType tipo de arquivo (Aceita {@code null}).
   */
  public void setFileType(String fileType) {
    this.fileType = fileType;
    fireVisiblityWasChangedEvent();
  }

  /**
   * Indica se o arquivo selecionado precisa existir.
   * 
   * @return mustExist {@code true} se o arquivo precisa existir, {@code false}
   *         caso contrrio.
   */
  public boolean mustExist() {
    return mustExist;
  }

  /**
   * Obtm o conjunto de protocolos aceitos na URL.
   * 
   * @return conjunto de protocolos.
   */
  public EnumSet<URLProtocol> getAllowedProtocols() {
    return allowedProtocols;
  }

  /**
   * Obtm o protocolo padro do parmetro (caso no tenha sido especificado
   * nenhum).
   * 
   * @return protocolo padro.
   */
  public URLProtocol getDefaultProtocol() {
    if (allowedProtocols.size() == 0) {
      return URLValue.DEFAULT_PROTOCOL;
    }
    return allowedProtocols.iterator().next();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLParameterValidator createParameterValidator() {
    return new URLParameterValidator(isOptional());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    if (getValue() == null) {
      return new HashMap<String, Object>();
    }

    List<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    Map<String, String> exportableValue = new HashMap<String, String>();

    URLValue value = getValue();
    exportableValue.put(PATH, value.getPath());
    exportableValue.put(TYPE, value.getType());

    URLProtocol protocol = value.getProtocol();
    if (protocol != null) {
      exportableValue.put(PROTOCOL, protocol.getType());
    }
    exportableValue.put(HOST, value.getHost());
    exportableValues.add(Collections.unmodifiableMap(exportableValue));

    Map<String, Object> result = new HashMap<String, Object>();
    result.put(getName(), Collections.unmodifiableCollection(exportableValues));
    return Collections.unmodifiableMap(result);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());

    if (exportableValues == null || exportableValues.isEmpty()) {
      setValue(null);
      return;
    }

    Map<String, String> exportableValue = exportableValues.iterator().next();
    String path = exportableValue.get(PATH);
    String type = exportableValue.get(TYPE);
    String protocolName = exportableValue.get(PROTOCOL);
    String host = exportableValue.get(HOST);
    if (protocolName != null) {
      try {
        URLProtocol protocol = new URLProtocolConverter().valueOf(protocolName);
        setValue(new URLValue(path, type, protocol, host));
        return;
      }
      catch (ParseException e) {
        throw new IllegalArgumentException("Protocolo " + protocolName
          + " invlido");
      }
    }

    setValue(new URLValue(path, type));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(CommandLineContext context) {
    URLValue file = getValue();
    if (file == null) {
      return null;
    }

    String path = file.getPath();
    URLProtocol protocol = file.getProtocol();
    if (!path.startsWith(CommandLineBuilder.REFERENCE_VAR_CHAR)) {
      char fileSeparator = context.getFileSeparator();
      path = file.getPath(fileSeparator);
      switch (protocol) {
        case PROJECT:
          String filePath = file.getPath(fileSeparator);
          if (filePath.startsWith("/")) {
            filePath = filePath.substring(1);
          }
          path =
            CommandLineBuilder.makePathWithEnvironmentVariable(
              EnvironmentVariable.PROJECT_DIR, filePath, fileSeparator);
          break;
        default:
          String commandValue = protocol.getCommandValue();
          if (commandValue != null) {
            path = commandValue + path;
          }
          break;
      }
    }

    String f = "%s %s=%s";
    return String.format(f, path, getLocalization(), protocol.getType());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    URLValue file = getValue();
    return URLValue.getStringValue(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) {
    URLValue value = URLValue.getURLFromString(mode, parameterValue);
    setValue(value);
  }

}
