package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.parameters.validators.FileListParameterValidator;
import csbase.logic.algorithms.parameters.validators.FileParameterValidator;

/**
 * <p>
 * Parmetro do tipo lista de arquivos.
 * </p>
 */
public abstract class FileListParameter extends ListParameter<FileURLValue> {

  /**
   * Chave utilizada para informar o <b>caminho</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static String PATH = ".path";
  /**
   * Chave utilizada para informar o <b>tipo</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static String TYPE = ".type";
  /**
   * O tipo de arquivo que este parmetro filtra. Se ele se importa com o tipo
   * do arquivo, ele valer {@code null}.
   */
  protected String fileType;
  /**
   * Associao com os observador do parmetro do tipo arquivo.
   */
  protected transient List<FileListParameterListener> listeners;
  /**
   * O modo de funcionamento atual (apenas arquivos, apenas diretrios ou
   * ambos).
   */
  private FileParameterMode mode;

  /**
   * Indica se um painel de filtro deve ser exibido.
   */
  protected boolean usesFilter;

  /**
   * Indica se o arquivo deve existir.
   */
  private final boolean mustExist;

  /**
   * Cria um parmetro do tipo lista de arquivos.
   *
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileType O tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}). {@value FileParameterPipeAcceptance#TRUE}, no aceita
   *        pipe {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@value FileParameterPipeAcceptance#ALWAYS}.
   * @param mustSort Indica se os caminhos para os arquivos precisam ser
   *        ordenados.
   * @param usesFilter Indica se o painel de filtro deve ser exibido.
   * @param mustExist Indica se os arquivso devem existir.
   */
  public FileListParameter(String name, String label, String description,
    List<FileURLValue> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String fileType, FileParameterMode mode,
    boolean mustSort, boolean usesFilter, boolean mustExist) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      mustSort, false, commandLinePattern);
    listeners = new LinkedList<FileListParameterListener>();
    this.fileType = fileType;
    this.usesFilter = usesFilter;
    this.mustExist = mustExist;
    setMode(mode);
  }

  /**
   * Adiciona um observador de parmetros do tipo arquivo a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addListener(FileListParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> valuesByName = new HashMap<String, Object>();
    Collection<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    List<FileURLValue> files = getValue();
    if (files != null) {
      for (FileURLValue file : files) {
        Map<String, String> exportableValue = new HashMap<String, String>();
        exportableValue.put(PATH, file.getPath());
        exportableValue.put(TYPE, file.getType());
        exportableValues.add(Collections.unmodifiableMap(exportableValue));
      }
      valuesByName.put(getName(), Collections
        .unmodifiableCollection(exportableValues));
    }
    return Collections.unmodifiableMap(valuesByName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandItemValue(CommandLineContext context, FileURLValue file) {
    char fileSeparator = context.getFileSeparator();
    if (file.getPath().startsWith(String.valueOf(fileSeparator))) {
      return file.getPath();
    }
    else {
      String path = file.getPath(fileSeparator);
      return CommandLineBuilder.makePathWithEnvironmentVariable(
        EnvironmentVariable.PROJECT_DIR, path, fileSeparator);
    }
  }

  /**
   * Obtm o tipo de arquivo que este parmetro aceita.
   *
   * @return O tipo de arquivo ou {@code null} se ele no se importar com um
   *         tipo especfico.
   */
  public String getFileType() {
    return fileType;
  }

  /**
   * Obtm o modo de funcionamento (apenas arquivos, apenas diretrios ou ambos)
   * deste parmetro.
   *
   * @return O modo de funcionamento.
   */
  public FileParameterMode getMode() {
    return mode;
  }

  /**
   * @return Se um filtro de arquivo deve ser usado.
   */
  public boolean usesFilter() {
    return usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    List<FileURLValue> files = getValue();
    if (files == null || files.isEmpty()) {
      return null;
    }
    String separator = "";
    String valueAsText = "";
    for (FileURLValue file : files) {
      valueAsText += separator;
      valueAsText += file.getPath() + ":" + file.getType();
      separator = "|";
    }
    return valueAsText;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileURLValue getItemValueFromText(String elementAsText)
    throws ParseException {
    Pattern withTypePattern = Pattern.compile("^(.+):(.+)$");
    Matcher withTypeMatcher = withTypePattern.matcher(elementAsText);
    if (withTypeMatcher.matches()) {
      String path = withTypeMatcher.group(1);
      String type = withTypeMatcher.group(2);
      return new FileURLValue(path, type);
    }
    else {
      String path = elementAsText;
      return new FileURLValue(path);
    }
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());
    if (exportableValues != null) {
      List<FileURLValue> files = new ArrayList<FileURLValue>();
      for (Map<String, String> exportableValue : exportableValues) {
        String path = exportableValue.get(PATH);
        String type = exportableValue.get(TYPE);
        FileURLValue file = new FileURLValue(path, type);
        if (!files.contains(file)) {
          files.add(file);
        }
      }
      if (files.isEmpty()) {
        setValue(null);
      }
      else {
        setValue(files);
      }
    }
  }

  /**
   * Move para cima o elemento do ndice
   *
   * @param index ndice
   */
  public void moveUp(final int index) {
    final List<FileURLValue> files = getValue();
    if (files == null) {
      return;
    }
    final int size = files.size();
    if (size == 0 || index == 0) {
      return;
    }

    final FileURLValue element = files.remove(index);
    files.add(index - 1, element);
    fireValueWasChangedEvent();
  }

  /**
   * Move para cima o elemento do ndice
   *
   * @param index ndice
   */
  public void moveDown(final int index) {
    final List<FileURLValue> files = getValue();
    if (files == null) {
      return;
    }
    final int size = files.size();
    if (size == 0 || index == size - 1) {
      return;
    }

    final FileURLValue element = files.remove(index);
    files.add(index + 1, element);
    fireValueWasChangedEvent();
  }

  /**
   * Remove um observador deste parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   *
   * @return {@code true} se ele removeu o observador ou {@code false} se o
   *         observador no estiver neste parmetro.
   */
  public boolean removeListener(FileListParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * Atribui um tipo de arquivo especfico a este parmetro.
   *
   * @param fileType O tipo de arquivo (Aceita {@code null}).
   */
  public void setFileType(String fileType) {
    this.fileType = fileType;
    fireFileTypeWasChanged();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if ((parameterValue == null) || (parameterValue.length() == 0)) {
      setValue(null);
      return;
    }
    List<FileURLValue> files = new ArrayList<FileURLValue>();
    String[] components = parameterValue.split("\\|");
    for (String component : components) {
      FileURLValue file = getItemValueFromText(component);
      files.add(file);
    }
    setValue(files);
  }

  /**
   * Dispara o evento
   * {@link FileListParameterListener#fileTypeWasChanged(FileListParameter)} .
   */
  private void fireFileTypeWasChanged() {
    for (FileListParameterListener listener : listeners) {
      listener.fileTypeWasChanged(this);
    }
  }

  /**
   * Atribui o modo de funcionamento (apenas arquivo, apenas diretrio ou ambos)
   * deste parmetro.
   *
   * @param mode O modo de funcionamento (No aceita {@code null}).
   */
  protected void setMode(FileParameterMode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    this.mode = mode;
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in stream com o objeto serializado.
   *
   * @throws IOException em caso de erro de I/O.
   * @throws ClassNotFoundException se a classe do objeto serializado no for
   *         encontrada.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
  ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<FileListParameterListener>();
  }

  /**
   * @param usesFilter Define se um filtro deve ser usado ou no.
   */
  public void setUsesFilter(final boolean usesFilter) {
    this.usesFilter = usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileListParameterValidator createParameterValidator() {
    return new FileListParameterValidator(createItemValidator());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected FileParameterValidator createItemValidator() {
    return new FileParameterValidator(isOptional());
  }

  /**
   * Indica se os arquivos selecionados precisam existir.
   *
   * @return mustExist verdadeiro se os arquivos precisam existir, ou falso,
   *         caso contrrio.
   */
  public boolean mustExist() {
    return mustExist;
  }

}