package csbase.logic.algorithms.parsers;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros <T> com
 * valores do tipo enumerao como {@link EnumerationParameter} e
 * {@link EnumerationListParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractEnumerationParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica se o itens de enumerao devem ser ordenados.  opcional, o seu
   * valor-padro  {@link #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE} e o seu
   * tipo  booleano.
   * </p>
   */
  static final String ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE = "ordenar";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE = true;

  /**
   * <p>
   * O elemento {@value #ENUMERATION_ITEM_ELEMENT}: descreve um
   * {@link EnumerationItem item de enumerao} da {@link EnumerationParameter
   * enumerao}, {@link EnumerationListParameter} ou {@link EnumerationColumn
   * coluna de enumerao} que  descrita pelo elemento que  pai deste
   * elemento.
   * </p>
   * <p>
   *  elemento-filho de: {@link EnumerationListParameter},
   * {@link EnumerationParameter} ou {@link EnumerationColumn} de um
   * {@link TableParameter}.
   * </p>
   */
  static final String ENUMERATION_ITEM_ELEMENT = "item_de_enumeracao";

  /**
   * O atributo {@value #ENUMERATION_ITEM_DESCRIPTION_ATTRIBUTE} do elemento
   * {@link #ENUMERATION_ITEM_ELEMENT}. Indica a descrio do item,  opcional e
   *  do tipo string.
   */
  static final String ENUMERATION_ITEM_DESCRIPTION_ATTRIBUTE = "dica";

  /**
   * O atributo {@value #ENUMERATION_ITEM_ID_ATTRIBUTE} do elemento
   * {@link #ENUMERATION_ITEM_ELEMENT}. Indica o identificador do item, 
   * obrigatrio e  do tipo string.
   */
  static final String ENUMERATION_ITEM_ID_ATTRIBUTE = "id";

  /**
   * O atributo {@value #ENUMERATION_ITEM_LABEL_ATTRIBUTE} do elemento
   * {@link #ENUMERATION_ITEM_ELEMENT}. Indica o rtulo do item,  obrigatrio e
   *  do tipo string.
   */
  static final String ENUMERATION_ITEM_LABEL_ATTRIBUTE = "rotulo";

  /**
   * O atributo {@value #ENUMERATION_ITEM_VALUE_ATTRIBUTE} do elemento
   * {@link #ENUMERATION_ITEM_ELEMENT}. Indica o valor do item,  obrigatrio e
   *  do tipo string.
   */
  static final String ENUMERATION_ITEM_VALUE_ATTRIBUTE = "valor";

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group) throws ParseException {
    boolean mustSort =
      parser.extractAttributeValueAsBoolean(
        ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE,
        ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE);
    List<EnumerationItem> items = loadEnumerationItems(parser, name);
    String defaultItemId =
      parser.extractAttributeValue(PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE,
        null);
    return createEnumerationParameter(parser, name, label, description,
      isOptional, isVisible, commandLinePattern, mustSort, items, defaultItemId);
  }

  /**
   * Cria uma instncia do parmetro de tipo <T>, a partir dos atributos bsicos
   * de um parmetro do tipo enumerao. As subclasses devem implementar esse
   * mtodo, fazendo a extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param mustSort Indica se a enumerao deve ser ordenada.
   * @param items Os itens da enumerao, o valor-padro deve ser um dos itens.
   * @param defaultItemId O valor-padro do parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createEnumerationParameter(XmlParser parser,
    String name, String label, String description, boolean isOptional,
    boolean isVisible, String commandLinePattern, boolean mustSort,
    List<EnumerationItem> items, String defaultItemId) throws ParseException;

  /**
   * <p>
   * Carrega os {@link EnumerationItem}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome da enumerao (No aceita {@code null}).
   * 
   * @return Os items (Se no houver itens a lista estar vazia).
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  protected List<EnumerationItem> loadEnumerationItems(XmlParser parser,
    String parameterName) throws ParseException {
    List<EnumerationItem> items = new LinkedList<EnumerationItem>();
    if (parser.goToFirstChild(ENUMERATION_ITEM_ELEMENT)) {
      do {
        EnumerationItem enumerationItem = loadEnumerationItem(parser);
        if (items.contains(enumerationItem)) {
          throw new ParseException(
            "O item {0} est duplicado na enumerao {1}.",
            enumerationItem.getId(), parameterName);
        }
        for (EnumerationItem item : items) {
          if (enumerationItem.getLabel().equals(item.getLabel())) {
            throw new ParseException(
              "O item {0} est duplicado na enumerao {1}.",
              enumerationItem.getLabel(), parameterName);
          }
        }
        items.add(enumerationItem);
      } while (parser.goToNextSibling(ENUMERATION_ITEM_ELEMENT));
      parser.goToParent();
    }
    return items;
  }

  /**
   * <p>
   * Carrega um {@link EnumerationItem}.
   * </p>
   * 
   * <p>
   * O elemento corrente tem que ser {@link #ENUMERATION_ITEM_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador(No aceita {@code null}).
   * 
   * @return O item de enumerao.
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  protected EnumerationItem loadEnumerationItem(XmlParser parser)
    throws ParseException {
    String id = parser.extractAttributeValue(ENUMERATION_ITEM_ID_ATTRIBUTE);
    String label =
      parser.extractAttributeValue(ENUMERATION_ITEM_LABEL_ATTRIBUTE);
    String value =
      parser.extractAttributeValue(ENUMERATION_ITEM_VALUE_ATTRIBUTE);
    String description =
      parser
        .extractAttributeValue(ENUMERATION_ITEM_DESCRIPTION_ATTRIBUTE, null);
    parser.checkAttributes();
    parser.checkChildElements();
    EnumerationItem enumerationItem =
      new EnumerationItem(id, label, value, description);
    return enumerationItem;
  }

  /**
   * Obtm um item de enumerao na relao de itens fornecida.
   * 
   * @param enumerationName O nome da enumerao (No aceita {@code null}).
   * @param items Os itens disponveis (No aceita {@code null}).
   * @param itemId O identificador do item (No aceita {@code null}).
   * 
   * @return O item de enumerao.
   * 
   * @throws ParseException Se o item no existir.
   */
  protected EnumerationItem getEnumerationItem(String enumerationName,
    List<EnumerationItem> items, String itemId) throws ParseException {
    Iterator<EnumerationItem> itemIterator = items.iterator();
    while (itemIterator.hasNext()) {
      EnumerationItem item = itemIterator.next();
      if (item.getId().equals(itemId)) {
        return item;
      }
    }
    throw new ParseException("Erro ao tentar criar a enumerao {0}:"
      + " O item padro {1} no est definido.", new Object[] {
        enumerationName, itemId });
  }

}
