package csbase.logic.algorithms.parsers;

import java.util.ArrayList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.InputFileListParameter;
import csbase.logic.algorithms.parameters.FileURLValue;

/**
 * <p>
 * Analisador de {@link InputFileListParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo "arquivo de entrada"
 * mltiplo. O elemento corrente do {@link XmlParser analisador de XML} precisa
 * ser um elemento {@link InputFileListParameter}.
 * </p>
 *
 */
class InputFileListParameterParser extends
AbstractFileParameterParser<InputFileListParameter> {

  /**
   * {@inheritDoc}
   */
  @Override
  public InputFileListParameter createFileParameter(XmlParser parser,
    String name, String label, String description, boolean isOptional,
    boolean isVisible, String commandLinePattern, String type)
      throws ParseException {

    final boolean pipeDefined =
      parser.hasAttributeValue(FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE);
    if (pipeDefined) {
      final FileParameterPipeAcceptance usesPipe =
        parser.extractAttributeValueAsEnumeration(
          FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE,
          new FileParameterPipeAcceptanceConverter());
      if (usesPipe == FileParameterPipeAcceptance.TRUE
        || usesPipe == FileParameterPipeAcceptance.ALWAYS) {
        throw new ParseException(
          "Arquivo de entrada (seleo mltipla) no admite pipes");
      }
    }

    String modeName =
      parser.extractAttributeValue(FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE);
    FileParameterMode mode = null;
    if (modeName.equals(FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY)) {
      mode = FileParameterMode.DIRECTORY;
    }
    else if (modeName
      .equals(FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY_AND_REGULAR_FILE)) {
      mode = FileParameterMode.DIRECTORY_AND_REGULAR_FILE;
    }
    else if (modeName
      .equals(FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE)) {
      mode = FileParameterMode.REGULAR_FILE;
    }
    else {
      throw new ParseException("A categoria informada {0} no  vlida.\n"
        + "Categorias vlidas:\n{1};\n{2};{3}.", new Object[] { modeName,
          FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE,
          FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY,
          FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY_AND_REGULAR_FILE });
    }

    boolean useRootDirectoryAsDefault =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE);
    List<FileURLValue> defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de entrada  um arquivo.",
          new Object[] { FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE });
      }
      defaultValue = new ArrayList<FileURLValue>();
      defaultValue.add(new FileURLValue("."));
    }

    boolean mustSort =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE);

    boolean useFilter =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE);

    InputFileListParameter parameter =
      new InputFileListParameter(name, label, description, defaultValue,
        isOptional, isVisible, commandLinePattern, type, mode, mustSort,
        useFilter);
    return parameter;
  }
}
