package csbase.logic.algorithms.parsers;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.FileParameter;
import csbase.logic.algorithms.parameters.IntegerListParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TextParameter;
import csbase.logic.algorithms.parameters.conditions.AndOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.conditions.GenericCondition;
import csbase.logic.algorithms.parameters.conditions.NotOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.OrOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.SimpleCondition;
import csbase.logic.algorithms.parameters.triggers.AllowEditParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeDefaultValueTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeFileTypeTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeLabelTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeMaximumForDoublesTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeMaximumForIntegersTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeMinimumForDoublesTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeMinimumForIntegersTrigger;
import csbase.logic.algorithms.parameters.triggers.DisableParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.EnableParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.HideGroupTrigger;
import csbase.logic.algorithms.parameters.triggers.HideParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.ProhibitEditParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.SetVisibleEnumerationItemTrigger;
import csbase.logic.algorithms.parameters.triggers.ShowGroupTrigger;
import csbase.logic.algorithms.parameters.triggers.ShowParameterTrigger;
import csbase.logic.algorithms.parameters.triggers.Trigger;

/**
 * <p>
 * Analisador de {@link Trigger}.
 * </p>
 * 
 * <p>
 * Este parser l os atributos que representam os gatilhos como
 * {@link ShowParameterTrigger} ou {@link AllowEditParameterTrigger}. O elemento
 * corrente do {@link XmlParser analisador de XML} precisa ser um elemento que
 * represente o tipo {@link Trigger}, como
 * {@link #SHOW_PARAMETER_TRIGGER_ELEMENT} ou
 * {@link #ALLOW_EDIT_TRIGGER_ELEMENT}.
 * </p>
 */
public class TriggerParser {

  /**
   * <p>
   * O elemento {@value #ALLOW_EDIT_TRIGGER_ELEMENT}: descreve as propriedades
   * de um {@link AllowEditParameterTrigger gatilho permitir edio}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String ALLOW_EDIT_TRIGGER_ELEMENT = "permitir_edicao";

  /**
   * <p>
   * O elemento {@value #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link ChangeDefaultValueTrigger gatilho trocar
   * valor-padro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT =
    "trocar_valor_padrao";

  /**
   * O atributo
   * {@value #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE}
   * do elemento {@link #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}: indica o novo
   * valor-padro para o parmetro,  opcional e o seu tipo depende do tipo do
   * parmetro (Ex.: {@link BooleanParameter} - booleano).
   */
  private static final String CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE =
    "novo_valor_padrao";

  /**
   * <p>
   * O elemento {@value #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link ChangeFileTypeTrigger gatilho trocar tipo de
   * arquivo}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String CHANGE_FILE_TYPE_TRIGGER_ELEMENT =
    "trocar_tipo_de_arquivo";

  /**
   * O atributo
   * {@value #CHANGE_FILE_TYPE_TRIGGER_ELEMENT_NEW_FILE_TYPE_ATTRIBUTE} do
   * elemento {@link #CHANGE_FILE_TYPE_TRIGGER_ELEMENT}: indica o novo tipo do
   * arquivo,  opcional e  do tipo string.
   */
  private static final String CHANGE_FILE_TYPE_TRIGGER_ELEMENT_NEW_FILE_TYPE_ATTRIBUTE =
    "novo_tipo";

  /**
   * <p>
   * O elemento {@value #CHANGE_LABEL_TRIGGER_ELEMENT}: descreve as propriedades
   * de um {@link ChangeLabelTrigger gatilho trocar rtulo}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String CHANGE_LABEL_TRIGGER_ELEMENT = "trocar_rotulo";

  /**
   * O atributo {@value #CHANGE_LABEL_TRIGGER_ELEMENT_NEW_LABEL_ATTRIBUTE} do
   * elemento {@link #CHANGE_FILE_TYPE_TRIGGER_ELEMENT}: indica o novo rtulo
   * para o parmetro,  obrigatrio e  do tipo string.
   */
  private static final String CHANGE_LABEL_TRIGGER_ELEMENT_NEW_LABEL_ATTRIBUTE =
    "novo_rotulo";

  /**
   * <p>
   * O elemento {@value #CHANGE_MAXIMUM_TRIGGER_ELEMENT}: descreve as
   * propriedades de um gatilho trocar mximo (
   * {@link ChangeMaximumForDoublesTrigger} ou
   * {@link ChangeMaximumForIntegersTrigger}).
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String CHANGE_MAXIMUM_TRIGGER_ELEMENT = "trocar_maximo";

  /**
   * O atributo
   * {@value #CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}: indica se o valor mximo
   *  considerado vlido,  booleano,  opcional e o valor-padro 
   * {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_DEFAULT_VALUE}.
   */
  private static final String CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE =
    "incluir_novo_maximo";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}.
   * </p>
   * <p>
   * O seu valor 
   * {@value #CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_DEFAULT_VALUE =
    true;

  /**
   * O atributo
   * {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}: indica o valor mximo, 
   * opcional,  do tipo inteiro ou real.
   */
  private static final String CHANGE_MAXIMUM_TRIGGER_ELEMENT_NEW_MAXIMUM_ATTRIBUTE =
    "novo_maximo";

  /**
   * <p>
   * O elemento {@value #CHANGE_MINIMUM_TRIGGER_ELEMENT}: descreve as
   * propriedades de um gatilho trocar mnimo (
   * {@link ChangeMinimumForDoublesTrigger} ou
   * {@link ChangeMinimumForIntegersTrigger}).
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String CHANGE_MINIMUM_TRIGGER_ELEMENT = "trocar_minimo";

  /**
   * O atributo
   * {@value #CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}: indica se o valor mnimo
   *  considerado vlido,  booleano,  opcional e o valor-padro 
   * {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_DEFAULT_VALUE}.
   */
  private static final String CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE =
    "incluir_novo_minimo";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}.
   * </p>
   * <p>
   * O seu valor 
   * {@value #CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_DEFAULT_VALUE =
    true;

  /**
   * O atributo
   * {@value #CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE} do
   * elemento {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}: indica o valor mnimo, 
   * opcional,  do tipo inteiro ou real.
   */
  private static final String CHANGE_MINIMUM_TRIGGER_ELEMENT_NEW_MINIMUM_ATTRIBUTE =
    "novo_minimo";

  /**
   * <p>
   * O elemento {@value #DISABLE_PARAMETER_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link DisableParameterTrigger gatilho desabilitar
   * parmetro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String DISABLE_PARAMETER_TRIGGER_ELEMENT = "desabilitar";

  /**
   * <p>
   * O elemento {@value #ENABLE_PARAMETER_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link EnableParameterTrigger gatilho habilitar
   * parmetro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String ENABLE_PARAMETER_TRIGGER_ELEMENT = "habilitar";

  /**
   * <p>
   * O elemento {@value #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link SetVisibleEnumerationItemTrigger gatilho
   * exibir/ocultar parmetro} configurado para ocultar o
   * {@link EnumerationItem item de enumerao}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT =
    "ocultar_item";

  /**
   * <p>
   * O elemento {@value #HIDE_PARAMETER_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link HideParameterTrigger gatilho ocultar parmetro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String HIDE_PARAMETER_TRIGGER_ELEMENT = "ocultar";

  /**
   * <p>
   * O elemento {@value #HIDE_GROUP_TRIGGER_ELEMENT}: descreve as propriedades
   * de um {@link HideGroupTrigger gatilho ocultar grupo}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String HIDE_GROUP_TRIGGER_ELEMENT = "ocultar_grupo";

  /**
   * <p>
   * O elemento {@value #PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link ProhibitEditParameterTrigger gatilho proibir
   * edio de parmetro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT =
    "proibir_edicao";

  /**
   * <p>
   * O atributo
   * {@value #SET_VISIBLE_ENUMERATION_ITEM_TRIGGER_ELEMENT_ITEM_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link #HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * </ul>
   * <p>
   * Indica se  para exibir ({@code true}) ou ocultar ({@code true}). 
   * obrigatria e o seu tipo  booleano.
   * </p>
   */
  private static final String SET_VISIBLE_ENUMERATION_ITEM_TRIGGER_ELEMENT_ITEM_ATTRIBUTE =
    "item";

  /**
   * <p>
   * O elemento {@value #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link SetVisibleEnumerationItemTrigger gatilho
   * exibir/ocultar parmetro} configurado para exibir o {@link EnumerationItem
   * item de enumerao}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT =
    "exibir_item";

  /**
   * <p>
   * O elemento {@value #SHOW_PARAMETER_TRIGGER_ELEMENT}: descreve as
   * propriedades de um {@link ShowParameterTrigger gatilho exibir parmetro}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String SHOW_PARAMETER_TRIGGER_ELEMENT = "exibir";

  /**
   * <p>
   * O elemento {@value #SHOW_GROUP_TRIGGER_ELEMENT}: descreve as propriedades
   * de um {@link ShowGroupTrigger gatilho exibir grupo}.
   * </p>
   * <p>
   *  filho do elemento {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  private static final String SHOW_GROUP_TRIGGER_ELEMENT = "exibir_grupo";

  /**
   * <p>
   * O atributo {@value #TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link #ALLOW_EDIT_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_FILE_TYPE_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_LABEL_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}</li>
   * <li>{@link #DISABLE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #ENABLE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * <li>{@link #HIDE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_PARAMETER_TRIGGER_ELEMENT}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o nome do parmetro que ser manipulado pelo gatilho.  opcional e 
   * do tipo string. Ele  opcional, pois pode-se utilizar o elemento
   * {@link #TRIGGER_PARAMETER_ELEMENT} com o mesmo sentido, portanto pelo menos
   * um dos 2 tem que existir.*
   * </p>
   */
  private static final String TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE = "parametro";

  /**
   * <p>
   * O elemento {@value #TRIGGER_PARAMETER_ELEMENT}: indica o parmetro que ser
   * manipulado pelos gatilhos.
   * </p>
   * <p>
   *  elemento-filho de:
   * <ul>
   * <li>{@link #ALLOW_EDIT_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_FILE_TYPE_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_LABEL_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}</li>
   * <li>{@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}</li>
   * <li>{@link #DISABLE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #ENABLE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * <li>{@link #HIDE_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_PARAMETER_TRIGGER_ELEMENT}</li>
   * </ul>
   * </p>
   */
  private static final String TRIGGER_PARAMETER_ELEMENT = "parametro";

  /**
   * <p>
   * O atributo {@value #TRIGGER_ELEMENT_GROUP_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link #HIDE_GROUP_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_GROUP_TRIGGER_ELEMENT}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o nome do parmetro que ser manipulado pelo gatilho.  opcional e 
   * do tipo string. Ele  opcional, pois pode-se utilizar o elemento
   * {@link #TRIGGER_GROUP_ELEMENT} com o mesmo sentido, portanto pelo menos um
   * dos 2 tem que existir.*
   * </p>
   */
  private static final String TRIGGER_ELEMENT_GROUP_ATTRIBUTE = "grupo";

  /**
   * <p>
   * O elemento {@value #TRIGGER_GROUP_ELEMENT}: indica o grupo que ser
   * manipulado pelos gatilhos.
   * </p>
   * <p>
   *  elemento-filho de:
   * <ul>
   * <li>{@link #HIDE_GROUP_TRIGGER_ELEMENT}</li>
   * <li>{@link #SHOW_GROUP_TRIGGER_ELEMENT}</li>
   * </ul>
   * </p>
   */
  private static final String TRIGGER_GROUP_ELEMENT = "grupo";

  /**
   * O atributo {@value #TRIGGER_GROUP_ELEMENT_ID_ATTRIBUTE} do elemento
   * {@link #TRIGGER_GROUP_ELEMENT}: indica o nome do grupo manipulado pelos
   * gatilhos.  obrigatrio e  do tipo string.
   */
  private static final String TRIGGER_GROUP_ELEMENT_ID_ATTRIBUTE = "id";

  /**
   * O atributo {@value #TRIGGER_PARAMETER_ELEMENT_NAME_ATTRIBUTE} do elemento
   * {@link #TRIGGER_PARAMETER_ELEMENT}: indica o nome do parmetro manipulado
   * pelos gatilhos.  obrigatrio e  do tipo string.
   */
  private static final String TRIGGER_PARAMETER_ELEMENT_NAME_ATTRIBUTE = "nome";

  /**
   * <p>
   * Carregas os gatilhos do configurador de algoritmos.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @throws ParseException Se houver um erro no XML.
   */
  public void loadTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    if (parser.goToFirstChild()) {
      do {
        Trigger<?> trigger = null;
        List<? extends Trigger<?>> triggers = null;
        String elementName = parser.getElementName();
        if (elementName.equals(ALLOW_EDIT_TRIGGER_ELEMENT)) {
          trigger = loadAllowEditTrigger(parser, configurator);
        }
        else if (elementName.equals(CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT)) {
          trigger = loadChangeDefaultValueTrigger(parser, configurator);
        }
        else if (elementName.equals(CHANGE_FILE_TYPE_TRIGGER_ELEMENT)) {
          trigger = loadChangeFileTypeTrigger(parser, configurator);
        }
        else if (elementName.equals(CHANGE_LABEL_TRIGGER_ELEMENT)) {
          trigger = loadChangeLabelTrigger(parser, configurator);
        }
        else if (elementName.equals(CHANGE_MAXIMUM_TRIGGER_ELEMENT)) {
          trigger = loadChangeMaximumTrigger(parser, configurator);
        }
        else if (elementName.equals(CHANGE_MINIMUM_TRIGGER_ELEMENT)) {
          trigger = loadChangeMinimumTrigger(parser, configurator);
        }
        else if (elementName.equals(DISABLE_PARAMETER_TRIGGER_ELEMENT)) {
          triggers = loadDisableParameterTriggers(parser, configurator);
        }
        else if (elementName.equals(ENABLE_PARAMETER_TRIGGER_ELEMENT)) {
          triggers = loadEnableParameterTrigger(parser, configurator);
        }
        else if (elementName.equals(HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT)) {
          trigger =
            loadSetVisibleEnumerationItemTrigger(parser, configurator, false);
        }
        else if (elementName.equals(HIDE_PARAMETER_TRIGGER_ELEMENT)) {
          triggers = loadHideParameterTriggers(parser, configurator);
        }
        else if (elementName.equals(HIDE_GROUP_TRIGGER_ELEMENT)) {
          triggers = loadHideGroupTriggers(parser, configurator);
        }
        else if (elementName.equals(PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT)) {
          trigger = loadProhibitEditTrigger(parser, configurator);
        }
        else if (elementName.equals(SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT)) {
          trigger =
            loadSetVisibleEnumerationItemTrigger(parser, configurator, true);
        }
        else if (elementName.equals(SHOW_PARAMETER_TRIGGER_ELEMENT)) {
          triggers = loadShowParameterTriggers(parser, configurator);
        }
        else if (elementName.equals(SHOW_GROUP_TRIGGER_ELEMENT)) {
          triggers = loadShowGroupTriggers(parser, configurator);
        }
        if (triggers != null) {
          for (Trigger<?> currentTrigger : triggers) {
            if (!configurator.addTrigger(currentTrigger)) {
              throw new ParseException(
                "Foi encontrado um gatilho repetido no algoritmo {0}.\n"
                  + "Gatilho repetido: {1}.\n", new Object[] { configurator,
                    currentTrigger });
            }
          }
        }
        if (trigger != null) {
          if (!configurator.addTrigger(trigger)) {
            throw new ParseException(
              "Foi encontrado um gatilho repetido no algoritmo {0}.\n"
                + "Gatilho repetido: {1}.\n", new Object[] { configurator,
                  trigger });
          }
        }
      } while (parser.goToNextSibling());
      parser.goToParent();
    }
  }

  /**
   * <p>
   * Carrega um {@link ProhibitEditParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #PROHIBIT_EDIT_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private ProhibitEditParameterTrigger loadProhibitEditTrigger(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    parser.checkAttributes();
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.", new Object[] { parameterName,
            ENABLE_PARAMETER_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    if (!(parameter instanceof TextParameter)) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no  do tipo texto.", new Object[] { parameterName,
            ENABLE_PARAMETER_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    TextParameter textParameter = (TextParameter) parameter;
    Condition condition = loadTriggerCondition(parser, configurator);
    return new ProhibitEditParameterTrigger(textParameter, condition);
  }

  /**
   * <p>
   * Carrega um {@link SetVisibleEnumerationItemTrigger gatilho exibir/ocultar
   * parmetro}.
   * </p>
   * 
   * <p>
   * O elemento corrente precisa ser
   * {@link #HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT} ou
   * {@link #SHOW_ENUMERATION_ITEM_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @param isVisible Indica se  para exibir ({@code true}) ou ocultar (
   *        {@code false}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  private SetVisibleEnumerationItemTrigger loadSetVisibleEnumerationItemTrigger(
    XmlParser parser, SimpleAlgorithmConfigurator configurator,
    boolean isVisible) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    Parameter<?> parameter = configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.", new Object[] { parameterName,
            HIDE_ENUMERATION_ITEM_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    if (!(parameter instanceof EnumerationParameter)) {
      throw new ParseException(
        "O parmetro {0} no  uma enumerao com seleo simples.",
        new Object[] { parameterName,
            AbstractEnumerationParameterParser.ENUMERATION_ITEM_ELEMENT });
    }
    EnumerationParameter enumerationParameter =
      (EnumerationParameter) parameter;
    String itemId =
      parser
        .extractAttributeValue(SET_VISIBLE_ENUMERATION_ITEM_TRIGGER_ELEMENT_ITEM_ATTRIBUTE);
    EnumerationItem item = enumerationParameter.getItem(itemId);
    if (item == null) {
      throw new ParseException(
        "No existe um item chamado {1} na enumerao {0}.", new Object[] {
            parameter, itemId });
    }
    parser.checkAttributes();
    Condition condition = loadTriggerCondition(parser, configurator);
    return new SetVisibleEnumerationItemTrigger(enumerationParameter,
      condition, itemId, isVisible);
  }

  /**
   * <p>
   * Carrega um {@link ShowParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #SHOW_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<ShowParameterTrigger> loadShowParameterTriggers(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    Condition condition = loadTriggerCondition(parser, configurator);
    Set<SimpleParameter<?>> parameters =
      loadTriggerParameters(parser, configurator);
    parser.checkAttributes();
    List<ShowParameterTrigger> triggers =
      new LinkedList<ShowParameterTrigger>();
    for (SimpleParameter<?> parameter : parameters) {
      triggers.add(new ShowParameterTrigger(parameter, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

  /**
   * <p>
   * Carrega um {@link ShowParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #SHOW_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<ShowGroupTrigger> loadShowGroupTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    Condition condition = loadTriggerCondition(parser, configurator);
    Set<ParameterGroup> groups = loadTriggerGroups(parser, configurator);
    parser.checkAttributes();
    List<ShowGroupTrigger> triggers = new LinkedList<ShowGroupTrigger>();
    for (ParameterGroup group : groups) {
      triggers.add(new ShowGroupTrigger(group, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

  /**
   * <p>
   * Carrega uma condio, que pode ser:
   * <ul>
   * <li>{@link SimpleCondition}</li>
   * <li>{@link GenericCondition}</li>
   * <li>{@link AndOperatorCondition}</li>
   * <li>{@link OrOperatorCondition}</li>
   * <li>{@link NotOperatorCondition}</li>
   * </ul>
   * </p>
   * 
   * <p>
   * O elemento corrente do aponta para o gatilho.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Condition loadTriggerCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    Condition condition = null;
    if (parser.goToFirstChild(ConditionParser.CONDITION_ELEMENT)
      || parser.goToFirstChild(ConditionParser.OR_ELEMENT)
      || parser.goToFirstChild(ConditionParser.AND_ELEMENT)
      || parser.goToFirstChild(ConditionParser.NOT_ELEMENT)) {
      ConditionParser conditionParser = new ConditionParser();
      condition = conditionParser.loadCondition(parser, configurator);
      parser.goToParent();
      return condition;
    }
    throw new ParseException("A condio de uma restrio no est definida.");
  }

  /**
   * Carrega os parmetros referenciados por um gatilho.
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return Os parmetros.
   * 
   * @throws ParseException Se houver algum erro no XML, inclusive se no houver
   *         indicao de que parmetros utilizar.
   */
  private Set<SimpleParameter<?>> loadTriggerParameters(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    Set<SimpleParameter<?>> parameters = new HashSet<SimpleParameter<?>>();
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE, null);
    if (parameterName != null) {
      SimpleParameter<?> parameter =
        configurator.getSimpleParameter(parameterName);
      if (parameter == null) {
        throw new ParseException(
          "O parmetro {0} que est referenciado no elemento {1} "
            + "atributo {2} no est definido.", parameterName, elementName,
          TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
      }
      parameters.add(parameter);
    }
    if (parser.goToFirstChild(TRIGGER_PARAMETER_ELEMENT)) {
      do {
        parameterName =
          parser.getAttributeValue(TRIGGER_PARAMETER_ELEMENT_NAME_ATTRIBUTE);
        SimpleParameter<?> parameter =
          configurator.getSimpleParameter(parameterName);
        if (parameter == null) {
          throw new ParseException(
            "O parmetro {0} que est referenciado no elemento {1} "
              + "atributo {2} no est definido.", parameterName,
            TRIGGER_PARAMETER_ELEMENT, TRIGGER_PARAMETER_ELEMENT_NAME_ATTRIBUTE);
        }
        parameters.add(parameter);
      } while (parser.goToNextSibling(TRIGGER_PARAMETER_ELEMENT));
      parser.goToParent();
    }
    if (parameters.isEmpty()) {
      throw new ParseException(String.format(
        "Os parmetros associados ao gatilho representado pelo elemento %s "
          + "no foram informados.", elementName));
    }
    return parameters;
  }

  /**
   * Carrega os grupos referenciados por um gatilho.
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return Os grupos.
   * 
   * @throws ParseException Se houver algum erro no XML, inclusive se no houver
   *         indicao de que parmetros utilizar.
   */
  private Set<ParameterGroup> loadTriggerGroups(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    Set<ParameterGroup> parameters = new HashSet<ParameterGroup>();
    String groupName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_GROUP_ATTRIBUTE, null);
    if (groupName != null) {
      ParameterGroup triggerGroup =
        findGroupByName(configurator.getGroups(), groupName);
      if (triggerGroup == null) {
        throw new ParseException(
          "O grupo {0} que est referenciado no elemento {1} "
            + "atributo {2} no est definido.", groupName, elementName,
          TRIGGER_ELEMENT_GROUP_ATTRIBUTE);
      }
      parameters.add(triggerGroup);
    }
    if (parser.goToFirstChild(TRIGGER_GROUP_ELEMENT)) {
      do {
        groupName =
          parser.getAttributeValue(TRIGGER_GROUP_ELEMENT_ID_ATTRIBUTE);
        ParameterGroup triggerGroup =
          findGroupByName(configurator.getGroups(), groupName);
        if (triggerGroup == null) {
          throw new ParseException(
            "O grupo {0} que est referenciado no elemento {1} "
              + "atributo {2} no est definido.", groupName,
            TRIGGER_GROUP_ELEMENT, TRIGGER_GROUP_ELEMENT_ID_ATTRIBUTE);
        }
        parameters.add(triggerGroup);
      } while (parser.goToNextSibling(TRIGGER_GROUP_ELEMENT));
      parser.goToParent();
    }
    if (parameters.isEmpty()) {
      throw new ParseException(String.format(
        "Os grupos associados ao gatilho representado pelo elemento %s "
          + "no foram informados.", elementName));
    }
    return parameters;
  }

  /**
   * Busca recursivamente um grupo pelo nome a partir de um conjunto de grupos.
   * 
   * @param groups o conjunto de grupos.
   * @param groupName o nome do grupo a ser buscado.
   * @return parameter o grupo com o nome especificado ou nulo, caso no seja
   *         encontrado nenhum grupo com o nome.
   * @throws ParseException em caso de se encontrar mais de um grupo com o mesmo
   *         nome.
   */
  private ParameterGroup findGroupByName(Collection<ParameterGroup> groups,
    String groupName) throws ParseException {
    ParameterGroup groupFound = null;
    for (ParameterGroup group : groups) {
      if (group.getName().equals(groupName)) {
        if (groupFound == null) {
          groupFound = group;
        }
        else {
          throw new ParseException(
            "Foi encontrado mais de um grupo com o identificador \"{0}\".",
            groupName);
        }
      }
      ParameterGroup childGroup = findGroupByName(group.getGroups(), groupName);
      if (childGroup != null) {
        if (groupFound == null) {
          groupFound = childGroup;
        }
        else {
          throw new ParseException(
            "Foi encontrado mais de um grupo com o identificador \"{0}\".",
            groupName);
        }
      }
    }
    return groupFound;
  }

  /**
   * <p>
   * Carrega um {@link AllowEditParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #ALLOW_EDIT_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Trigger<?> loadAllowEditTrigger(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    parser.checkAttributes();
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.", new Object[] { parameterName,
            ENABLE_PARAMETER_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    if (!(parameter instanceof TextParameter)) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no  do tipo texto.", new Object[] { parameterName,
            ENABLE_PARAMETER_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    TextParameter textParameter = (TextParameter) parameter;
    Condition condition = loadTriggerCondition(parser, configurator);
    return new AllowEditParameterTrigger(textParameter, condition);
  }

  /**
   * <p>
   * Carrega um {@link ChangeDefaultValueTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private ChangeDefaultValueTrigger<?> loadChangeDefaultValueTrigger(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.",
        new Object[] { parameterName, CHANGE_MAXIMUM_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    Condition condition = loadTriggerCondition(parser, configurator);
    if (parameter instanceof TextParameter) {
      TextParameter textParam = (TextParameter) parameter;
      String defaultValue =
        parser.extractAttributeValue(
          CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE,
          null);
      parser.checkAttributes();
      return createChangeDefaultValueTrigger(textParam, condition, defaultValue);
    }
    else if (parameter instanceof IntegerParameter) {
      IntegerParameter intParam = (IntegerParameter) parameter;
      Integer defaultValue =
        parser.extractAttributeValueAsInteger(
          CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE,
          null);
      parser.checkAttributes();
      return createChangeDefaultValueTrigger(intParam, condition, defaultValue);
    }
    else if (parameter instanceof DoubleParameter) {
      DoubleParameter doubleParam = (DoubleParameter) parameter;
      Double defaultValue =
        parser.extractAttributeValueAsDouble(
          CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE,
          null);
      parser.checkAttributes();
      return createChangeDefaultValueTrigger(doubleParam, condition,
        defaultValue);
    }
    else if (parameter instanceof BooleanParameter) {
      BooleanParameter booleanParam = (BooleanParameter) parameter;
      Boolean defaultValue =
        new Boolean(
          parser
            .extractAttributeValueAsBoolean(CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE));
      parser.checkAttributes();
      return createChangeDefaultValueTrigger(booleanParam, condition,
        defaultValue);

    }
    else if (parameter instanceof EnumerationParameter) {
      EnumerationParameter enumParameter = (EnumerationParameter) parameter;
      String itemId =
        parser
          .extractAttributeValue(CHANGE_DEFAULT_VALUE_TRIGGER_ELEMENT_NEW_DEFAULT_VALUE_ATTRIBUTE);
      EnumerationParameter enumerationParameter =
        (EnumerationParameter) parameter;
      EnumerationItem defaultValue = enumerationParameter.getItem(itemId);
      if (defaultValue == null) {
        throw new ParseException(
          "O item de enumerao {0} no est presente na enumerao {1}.",
          new Object[] { itemId, parameter });
      }
      parser.checkAttributes();
      return createChangeDefaultValueTrigger(enumParameter, condition,
        defaultValue);
    }
    else {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2};\n{3};\n{4};\n{5}.", new Object[] {
          parameterName, "texto", "inteiro", "real", "booleano", "enumeracao" });
    }
  }

  /**
   * Mtodo genrico para criao de um {@link ChangeDefaultValueTrigger} do
   * tipo especfico parametrizado V.
   * 
   * @param parameter o parmetro.
   * @param condition a condio para ativao do gatinho.
   * @param defaultValue o novo valor padro do parmetro.
   * @return o gatinho.
   */
  private <V> ChangeDefaultValueTrigger<V> createChangeDefaultValueTrigger(
    SimpleParameter<V> parameter, Condition condition, V defaultValue) {
    return new ChangeDefaultValueTrigger<V>(parameter, condition, defaultValue);
  }

  /**
   * <p>
   * Carrega um {@link ChangeFileTypeTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CHANGE_FILE_TYPE_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private ChangeFileTypeTrigger loadChangeFileTypeTrigger(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    Parameter<?> parameter = configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.",
        new Object[] { parameterName, CHANGE_MAXIMUM_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    if (!(parameter instanceof FileParameter)) {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2}.", new Object[] { parameterName,
          InputFileParameterParser.INPUT_FILE_PARAMETER_ELEMENT,
          OutputFileParameterParser.OUTPUT_FILE_PARAMETER_ELEMENT });
    }
    Condition condition = loadTriggerCondition(parser, configurator);
    String newFileType =
      parser.extractAttributeValue(
        CHANGE_FILE_TYPE_TRIGGER_ELEMENT_NEW_FILE_TYPE_ATTRIBUTE, null);
    parser.checkAttributes();
    return new ChangeFileTypeTrigger((FileParameter) parameter, condition,
      newFileType);
  }

  /**
   * <p>
   * Carrega um {@link ChangeLabelTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CHANGE_LABEL_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Trigger<?> loadChangeLabelTrigger(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.", new Object[] { parameterName,
            CHANGE_LABEL_TRIGGER_ELEMENT, TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    String newLabel =
      parser
        .extractAttributeValue(CHANGE_LABEL_TRIGGER_ELEMENT_NEW_LABEL_ATTRIBUTE);
    Condition condition = loadTriggerCondition(parser, configurator);
    parser.checkAttributes();
    return new ChangeLabelTrigger(parameter, condition, newLabel);
  }

  /**
   * <p>
   * Carrega um gatilho trocar mximo ({@link ChangeMaximumForDoublesTrigger} ou
   * {@link ChangeMaximumForIntegersTrigger}).
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CHANGE_MAXIMUM_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Trigger<?> loadChangeMaximumTrigger(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    Parameter<?> parameter = configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.",
        new Object[] { parameterName, CHANGE_MAXIMUM_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    Condition condition = loadTriggerCondition(parser, configurator);
    Trigger<?> trigger;
    if (parameter instanceof DoubleParameter) {
      Double maximum =
        parser.extractAttributeValueAsDouble(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_NEW_MAXIMUM_ATTRIBUTE, null);
      boolean isMaximumIncluded =
        parser.extractAttributeValueAsBoolean(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE,
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_DEFAULT_VALUE);
      DoubleParameter doubleParameter = (DoubleParameter) parameter;
      trigger =
        new ChangeMaximumForDoublesTrigger(doubleParameter, condition, maximum,
          isMaximumIncluded);
    }
    else if (parameter instanceof DoubleListParameter) {
      Double maximum =
        parser.extractAttributeValueAsDouble(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_NEW_MAXIMUM_ATTRIBUTE, null);
      boolean isMaximumIncluded =
        parser.extractAttributeValueAsBoolean(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MAXIMUM_ATTRIBUTE, true);
      DoubleListParameter doubleListParameter = (DoubleListParameter) parameter;
      trigger =
        new ChangeMaximumForDoublesTrigger(doubleListParameter, condition,
          maximum, isMaximumIncluded);
    }
    else if (parameter instanceof IntegerParameter) {
      Integer maximum =
        parser.extractAttributeValueAsInteger(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_NEW_MAXIMUM_ATTRIBUTE, null);
      IntegerParameter integerParameter = (IntegerParameter) parameter;
      trigger =
        new ChangeMaximumForIntegersTrigger(integerParameter, condition,
          maximum);
    }
    else if (parameter instanceof IntegerListParameter) {
      Integer maximum =
        parser.extractAttributeValueAsInteger(
          CHANGE_MAXIMUM_TRIGGER_ELEMENT_NEW_MAXIMUM_ATTRIBUTE, null);
      IntegerListParameter integerListParameter =
        (IntegerListParameter) parameter;
      trigger =
        new ChangeMaximumForIntegersTrigger(integerListParameter, condition,
          maximum);
    }
    else {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2};\n{3};\n{4}.", parameterName,
        DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
        DoubleListParameterFactory.DOUBLE_LIST_PARAMETER_ELEMENT,
        IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
        IntegerListParameterFactory.INTEGER_LIST_PARAMETER_ELEMENT);
    }
    parser.checkAttributes();
    return trigger;
  }

  /**
   * <p>
   * Carrega um gatilho trocar mnimo ({@link ChangeMinimumForDoublesTrigger} ou
   * {@link ChangeMinimumForIntegersTrigger}).
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CHANGE_MINIMUM_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Trigger<?> loadChangeMinimumTrigger(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String parameterName =
      parser.extractAttributeValue(TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE);
    Parameter<?> parameter = configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(
        "O parmetro {0} que est referenciado no elemento {1} "
          + "atributo {2} no est definido.",
        new Object[] { parameterName, CHANGE_MINIMUM_TRIGGER_ELEMENT,
            TRIGGER_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    Condition condition = loadTriggerCondition(parser, configurator);
    Trigger<?> trigger;
    if (parameter instanceof DoubleParameter) {
      Double minimum =
        parser.extractAttributeValueAsDouble(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_NEW_MINIMUM_ATTRIBUTE, null);
      boolean isMinimumIncluded =
        parser.extractAttributeValueAsBoolean(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE,
          CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_DEFAULT_VALUE);
      DoubleParameter doubleParameter = (DoubleParameter) parameter;
      trigger =
        new ChangeMinimumForDoublesTrigger(doubleParameter, condition, minimum,
          isMinimumIncluded);

    }
    else if (parameter instanceof DoubleListParameter) {
      Double minimum =
        parser.extractAttributeValueAsDouble(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_NEW_MINIMUM_ATTRIBUTE, null);
      boolean isMinimumIncluded =
        parser.extractAttributeValueAsBoolean(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_INCLUDE_NEW_MINIMUM_ATTRIBUTE, true);
      DoubleListParameter doubleListParameter = (DoubleListParameter) parameter;
      trigger =
        new ChangeMinimumForDoublesTrigger(doubleListParameter, condition,
          minimum, isMinimumIncluded);
    }
    else if (parameter instanceof IntegerParameter) {
      Integer minimum =
        parser.extractAttributeValueAsInteger(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_NEW_MINIMUM_ATTRIBUTE, null);
      IntegerParameter integerParameter = (IntegerParameter) parameter;
      trigger =
        new ChangeMinimumForIntegersTrigger(integerParameter, condition,
          minimum);
    }
    else if (parameter instanceof IntegerListParameter) {
      Integer minimum =
        parser.extractAttributeValueAsInteger(
          CHANGE_MINIMUM_TRIGGER_ELEMENT_NEW_MINIMUM_ATTRIBUTE, null);
      IntegerListParameter integerListParameter =
        (IntegerListParameter) parameter;
      trigger =
        new ChangeMinimumForIntegersTrigger(integerListParameter, condition,
          minimum);
    }
    else {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2};\n{3};\n{4}..", parameterName,
        DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
        DoubleListParameterFactory.DOUBLE_LIST_PARAMETER_ELEMENT,
        IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
        IntegerListParameterFactory.INTEGER_LIST_PARAMETER_ELEMENT);
    }
    parser.checkAttributes();
    return trigger;
  }

  /**
   * <p>
   * Carrega um {@link DisableParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #DISABLE_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<DisableParameterTrigger> loadDisableParameterTriggers(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    Set<SimpleParameter<?>> parameters =
      loadTriggerParameters(parser, configurator);
    Condition condition = loadTriggerCondition(parser, configurator);
    parser.checkAttributes();
    List<DisableParameterTrigger> triggers =
      new LinkedList<DisableParameterTrigger>();
    for (SimpleParameter<?> parameter : parameters) {
      triggers.add(new DisableParameterTrigger(parameter, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

  /**
   * <p>
   * Carrega um {@link EnableParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #ENABLE_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<EnableParameterTrigger> loadEnableParameterTrigger(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    Set<SimpleParameter<?>> parameters =
      loadTriggerParameters(parser, configurator);
    Condition condition = loadTriggerCondition(parser, configurator);
    parser.checkAttributes();
    List<EnableParameterTrigger> triggers =
      new LinkedList<EnableParameterTrigger>();
    for (SimpleParameter<?> parameter : parameters) {
      triggers.add(new EnableParameterTrigger(parameter, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

  /**
   * <p>
   * Carrega um {@link HideParameterTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #HIDE_PARAMETER_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<HideParameterTrigger> loadHideParameterTriggers(
    XmlParser parser, SimpleAlgorithmConfigurator configurator)
    throws ParseException {
    List<HideParameterTrigger> triggers =
      new LinkedList<HideParameterTrigger>();
    Condition condition = loadTriggerCondition(parser, configurator);
    Set<SimpleParameter<?>> parameters =
      loadTriggerParameters(parser, configurator);
    parser.checkAttributes();
    for (SimpleParameter<?> parameter : parameters) {
      triggers.add(new HideParameterTrigger(parameter, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

  /**
   * <p>
   * Carrega um {@link HideGroupTrigger}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #HIDE_GROUP_TRIGGER_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private List<HideGroupTrigger> loadHideGroupTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    List<HideGroupTrigger> triggers = new LinkedList<HideGroupTrigger>();
    Condition condition = loadTriggerCondition(parser, configurator);
    Set<ParameterGroup> groups = loadTriggerGroups(parser, configurator);
    parser.checkAttributes();
    for (ParameterGroup group : groups) {
      triggers.add(new HideGroupTrigger(group, condition));
    }
    return Collections.unmodifiableList(triggers);
  }

}
