package csbase.logic.algorithms.validation;

/**
 * Resultado da validao de um elemento especfico do tipo E.
 * 
 * @param <E> Tipo do elemento validado.
 */
public abstract class ElementValidationResult<E> implements Validation {

  /**
   * O elemento validado.
   */
  protected E element;

  /**
   * A mensagem de erro.
   */
  protected LocalizedMessage message;

  /**
   * Construtor para o caso de erro.
   * 
   * @param validation A validao.
   * @param element O elemento validado.
   */
  public ElementValidationResult(final Validation validation, final E element) {
    if (validation == null) {
      throw new IllegalArgumentException("O parmetro validation est nulo.");
    }
    if (!validation.isWellSucceded()) {
      setErrorMessage(validation.getMessage());
    }
    setElement(element);
  }

  /**
   * Construtor para o caso de erro.
   * 
   * @param message A mensagem.
   * @param element O elemento validado.
   */
  public ElementValidationResult(final LocalizedMessage message, final E element) {
    setErrorMessage(message);
    setElement(element);
  }

  /**
   * Retorna o elemento validado.
   * 
   * @return element O elemento.
   */
  public E getElement() {
    return element;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LocalizedMessage getMessage() {
    return message;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWellSucceded() {
    return (message == null);
  }

  /**
   * Atribui a mensagem de erro.
   * 
   * @param errorMessage A mensagem de erro (No aceita {@code null}).
   */
  protected void setErrorMessage(LocalizedMessage errorMessage) {
    if (errorMessage == null) {
      throw new IllegalArgumentException("O parmetro errorMessage est nulo.");
    }
    this.message = errorMessage;
  }

  /**
   * Atribui o elemento.
   * 
   * @param element O elemento.
   */
  protected void setElement(E element) {
    if (element == null) {
      throw new IllegalArgumentException("O parmetro element est nulo.");
    }
    this.element = element;
  }
}
