/*
 * $Id: ProjectServiceInterface.java 170919 2015-12-29 17:02:16Z isabella $
 */
package csbase.remote;

import java.io.File;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import csbase.logic.ClientOptimizationMode;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientProjectFileInfo;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.FileInfoSearchResult;
import csbase.logic.FileLockListenerInterface;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectFileInfo;
import csbase.logic.ProjectFileTypeInfo;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.UserProjectInfo;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Define a interface remota do servio de projetos do Server.
 */
public interface ProjectServiceInterface
  extends ServiceInterface, RemoteObservable {
  /** Nome do servio para o <code>ServiceManager</code> */
  static final String SERVICE_NAME = "ProjectService";

  /**
   * Nome da propriedade da sesso do usurio no servidor que indica o projeto
   * corrente (na rvore) do usurio.
   */
  static final String CURRENT_PROJECT_SESSION_PROPERTY = "current_project";

  /**
   * Constante que representa a espera, sem timeout, por um lock de arquivo.
   */
  static final int INFINITE_TIMEOUT = -1;

  /**
   * Verifica se o usurio possui algum projeto.
   * 
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio possui algum projeto prprio
   * @throws RemoteException
   * 
   * @see #userParticipatesOnSharedProjects(Object)
   * @see #getProjectsFromUser(Object)
   */
  boolean userHasHisOwnProjects(Object userId) throws RemoteException;

  /**
   * Verifica se um usurio possui acesso de escrita a um projeto.
   * 
   * @param projectId identificador do projeto
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio tem permisso de escrita no projeto
   * @throws RemoteException
   */
  boolean userCanWrite(Object projectId, Object userId) throws RemoteException;

  /**
   * Verifica se um usurio possui permisso de escrita em diferentes projetos.
   * 
   * @param projects lista com identificadores dos projetos
   * @param userId identificador do usurio
   * @return lista com booleanos indicando com <code>true</code> aqueles para os
   *         quais o usurio tem permisso de escrita. Os ndices correspondem
   *         aos ndices dos projetos na lista recebida como parmetro
   * @throws RemoteException
   */
  public List<Boolean> userCanWrite(List<Object> projects, Object userId)
    throws RemoteException;

  /**
   * Retorna os nomes de todos os projetos de um usurio. A lista retornada
   * inclui apenas os projetos dos quais o usurio especificado  dono, no
   * incluindo os projetos de outros usurios aos quais ele tem acesso.
   * 
   * @param userId Identificador do usurio.
   * @return Lista de UserProjectInfo contendo informaes dos projetos dos
   *         quais o usurio indicado  o dono. Retorna <code>null</code> se o
   *         usurio ainda no possui "home".
   * @throws RemoteException
   */
  List<UserProjectInfo> getProjectsFromUser(Object userId)
    throws RemoteException;

  /**
   * Obtm todos os projetos que um usurio participa, mas no  o administrador
   * do projeto.
   * 
   * @param userId Identificador do usurio.
   * @return Lista de UserProjectInfo contendo informaes dos projetos aos
   *         quais o usurio indicado tem acesso.
   * @throws RemoteException
   */
  List<UserProjectInfo> getProjectsSharedWithUser(Object userId)
    throws RemoteException;

  /**
   * Verifica se um usurio participa de projetos de outros usurios.
   * 
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio participa de projetos de outros
   *         usurios
   * @throws RemoteException
   * 
   * @see #userHasHisOwnProjects(Object)
   * @see #getProjectsSharedWithUser(Object)
   */
  boolean userParticipatesOnSharedProjects(Object userId)
    throws RemoteException;

  /**
   * Configura as permisses de acesso ao projeto.
   * 
   * @param projectId - identificador do projeto
   * @param sharingType - tipo do compartilhamento
   * @param usersRO - conjunto de usurios com acesso RO. O valor
   *        <code>null</code> indica que a lista no foi alterada
   * @param usersRW - conjunto de usurios com acesso RW. O valor
   *        <code>null</code> indica que a lista no foi alterada
   * @throws RemoteException
   */
  void updateUsers(Object projectId, SharingType sharingType,
    Set<Object> usersRO, Set<Object> usersRW) throws RemoteException;

  /**
   * Remove um usurio das listas de acesso. Cada usurio por construo pode
   * pertencer a apenas uma das listas; este mtodo tenta a remoo em ambas as
   * listas antes de falhar.
   * 
   * @param projectId - identificador do projeto
   * @param userId - identificador do usurio
   * @return flag indicando se o usurio foi removido de alguma das listas
   * @throws RemoteException
   */
  boolean removeUser(Object projectId, Object userId) throws RemoteException;

  /**
   * Remove um conjunto de usurios de ambas as listas de acesso (RO e RW).
   * 
   * @param projectId - identificador do projeto
   * @param usersId - conjunto de identificadores de usurios a serem removidos
   * @throws RemoteException
   */
  void removeUsers(Object projectId, Set<Object> usersId)
    throws RemoteException;

  /**
   * Remove todos os projetos de um usurio e seu diretrio-base
   * 
   * @param userId Id do usurio
   * 
   * @return <code>true</code> se removeu os projetos e o diretrio-base com
   *         sucesso ou se o diretrio-base j no existia ou <code>false</code>
   *         caso a remoo tenha falhado.
   * 
   * @throws RemoteException
   */
  boolean removeUserProjects(Object userId) throws RemoteException;

  /**
   * Abre um projeto de um usurio.
   * 
   * @param projectId Identificador do projeto a ser aberto.
   * @param notify indicativo de notificao aos clientes de projeto aberto.
   * 
   * @return Objeto da classe {@link CommonClientProject} que representa o
   *         projeto.
   * 
   * @throws RemoteException
   */
  CommonClientProject openProject(Object projectId, boolean notify)
    throws RemoteException;

  /**
   * Cria, abre e retorna um novo projeto.
   * 
   * @param info Informaes do projeto.
   * 
   * @return o projeto criado
   * 
   * @throws RemoteException
   */
  CommonClientProject createProject(CommonProjectInfo info)
    throws RemoteException;

  /**
   * Cria um novo projeto solicitando alocao de rea em disco.
   * 
   * @param info Informaes do projeto.
   * @param size O tamanho da rea de alocao.
   * 
   * @throws RemoteException
   */
  void createProjectWithAllocation(CommonProjectInfo info, long size)
    throws RemoteException;

  /**
   * Informa que um ou mais projetos tiveram sua rea concedida. Os projetos
   * passam a estar bloqueados. Esse mtodo somente pode ser chamado pelo
   * administrador do sistema.
   * 
   * @param infos Um array com os objetos <code>ProjectAdminInfo</code> com as
   *        informaes de desbloqueio de projetos devido a alocao de rea
   *        pelo administrador.
   * 
   * @throws RemoteException
   */
  void unlockProjectsWithAllocation(ProjectAdminInfo[] infos)
    throws RemoteException;

  /**
   * Informa que um ou mais projetos tiveram sua rea liberada. Esse mtodo
   * somente pode ser chamado pelo administrador do sistema.
   * 
   * @param infos Um array com os objetos <code>ProjectAdminInfo</code> com as
   *        informaes de liberao da rea.
   * 
   * @throws RemoteException
   */
  void freeAreaForProjects(ProjectAdminInfo[] infos) throws RemoteException;

  /**
   * Modifica as informaes de um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param info Novas informaes do projeto.
   * 
   * @return As novas informaes do projeto. Caso alguma informao no possa
   *         ser alterada, o valor anterior  retornado.
   * 
   * @throws RemoteException
   */
  CommonProjectInfo modifyProject(Object projectId, CommonProjectInfo info)
    throws RemoteException;

  /**
   * Remove um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @throws RemoteException
   */
  void removeProject(Object projectId) throws RemoteException;

  /**
   * Fecha um projeto.
   * 
   * @param projectId Identificador do projeto a ser fechado.
   * @param notify indicativo de notificao ao cliente.
   * 
   * @throws RemoteException
   */
  void closeProject(Object projectId, boolean notify) throws RemoteException;

  /**
   * Informa se um dado projeto existe.
   *
   * @param projectId Identificador do projeto.
   * @return Verdadeiro caso exista, falso caso contrrio.
   */
  boolean existsProject(Object projectId) throws RemoteException;

  /**
   * Reconstri a rvore do projeto a partir do sistema de arquivos do servidor.
   * <p>
   * <b>IMPORTANTE:</b> toda a rvore  obtida recursivamente a partir da raiz;
   * portanto, esta  uma operao potencialmente custosa.
   * 
   * @param projectId Identificador do projeto.
   * 
   * @throws RemoteException
   * 
   * @see #refreshTree(Object)
   */
  void rebuildTree(Object projectId) throws RemoteException;

  /**
   * Atualiza a rvore, a partir da sua raiz. Ao contrrio de
   * {@link #rebuildTree(Object)}, no obtm toda a rvore a priori, deixando
   * que isto seja feito por demanda pelos handlers do evento correspondente.
   * 
   * @param projectId identificador do projeto
   * @throws RemoteException
   * 
   * @see #rebuildTree(Object)
   */
  void refreshTree(Object projectId) throws RemoteException;

  /**
   * Reconstri a rvore do diretrio especificado a partir do sistema de
   * arquivos do servidor.
   * <p>
   * <b>IMPORTANTE:</b> toda a rvore  obtida recursivamente a partir do
   * diretrio especificado; portanto, esta  uma operao potencialmente
   * custosa.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio a ser atualizado. Se o array tiver tamanho
   *        zero, significa que a atualizao ser feita a partir da raiz.
   * @throws RemoteException
   */
  void rebuildDir(Object projectId, String[] path) throws RemoteException;

  /**
   * Atualiza a rvore, a partir do diretrio especificado. Ao contrrio de
   * {@link #rebuildDir(Object, String[])}, no obtm toda a rvore a priori,
   * deixando que isto seja feito por demanda pelos handlers do evento
   * correspondente.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio a ser atualizado. Se o array tiver tamanho
   *        zero, significa que a atualizao ser feita a partir da raiz.
   * @throws RemoteException
   */
  void refreshDir(Object projectId, String[] path) throws RemoteException;

  /**
   * Cria um novo diretrio, ou uma nova rvore de diretrios, dentro de um
   * projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio a ser criado. Todos os elementos do array
   *        que ainda no existam sero criados no projeto como diretrios, a
   *        partir da raiz.
   * @throws RemoteException
   */
  void createDirectory(Object projectId, String[] path) throws RemoteException;

  /**
   * Retorna a lista de filhos imediatos de um diretrio (ou <code>null</code>
   * se o arquivo no for um diretrio). <b>No</b> percorre a subrvore
   * recursivamente.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio, a partir da raiz.
   * @return Lista de filhos imediatos do diretrio, ou <code>null</code> se
   *         arquivo indicado no for um diretrio.
   * @throws RemoteException
   * @see #getChildren(Object, String[], boolean)
   */
  ClientProjectFile[] getChildren(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Retorna a lista de filhos imediatos de um diretrio (ou <code>null</code>
   * se o arquivo no for um diretrio).
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio, a partir da raiz.
   * @param recursive Indica se a sub-rvore deve ser obtida recursivamente.
   * @return Lista de filhos imediatos do diretrio, ou <code>null</code> se
   *         arquivo indicado no for um diretrio.
   * @throws RemoteException
   * @see #getChildren(Object, String[])
   */
  ClientProjectFile[] getChildren(Object projectId, String[] path,
    boolean recursive) throws RemoteException;

  /**
   * Retorna um arquivo.
   * 
   * @param projectId identificador do projeto.
   * @param path caminho para o <b>diretrio que contm o arquivo</b>, a partir
   *        da raiz. No inclui o nome do projeto, nem o nome do arquivo.
   * @param name nome do arquivo.
   * @return o arquivo
   * @throws RemoteException
   * 
   * @see #getChild(Object, String[])
   */
  ClientProjectFile getChild(Object projectId, String[] path, String name)
    throws RemoteException;

  /**
   * Retorna um arquivo.
   * 
   * @param projectId identificador do projeto
   * @param path path para o arquivo, a partir da raiz. No inclui o nome do
   *        projeto
   * @return o arquivo
   * @throws RemoteException
   * 
   * @see #getChild(Object, String[], String)
   */
  ClientProjectFile getChild(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Cria um novo arquivo em um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do diretrio pai, a partir da raiz.
   * @param name Nome do arquivo a ser criado.
   * @param type Tipo do arquivo a ser criado.
   * @throws RemoteException
   */
  void createFile(Object projectId, String[] path, String name, String type)
    throws RemoteException;

  /**
   * Cria simultaneamente vrios arquivos dentro de um mesmo diretrio. Os
   * caminhos dos arquivos devem ser passados relativamente ao diretrio-base,
   * que atua como raiz da rvore de arquivos.
   * 
   * @param projectId Identificador do projeto.
   * @param parentPath Caminho do diretrio-base, a partir da raiz, ancestral de
   *        todos os arquivos a serem criados.
   * @param fileInfoList Lista de dados dos arquivos a serem criados.
   * @throws RemoteException
   */
  void createFiles(Object projectId, String[] parentPath,
    List<ProjectFileInfo> fileInfoList) throws RemoteException;

  /**
   * Troca o nome de um arquivo de um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo a ser renomeado, a partir da raiz.
   * @param name Novo nome do arquivo.
   * @throws RemoteException
   */
  void renameFile(Object projectId, String[] path, String name)
    throws RemoteException;

  /**
   * Muda o tipo de um arquivo de um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz, cujo tipo ser alterado.
   * @param type Novo tipo do arquivo.
   * @throws RemoteException
   */
  void changeFileType(Object projectId, String[] path, String type)
    throws RemoteException;

  /**
   * Copia um arquivo de um projeto para outro diretrio.
   * 
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo a ser copiado, a partir da raiz.
   * @param dirPath Caminho do diretrio destino, a partir da raiz.
   * @throws RemoteException
   */
  void copyFile(Object projectId, String[] filePath, String[] dirPath)
    throws RemoteException;

//  /**
//   * Copia um arquivo de um projeto para outro diretrio.
//   * 
//   * @param projectId Identificador do projeto.
//   * @param filePath Caminho do arquivo a ser copiado, a partir da raiz.
//   * @param dirPath Caminho do diretrio destino, a partir da raiz.
//   * @param fileName Nome do arquivo no diretrio destino.
//   * @throws RemoteException
//   */
//  void copyFile(Object projectId, String[] filePath, String[] dirPath,
//    String fileName) throws RemoteException;

  /**
   * Copia um arquivo de um projeto para um diretrio de outro projeto.
   * 
   * @param sourceProjectId Identificador do projeto Origem.
   * @param filePathSource Caminho do arquivo a ser copiado, a partir da raiz.
   * @param targetProjectId Identificador do projeto Destino.
   * @param dirPathTarget Caminho do diretrio destino, a partir da raiz.
   * @throws RemoteException
   */
  void copyFile(Object sourceProjectId, String[] filePathSource,
    Object targetProjectId, String[] dirPathTarget) throws RemoteException;

  /**
   * Move um arquivo de um projeto para outro diretrio.
   * 
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo a ser movido, a partir da raiz.
   * @param dirPath Caminho do diretrio destino, a partir da raiz.
   * @throws RemoteException
   */
  void moveFile(Object projectId, String[] filePath, String[] dirPath)
    throws RemoteException;

  /**
   * Move um arquivo de um projeto para um diretrio de outro projeto.
   * 
   * @param sourceProjectId Identificador do projeto Origem.
   * @param filePathSource Caminho do arquivo a ser copiado, a partir da raiz.
   * @param targetProjectId Identificador do projeto Destino.
   * @param dirPathTarget Caminho do diretrio destino, a partir da raiz.
   * @throws RemoteException
   */
  void moveFile(Object sourceProjectId, String[] filePathSource,
    Object targetProjectId, String[] dirPathTarget) throws RemoteException;

  /**
   * Remove um arquivo de um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo a ser removido, a partir da raiz.
   * @throws RemoteException
   */
  void removeFile(Object projectId, String[] path) throws RemoteException;

  /**
   * Atualiza as informaes de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo a ser atualizado, a partir da raiz.
   * @return A nova representao do arquivo.
   * @throws RemoteException
   */
  ClientProjectFileInfo getUpdatedFileInfo(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Remove vrios arquivos de um projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param paths Lista de caminhos, a partir da raiz, especificando os arquivos
   *        a serem removidos.
   * @throws RemoteException
   */
  void removeFiles(Object projectId, String[][] paths) throws RemoteException;

  /**
   * Obtm a descrio atual deste arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return O texto com a descrio do arquivo.
   * @throws RemoteException
   */
  String getFileDescription(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Muda a descrio atual deste arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param text O novo texto da descrio do arquivo.
   * @throws RemoteException
   */
  void setFileDescription(Object projectId, String[] path, String text)
    throws RemoteException;

  /**
   * Anexa um texto  descrio atual deste arquivo. Permite a gerao de
   * histricos de modificao.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param text O texto a ser anexado.
   * @throws RemoteException
   */
  void appendFileDescription(Object projectId, String[] path, String text)
    throws RemoteException;

  /**
   * Verifica se um arquivo existe no servidor.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho para o arquivo, a partir da raiz.
   * @return Verdadeiro se o arquivo existe no servidor, falso caso contrrio.
   * @throws RemoteException
   */
  boolean existsFile(Object projectId, String[] path) throws RemoteException;

  /**
   * Abre um arquivo para leitura e escrita.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo a ser aberto, a partir da raiz.
   * @param readOnly Indica se o arquivo deve ser aberto apenas para leitura ou
   *        para leitura e escrita.
   * @return O channel que conecta ao arquivo no servidor.
   * @throws RemoteException
   */
  RemoteFileChannelInfo openFileChannel(Object projectId, String[] path,
    boolean readOnly) throws RemoteException;

  /**
   * Informa onde o diretrio do projeto est instalado no servidor.
   * 
   * @param projectId Identificador do projeto.
   * @return O caminho canonico do diretrio do projeto no servidor.
   * @see File#getCanonicalPath()
   * @throws RemoteException
   */
  String getProjectLocationInServer(Object projectId) throws RemoteException;

  /**
   * Obtm lock de escrita exclusivo para um arquivo. Locks exclusivos so
   * obtidos apenas uma vez.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Identificador do lock se o lock foi obtido, ou null caso contrrio.
   * @throws RemoteException
   */
  Object acquireExclusiveLock(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Tenta obter um lock de escrita exclusivo para um arquivo. Locks exclusivos
   * so obtidos apenas uma vez.
   * 
   * Este mtodo cadastra uma solicitao de lock numa fila priorizada pela
   * ordem de chegada dos pedidos de lock. Quando o lock  obtido, o mtodo
   * FileLockListenerInterface.fileLocked  chamado. Quando o tempo mximo de
   * espera do lock  atingido e o lock no foi obtido o mtodo
   * FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param listener Observador do lock cadastrado.
   * @param timeout Tempo mximo de espera pelo lock em milisegundos. A
   *        constante INFINITE_TIMOUT existe para especificar timeout infinito.
   * @return Identificador do pedido de lock.
   * @throws RemoteException
   */
  Object acquireExclusiveLock(Object projectId, String[] path,
    FileLockListenerInterface listener, long timeout) throws RemoteException;

  /**
   * Obtm lock de escrita compartilhado para um arquivo. Locks compartilhados
   * podem ser obtidos mltiplas vezes; cada novo lock para o arquivo incrementa
   * o contador de referncias para o lock. Locks compartilhados s podem ser
   * removidos se o contador de referncias for igual a zero.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Identificador do lock se o lock foi obtido, ou null caso contrrio.
   * @throws RemoteException
   */
  Object acquireSharedLock(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Tenta obter lock de escrita compartilhado para um arquivo. Locks
   * compartilhados podem ser obtidos mltiplas vezes; cada novo lock para o
   * arquivo incrementa o contador de referncias para o lock. Locks
   * compartilhados s podem ser removidos se o contador de referncias for
   * igual a zero.
   * 
   * Este mtodo cadastra uma solicitao de lock compartilhado em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock. Quando o lock 
   * obtido, o mtodo FileLockListenerInterface.fileLocked  chamado. Quando o
   * tempo mximo de espera do lock  atingido e o lock no foi obtido o mtodo
   * FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param listener Observador do lock cadastrado.
   * @param timeout Tempo mximo de espera pelo lock em milisegundos. A
   *        constante INFINITE_TIMOUT existe para especificar timeout infinito.
   * @return Identificador do pedido de lock.
   * @throws RemoteException
   */
  Object acquireSharedLock(Object projectId, String[] path,
    FileLockListenerInterface listener, long timeout) throws RemoteException;

  /**
   * Libera lock de um arquivo. O lock s  liberado se o identificador for o
   * mesmo do lock obtido. Se o lock  compartilhado, a contagem de referncias
   * para o lock  decrementada. Quando o nmero de referncias chega a zero, o
   * lock  efetivamente removido.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param lockId Identificador do lock.
   * @return Contagem de referncias para o lock.
   * @throws RemoteException
   */
  int releaseLock(Object projectId, String[] path, Object lockId)
    throws RemoteException;

  /**
   * Fora a remoo do lock do arquivo. S pode ser usado pelo administrador ou
   * pelo usurio dono do projeto, se acontencer alguma situao de ter um lock
   * "preso". Situaes onde as sesso do usurio que foi finalizadas por falha
   * na comunicao, ou usurios que terminam o cliente abruptamente j esto
   * previstas e no devem precisar deste mecanismo de segurana.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Contagem de referncias para o lock.
   * @throws RemoteException
   */
  int forceReleaseLock(Object projectId, String[] path) throws RemoteException;

  /**
   * Verifica se o usurio corrente  o dono do lock de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Verdadeiro se o usurio  o dono do lock, falso caso contrrio.
   * @throws RemoteException
   */
  boolean userOwnsLock(Object projectId, String[] path) throws RemoteException;

  /**
   * Retorna o tamanho de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Tamanho do arquivo.
   * @throws RemoteException
   */
  long fileSize(Object projectId, String[] path) throws RemoteException;

  /**
   * Retorna a data de modificao de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Data de modificao do arquivo.
   * @throws RemoteException
   */
  long getModificationDate(Object projectId, String[] path)
    throws RemoteException;

  /**
   * Registra um observador interessado nas mudanas que ocorrerem em um
   * projeto.
   * 
   * @param observer O observador remoto que registra o interesse nas mudanas
   *        sobre um projeto.
   * @param arg Identificador do projeto no qual o observador est interessado.
   * @throws RemoteException
   */
  @Override
  void addObserver(RemoteObserver observer, Object arg) throws RemoteException;

  /**
   * Remove um observador registrado em um projeto.
   * 
   * @param observer O observador remoto a ser removido.
   * @param arg Identificador do projeto.
   * @return Verdadeiro se o observador foi removido, falso caso contrrio.
   * @throws RemoteException
   */
  @Override
  boolean deleteObserver(RemoteObserver observer, Object arg)
    throws RemoteException;

  /**
   * Obtm as propriedades de todos os tipos de arquivos cadastrados.
   * 
   * @param loc O locale para traduo dos termos, quando aplicvel.
   * @return Map com as propriedades de todos os tipos de arquivos.
   * @throws RemoteException
   */
  Map<String, ProjectFileTypeInfo> getAllFileTypes(Locale loc)
    throws RemoteException;

  /**
   * Obtem todas as propriedades de um tipo de arquivo.
   * 
   * @param type o tipo do arquivo desejado
   * @return informaes de um tipo de arquivo
   * @throws RemoteException
   */
  ProjectFileTypeInfo getFileType(String type) throws RemoteException;

  /**
   * Obtem o mimeType a partir do tipo do arquivo.
   * 
   * @param type O tipo de arquivo.
   * @return O mime-type correspondente.
   * @throws RemoteException
   */
  String getMimeType(String type) throws RemoteException;

  /**
   * Obtm as informaes de todos os projetos dos usurios.
   * 
   * @return um array de <code>ProjectAdminInfo</code>.
   * @throws RemoteException
   */
  ProjectAdminInfo[] getLockedProjectAdminInfo() throws RemoteException;

  /**
   * Verifica se um projeto est desbloqueado utilizando rea reservada.
   * 
   * @param projectId O identificador do projeto.
   * @return verdadeiro, se o projeto est desbloqueado utilizando rea
   *         reservada ou falso, caso contrrio.
   * @throws RemoteException
   */
  boolean isUnlockedWithAreaAllocated(Object projectId) throws RemoteException;

  /**
   * Muda a data de alterao de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param date Data, como um long.
   * @throws RemoteException
   */
  void setFileModificationDate(Object projectId, String[] path, long date)
    throws RemoteException;

  /**
   * Altera o estado do arquivo para o valor dado no parmetro
   * <code>underConstruction</code>.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param underConstruction Indica se o arquivo deve ficar em construo.
   * @throws RemoteException
   */
  void setUnderConstruction(Object projectId, String[] path,
    boolean underConstruction) throws RemoteException;

  /**
   * Inicia ou altera o intervalo de atualizao de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param interval O intervalo (em segundos) de atualizao do arquivo.
   * @param notification Indica se o usurio deseja ou no receber notificaes.
   * 
   * @throws RemoteException
   */
  void startUpdate(Object projectId, String[] path, long interval,
    boolean notification) throws RemoteException;

  /**
   * Inicia ou altera o intervalo de atualizao de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param interval O intervalo (em segundos) de atualizao do arquivo.
   * @param notification Indica se o usurio deseja ou no receber notificaes.
   * @param extraInfo informaes extras para parametrizar a atualizao
   * 
   * @throws RemoteException
   */
  void startUpdate(Object projectId, String[] path, long interval,
    boolean notification, Serializable extraInfo) throws RemoteException;

  /**
   * Pra a atualizao de um arquivo.
   * 
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @throws RemoteException
   */
  void stopUpdate(Object projectId, String[] path) throws RemoteException;

  /**
   * Obtm a lista de resultados de uma busca de arquivos em um projeto, de
   * acordo com as condies solicitadas (nome do arquivo, se deve ignorar
   * mausculas/minsculas, se usa ou no uma expresso regular).
   * 
   * @param projectId O identificador do projeto.
   * @param text - Texto sendo pesquisado.
   * @param isCaseInsensitive - Considerar maisculas/minsculas.
   * @param isRegex - Indica que o texto  uma expresso regular.
   * @return Lista com as informaes sobre todos arquivos encontrados. Caso
   *         nenhum arquivo tenha sido encontrado, retorna uma lista vazia.
   * @throws RemoteException
   */
  List<FileInfoSearchResult> getAllFileInfoSearchResult(Object projectId,
    String text, boolean isCaseInsensitive, boolean isRegex)
      throws RemoteException;

  /**
   * Obtm um conjunto com o identificador de todos os projetos gerenciveis
   * pelo usurio corrente.<br>
   * No caso de um usurio comum isso representa todos os projetos daquele
   * usurio, diferente do administrador que pode gerenciar todos os projetos.
   * 
   * @return Um conjunto com o identificador de todos os projetos gerenciveis
   *         pelo usurio corrente. Caso o usurio no tenha acesso a nenhum
   *         projeto, o conjunto ter tamanho zero.
   * @throws RemoteException
   */
  Set<Object> getManageableProjectsId() throws RemoteException;

  /**
   * Converte um projeto sem alocao de rea em um projeto que aguarda
   * alocao.
   * 
   * @param projectId Identificador do projeto.
   * @param areaSize Tamanho (em Gb) da rea a ser alocada ao projeto.
   * @throws RemoteException Exceo no acesso remoto aos dados.
   */
  void setUnallocatedProjectAsLocked(Object projectId, int areaSize)
    throws RemoteException;

  /**
   * Converte um projeto com alocao de rea em um projeto sem alocao.
   * 
   * OBS. Este mtodo foi criado para atender uma necessidade especfica. Deve
   * ser usado com a mesma cautela exigida pelo mtodo freeAreaForProjects()
   * (ambos descartam informaes administrativas).
   * 
   * @param projectId Identificador do projeto.
   * @throws RemoteException Exceo no acesso remoto aos dados.
   */
  void setAllocatedProjectAsUnallocated(Object projectId)
    throws RemoteException;

  /**
   * Converte um projeto sem alocao de rea em um projeto com alocao.
   * 
   * OBS. Este mtodo foi criado para atender uma necessidade especfica e deve
   * ser usado com cautela.
   * 
   * @param projectId Identificador do projeto.
   * @param areaSize Tamanho (em Gb) da rea a ser alocada ao projeto.
   * @throws RemoteException Exceo no acesso remoto aos dados.
   */
  void setUnallocatedProjectAsAllocated(Object projectId, int areaSize)
    throws RemoteException;

  /**
   * Obtm as informaes administrativas de um projeto de um usurio.
   * 
   * @param projectId Identificador do projeto.
   * @return O <code>ProjectAdminInfo</code> do projeto de um usurio ou nulo
   *         caso o projeto no esteja bloqueado.
   * @throws RemoteException Exceo no acesso remoto aos dados.
   */
  ProjectAdminInfo getProjectAdminInfo(Object projectId) throws RemoteException;

  /**
   * Indica se o servio de projetos utiliza reserva de rea.
   * 
   * @return Boolean que indica se o servio de projetos utiliza reserva de
   *         rea.
   * @throws RemoteException Exceo no acesso remoto aos dados.
   */
  boolean isAreaReserved() throws RemoteException;

  /***
   * 
   * @return O modo de otimizao que foi definido neste servio.
   * @throws RemoteException
   */
  ClientOptimizationMode getOptimizationMode() throws RemoteException;

  /**
   * 
   * @return O path de acesso local no cliente para a pasta de projetos. 
   *         utilizada quando o Modo de Otimizao estiver definido como
   *         <code>ClientOptimizationMode.GLOBAL</code>
   * @throws RemoteException
   */
  String getLocalProjectPath() throws RemoteException;
}
