package csbase.logic.algorithms.parsers;

import static csbase.logic.algorithms.parameters.FileParameterMode.REGULAR_FILE;

import java.util.Arrays;
import java.util.EnumSet;

import csbase.exception.BugException;
import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.InputURLListParameter;
import csbase.logic.algorithms.parameters.InputURLParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.URLParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.URLProtocolConverter;

/**
 * Analisador base para parmetros de URL.
 *
 * @param <T> Tipo do parmetro.
 * @author Tecgraf
 */
public abstract class AbstractURLParameterParser <T extends
  SimpleParameter<?>> extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #MUST_SORT_ATTRIBUTE} do elemento
   * {@link InputURLListParameter}.
   * </p>
   *
   * Indica se o {@link InputURLParameter arquivo de entrada} se a relao de
   * arquivos ser ordenada,  opcional, o seu valor-padro 
   * {@link #MUST_SORT_DEFAULT_VALUE} e  do tipo booleano.
   * </p>
   */
  protected static final String MUST_SORT_ATTRIBUTE = "ordenar";

  /**
   * <p>
   * O valor-padro para o atributo {@link #MUST_SORT_ATTRIBUTE} do elemento
   * {@link InputURLListParameter}
   * </p>
   * <p>
   * O seu valor  {@value #MUST_SORT_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean MUST_SORT_DEFAULT_VALUE = true;

  /**
   * <p>
   * O atributo {@value #CAN_USE_PIPE_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputURLParameter}</li>
   * <li>{@link OutputURLParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o se o parmetro do tipo URL aceita trabalhar com pipes,  opcional,
   * o seu valor-padro  {@link #CAN_USE_PIPE_DEFAULT_VALUE} e  do tipo string
   * e os valores vlidos para ele so:
   * <ul>
   * <li>{@link FileParameterPipeAcceptance#TRUE}</li>
   * <li>{@link FileParameterPipeAcceptance#FALSE}</li>
   * <li>{@link FileParameterPipeAcceptance#ALWAYS}</li>
   * </ul>
   * </p>
   */
  protected static final String CAN_USE_PIPE_ATTRIBUTE = "permitir_pipe";

  /**
   * <p>
   * O valor-padro para o atributo {@link #CAN_USE_PIPE_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link InputURLParameter}</li>
   * <li>{@link OutputURLParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@link #CAN_USE_PIPE_DEFAULT_VALUE}.
   * </p>
   */
  protected static final FileParameterPipeAcceptance CAN_USE_PIPE_DEFAULT_VALUE =
    FileParameterPipeAcceptance.TRUE;

  /**
   * Atributo que define o modo de funcionamento do parmetro do tipo URL. Este
   * atributo  opcional, o seu valor-padro 
   * {@link #CATEGORY_VALUE_REGULAR_FILE},
   *
   * Valores vlidos: {@link #CATEGORY_VALUE_REGULAR_FILE} e
   * {@link #CATEGORY_VALUE_DIRECTORY}
   */
  protected static final String CATEGORY_ATTRIBUTE = "categoria";

  /**
   * Valor que indica o {@link FileParameterMode#DIRECTORY modo apenas
   * diretrio} para o atributo {@link #CATEGORY_ATTRIBUTE} dodo elemento
   * {@link URLParameter}.
   */
  protected static final String CATEGORY_VALUE_DIRECTORY = "diretorio";

  /**
   * Valor que indica o {@link FileParameterMode#REGULAR_FILE modo apenas
   * arquivos} para o atributo {@link #CATEGORY_ATTRIBUTE} do elemento
   * {@link URLParameter}.
   */
  protected static final String CATEGORY_VALUE_REGULAR_FILE = "arquivo";

  /**
   * Atributo que define os protocolos permitidos no parmetro.
   *
   * Indica o {@link URLProtocol protocolo} aceito pelo parmetro do tipo URL},
   *  opcional e  do tipo string e o valores vlidos para ele so:
   * <ul>
   * <li>{@link URLProtocol#PROJECT}</li>
   * <li>{@link URLProtocol#LOCAL}</li>
   * <li>{@link URLProtocol#SGA}</li>
   * </ul>
   */
  protected static final String PROTOCOLS_ATTRIBUTE = "protocolos";

  /**
   * Atributo que indica se o {@link URLParameter url de entrada} deve exibir
   * como valor default o diretrio raz do projeto. Atributo opcional e o seu
   * valor-padro  {@link #DEFAULT_DIRECTORY_DEFAULT_VALUE} e  do tipo
   * booleano.
   */
  protected static final String DEFAULT_DIRECTORY_ATTRIBUTE =
    "usar_diretorio_raiz_como_padrao";

  /**
   * Valor-padro para o atributo {@link #DEFAULT_DIRECTORY_ATTRIBUTE} do
   * elemento {@link URLParameter}.
   */
  protected static final boolean DEFAULT_DIRECTORY_DEFAULT_VALUE = false;

  /**
   * Atributo que indica o tipo do arquivo aceito pelo parmetro.  opcional e o
   * seu tipo  {@link String}.
   * </p>
   */
  protected static final String TYPE_ATTRIBUTE = "tipo";

  /**
   * Atributo que define a localizao. A localizao  uma chave usada para
   * indicar o protocolo da URL na linha de comando.
   */
  protected static final String LOCALIZATION_ATTRIBUTE = "localizacao";

  /**
   * Converte strings para valores da enumerao FileParameterPipeAcceptance.
   */
  protected class URLParameterPipeAcceptanceConverter implements
    StringToEnumConverter<FileParameterPipeAcceptance> {

    /**
     * Retorna as strings que correspondem a cada um dos possveis valores da
     * enumerao {@link FileParameterPipeAcceptance}.
     *
     * @param value O valor da enumerao.
     * @return Um array com todas as strings que podem representar o valor.
     */
    public String[] getPossibleMatches(FileParameterPipeAcceptance value) {
      switch (value) {
        case TRUE:
          return XmlParser.TRUE_VALUES;
        case FALSE:
          return XmlParser.FALSE_VALUES;
        case ALWAYS:
          return new String[] { "sempre", "always" };
        default:
          throw new BugException();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public FileParameterPipeAcceptance valueOf(
      String string) throws ParseException {
      if (string != null) {
        String upperCaseString = string.toUpperCase();
        for (FileParameterPipeAcceptance mode : FileParameterPipeAcceptance
          .values()) {
          String[] possibleMatches = getPossibleMatches(mode);
          for (String possibleMatch : possibleMatches) {
            if (possibleMatch.toUpperCase().equals(upperCaseString)) {
              return mode;
            }
          }
        }
      }
      throw new ParseException(
        "A string " + string + " no pde ser mapeada para um item de " +
          "enumerao do tipo " + FileParameterPipeAcceptance.class
          .getName());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final T createSimpleParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {

    String localization = parser.extractAttributeValue(LOCALIZATION_ATTRIBUTE);

    String[] types = parser.extractAttributeValueAsArray(TYPE_ATTRIBUTE, null);
    String modeName = parser.extractAttributeValue(CATEGORY_ATTRIBUTE, null);

    FileParameterMode mode = null;
    if (modeName == null) {
      if (types != null) {
        for (String type : types) {
          ProjectFileType projectFileType = ProjectFileType.getFileType(type);
          if (projectFileType != null && projectFileType.isDirectory()) {
            if (mode == null) {
              mode = FileParameterMode.DIRECTORY;
            }
            else if (mode != FileParameterMode.DIRECTORY) {
              throw new ParseException(
                "Os tipos informados {0} no pertecem  mesma categoria.",
                Arrays.toString(types));
            }
          }
          else {
            if (mode == null) {
              mode = FileParameterMode.REGULAR_FILE;
            }
            else if (mode != FileParameterMode.REGULAR_FILE) {
              throw new ParseException(
                "Os tipos informados {0} no pertecem  mesma categoria.",
                Arrays.toString(types));
            }
          }
        }
      }
      else {
        mode = REGULAR_FILE;
      }
    }
    else if (modeName.equals(CATEGORY_VALUE_DIRECTORY)) {
      mode = FileParameterMode.DIRECTORY;
    }
    else if (modeName.equals(CATEGORY_VALUE_REGULAR_FILE)) {
      mode = REGULAR_FILE;
    }
    else {
      throw new ParseException(
        "A categoria informada {0} no  vlida.\nCategorias vlidas:\n{1};" +
          "\n{2}.",
        modeName, CATEGORY_VALUE_REGULAR_FILE, CATEGORY_VALUE_DIRECTORY);
    }

    if (types != null) {
      for (String type : types) {
        ProjectFileType projectFileType = ProjectFileType.getFileType(type);
        if (projectFileType != null && !projectFileType
          .isDirectory() && mode == FileParameterMode.DIRECTORY) {
          throw new ParseException(
            "A categoria informada {0} no  compatvel com o tipo informado " +
              "{1}, " + "pois o tipo  aplicavel somente a arquivos.", modeName,
            type);
        }

        if (projectFileType != null && projectFileType
          .isDirectory() && mode == FileParameterMode.REGULAR_FILE) {
          throw new ParseException(
            "A categoria informada {0} no  compatvel com o tipo informado " +
              "{1}, " + "pois o tipo  aplicavel somente a diretrios.",
            modeName, type);
        }
      }
    }
    else {
      types = new String[0];
    }

    boolean useRootDirectoryAsDefault = parser
      .extractAttributeValueAsBoolean(DEFAULT_DIRECTORY_ATTRIBUTE,
        DEFAULT_DIRECTORY_DEFAULT_VALUE);
    FileURLValue defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de sada  um arquivo.",
          DEFAULT_DIRECTORY_ATTRIBUTE);
      }
      defaultValue = new FileURLValue(".", ProjectFileType.DIRECTORY_TYPE);
    }

    String[] protocols =
      parser.extractAttributeValueAsArray(PROTOCOLS_ATTRIBUTE, null);
    EnumSet<URLProtocol> allowedProtocols;
    if (protocols != null) {
      allowedProtocols = EnumSet.noneOf(URLProtocol.class);
      URLProtocolConverter converter = new URLProtocolConverter();
      for (String protocolString : protocols) {
        URLProtocol protocol = converter.valueOf(protocolString);
        allowedProtocols.add(protocol);
      }
    }
    else {
      allowedProtocols = EnumSet.allOf(URLProtocol.class);
    }

    FileParameterPipeAcceptance usesPipe = parser
      .extractAttributeValueAsEnumeration(CAN_USE_PIPE_ATTRIBUTE,
        CAN_USE_PIPE_DEFAULT_VALUE, new URLParameterPipeAcceptanceConverter());

    return createURLParameter(parser, name, label, description, isOptional,
      isVisible, commandLinePattern, types, localization, mode, defaultValue,
      usesPipe, allowedProtocols);
  }

  protected abstract T createURLParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, String[] types, String localization,
    FileParameterMode mode, FileURLValue defaultValue,
    FileParameterPipeAcceptance usesPipe,
    EnumSet<URLProtocol> allowedProtocols) throws ParseException;
}
