/*
 * AlgorithmInfo.java
 * 
 * $Author$ $Revision$ - $Date: 2007-07-20 09:19:29 -0300
 * (Fri, 20 Jul 2007) $
 */
package csbase.logic.algorithms;

import java.io.Serializable;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import csbase.logic.AlgoEvent;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * Representa os dados de um algoritmo para ser mostrado no cliente.
 * 
 * @author clinio
 * @version $Revision$
 */
public class AlgorithmInfo implements Serializable, Comparable<AlgorithmInfo> {
  /** Tamanho mximo do nome de um algoritmo. */
  public static final int MAX_NAME_SIZE = 50;

  /**
   * Nome do subdiretrio de verses (dentro do diretrio de um algoritmo).
   * Usado para localizar as verses de um algoritmos e para compor o nome do
   * pacote que contm o configurador de uma verso de algoritmo (Configurator).
   */
  public static final String VERSIONS_DIR = "versions";

  /** Identificador do algoritmo. */
  private String id;

  /** Diretrio de localizao do algoritmo. */
  private String dir;

  /** Nome do algoritmo. */
  private String name;

  /** Descrio do algoritmo */
  private String description;

  /** Vetor com as verses de um algoritmo. */
  private Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> versions;

  /**
   * Nome do arquivo de informacoes do algoritmo. Contem os VALORES atribuidos a
   * cada um dos atributos estendidos de uma dada instancia do algoritmo.
   */
  public static final String PROPERTY_VALUES_FILE = "algorithm.properties";

  /**
   * Chave da propriedade que determina se o algoritmo deve ser exibido
   */
  public static final String HIDE_ALGORITHM_PROPERTY = "ocultar";

  /**
   * Valor da propriedade que determina se o algoritmo deve ser exibido
   */
  public static final String HIDE_ALGORITHM_VALUE = "sim";

  /** Chave da propriedade que determina o usurio (login) que criou o algoritmo */
  public static final String OWNER_ALGORITHM_PROPERTY = "criador";

  /**
   * Mapa de valores para os atributos estendidos.
   */
  private Hashtable<String, String> propertyValues;

  /**
   * Caminho para o repositrio de algoritmos.
   */
  private String algorithmRepositoryPath;

  /** Nomes completos das categorias em que o algoritmo participa */
  private List<String> categoryFullNames;

  /** Objeto que permite a observao da classe AlgorithmVersionInfo */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Construtor de um AlgorithmInfo. <b>Deve ser utilizado para a criacao e
   * inlcusao de um algoritmo</b>. A classe  criada com propriedades estendidas
   * e sem informcao das versoes e, por isso, ela ir <b>salvar a informacao de
   * atributo no arquivo de propriedades</b>.
   * 
   * @param id id do algoritmo
   * @param name nome do algoritmo
   * @param dir diretrio do algoritmo
   * @param algorithmRepositoryPath caminho para o repositrio de algoritmos.
   * @param propertyValues Vector com os valores dos atributos estendidos
   */
  public AlgorithmInfo(String id, String name, String dir,
    String algorithmRepositoryPath, Hashtable<String, String> propertyValues) {
    this(id, name, dir, algorithmRepositoryPath, propertyValues,
      new Hashtable<AlgorithmVersionId, AlgorithmVersionInfo>());
  }

  /**
   * Construtor de um AlgorithmInfo. <b>Deve ser utilizado para a transoformacao
   * de um ModifiableAlgorithmInfo em um AlgorithmInfo</b>. A classe  criada
   * com propriedades estendidas e com informcao das versoes. Porem, como nao
   * esta havendo modificacao nos atributos estendidos, <b>esse construtor nao
   * manipula o arquivo de propriedades</b>.
   * 
   * @param id id do algoritmo
   * @param name nome do algoritmo
   * @param dir diretrio do algoritmo
   * @param algorithmRepositoryPath caminho para o repositrio de algoritmos.
   * @param propertyValues vector com os valores dos atributos estendidos
   * @param versions versoes do algoritmo
   */
  public AlgorithmInfo(String id, String name, String dir,
    String algorithmRepositoryPath, Hashtable<String, String> propertyValues,
    Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> versions) {
    this.id = id;
    this.name = name;
    this.dir = dir;
    this.versions = versions;
    this.propertyValues = propertyValues;
    this.algorithmRepositoryPath = algorithmRepositoryPath;
  }

  /**
   * Constri um algoritmo a partir de um algoritmo base.
   * 
   * @param algorithmInfo algoritmo base
   */
  public AlgorithmInfo(AlgorithmInfo algorithmInfo) {
    this(algorithmInfo.getId(), algorithmInfo.getName(), algorithmInfo
      .getDirectory(), algorithmInfo.getAlgorithmRepositoryPath(),
      algorithmInfo.getPropertyValues(), algorithmInfo.getAllVersionInfo());
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de algoritmos de que alguma mudana ocorreu, os observadores locais
   * tambm so notificados.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Obtm o caminho para o repositrio de algoritmos.
   * 
   * @return repositoryPath Caminho para o repositrio de algoritmos.
   */
  public String getAlgorithmRepositoryPath() {
    return algorithmRepositoryPath;
  }

  /**
   * Solicita ao servico de algoritmos um algoritmo que possui determinada
   * identificacao.
   * 
   * @param name nome do algoritmo
   * 
   * @return o algoritmo que possui o identificador, ou null
   * 
   * @throws Exception se houver falha de comunicao.
   */
  public static AlgorithmInfo getAlgorithmInfo(String name) throws Exception {
    AlgorithmInfo algoInfo = null;
    if (name != null) {
      algoInfo = ClientRemoteLocator.algorithmService.getInfo(name);
    }
    return algoInfo;
  }

  /**
   * Solicita ao servico de algoritmos um algoritmo que possui determinada
   * identificacao.
   * 
   * @param id identificador do algoritmo
   * 
   * @return o algoritmo que possui o identificador, ou null
   * 
   * @throws Exception se houver falha de comunicao.
   */
  public static AlgorithmInfo getAlgorithmInfo(Object id) throws Exception {
    AlgorithmInfo algoInfo = null;
    if (id != null) {
      algoInfo = ClientRemoteLocator.algorithmService.getInfo(id);
    }
    return algoInfo;
  }

  /**
   * Esse mtodo  chamado quando o servio de algoritmos sofre alguma alterao
   * relativa a do algo.
   * 
   * @param event o evento que ocorreu no servio de algoritmo
   * 
   * @throws Exception Em caso de erro.
   */
  public static void update(AlgoEvent event) throws Exception {
    observable.notifyObservers(event);
  }

  /**
   * (non-Javadoc)
   * 
   * @see java.lang.Comparable#compareTo(java.lang.Object) *
   * @param o informao de algoritmo a ser comparado
   * @return o resultado da comparao entre os nomes dos algoritmos
   */
  @Override
  public int compareTo(AlgorithmInfo o) {
    return this.name.compareToIgnoreCase(o.name);
  }

  /**
   * (non-Javadoc)
   * 
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object object) {
    if (object == null) {
      return false;
    }
    if (!(object instanceof AlgorithmInfo)) {
      return false;
    }
    AlgorithmInfo algorithmInfo = (AlgorithmInfo) object;
    return algorithmInfo.getId().equals(getId());
  }

  /**
   * Obtm o nome do diretrio do algoritmo.
   * 
   * @return a diretrio desejado.
   */
  public String getDirectory() {
    return this.dir;
  }

  /**
   * Obtm o identificador.
   * 
   * @return a identificao
   */
  public String getId() {
    return this.id;
  }

  /**
   * Obtm a ltima verso deste algoritmo.
   * 
   * @return A ltima verso deste algoritmo ou {@code null} se no houver
   *         verses.
   */
  public AlgorithmVersionInfo getLastVersion() {
    if (this.versions.isEmpty()) {
      return null;
    }
    Iterator<AlgorithmVersionInfo> versionIterator =
      this.versions.values().iterator();
    AlgorithmVersionInfo lastVersion = versionIterator.next();
    while (versionIterator.hasNext()) {
      AlgorithmVersionInfo currentVersion = versionIterator.next();
      if (currentVersion.compareTo(lastVersion) < 0) {
        lastVersion = currentVersion;
      }
    }
    return lastVersion;
  }

  /**
   * Obtm o nome do algoritmo.
   * 
   * @return nome do algoritmo
   */
  public String getName() {
    return this.name;
  }

  /**
   * Obtm um resumo do algoritmo.
   * 
   * @return o resumo
   */
  public AlgorithmOutline getOutline() {
    return new AlgorithmOutline(this.id, this.name);
  }

  /**
   * Obteno dos identificadores de verso do algoritmo.
   * 
   * @return as verses do algoritmo.
   */
  public Object[] getVersionIds() {
    if (this.versions.size() == 0) {
      return null;
    }
    Object[] allInfo = this.versions.values().toArray();
    Object[] send = new Object[allInfo.length];
    for (int i = 0; i < allInfo.length; i++) {
      send[i] = ((AlgorithmVersionInfo) allInfo[i]).getId();
    }
    return send;
  }

  /**
   * Obtem as informaes de uma verso.
   * 
   * @param versionId .
   * 
   * @return .
   */
  public AlgorithmVersionInfo getVersionInfo(Object versionId) {
    if (versionId == null) {
      return null;
    }
    return this.versions.get(versionId);
  }

  /**
   * Consulta ao nome do sub-pacote de verses.
   * 
   * @return retorna o nome do sub-pacote como uma string.
   */
  public String getVersionsDirName() {
    return VERSIONS_DIR;
  }

  /**
   * Obtm as verses deste algoritmo.
   * 
   * @return Um vetor (Vector) com as verses deste algoritmo.
   */
  public Vector<AlgorithmVersionInfo> getVersions() {
    return new Vector<AlgorithmVersionInfo>(this.versions.values());
  }

  /**
   * (non-Javadoc)
   * 
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return this.id.hashCode();
  }

  /**
   * (non-Javadoc)
   * 
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    return this.name;
  }

  /**
   * Remove uma verso
   * 
   * @param versionId .
   */
  protected void deleteVersion(Object versionId) {
    this.versions.remove(versionId);
  }

  /**
   * Atribui um novo nome ao algoritmo.
   *
   * @param newName o novo nome.
   */
  protected void rename(String newName) {
    this.name = newName;
    for (AlgorithmVersionInfo versionInfo : versions.values()) {
     versionInfo.setInfo(new AlgorithmInfo(this));
    }
  }


  /**
   * Obtem as informaes de todas as verses de um algoritmo.
   * 
   * @return uma Hashtable com as informaes, onde a chave e' o identificador
   *         da verso
   */
  protected Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> getAllVersionInfo() {
    return this.versions;
  }

  /**
   * Inclui uma verso
   * 
   * @param versionId .
   * @param vInfo .
   */
  protected void includeVersion(AlgorithmVersionId versionId,
    AlgorithmVersionInfo vInfo) {
    this.versions.put(versionId, vInfo);
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        AlgorithmInfo.observable.deleteObservers();
      }
    });
  }

  /**
   * Retorna o <b>valor</b> associado a um determinado atributo estendido, como
   * por exemplo, o nome do analista responsavel.
   * 
   * @param propertyName .
   * 
   * @return valor no atributo estendido
   */
  public String getPropertyValue(String propertyName) {
    String result = this.propertyValues.get(propertyName);
    if (result == null || result.length() == 0) {
      return null;
    }
    return result;
  }

  /**
   * Retorna uma lista com os valores dos atributos estendidos do algoritmo
   * 
   * @return vector com os valores dos atributos estendidos do algoritmo
   */
  public Hashtable<String, String> getPropertyValues() {
    return this.propertyValues;
  }

  /**
   * Modifica o valor dos atributos estendidos. O mtodo garante a persistencia
   * da modificacao.
   * 
   * @param atributeValues vector com os novos valores dos atributos
   */
  public void setPropertyValues(Hashtable<String, String> atributeValues) {
    /* Seta em memoria a lista de atributos */
    this.propertyValues = atributeValues;
  }

  /**
   * Obtm o valor da propriedade que indica o usurio (login) que criou esse
   * algoritmo.
   * 
   * @return retorna o login do usurio que criou esse algoritmo
   */
  public String getOwner() {
    return getPropertyValue(AlgorithmInfo.OWNER_ALGORITHM_PROPERTY);
  }

  /**
   * Atribui uma descrio para o algoritmo.
   * 
   * @param description descrio do algoritmo
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Obtm a descrio do algoritmo.
   * 
   * @return a descrio do algoritmo
   */
  public String getDescription() {
    return description;
  }

  /**
   * Estabelece os nomes completos das categorias em que o algoritmo do Pacote
   * de Algoritmos faz parte.
   * 
   * No servio de algoritmos, esses nomes completos das categorias devem ser
   * analisados, e ento criadas as respectivas categorias e sub-categorias em
   * que o algorimto participa.
   * 
   * @param categoryFullNames nomes completos das categorias
   */
  public void setAlgoPackCategoryFullNames(List<String> categoryFullNames) {
    this.categoryFullNames = categoryFullNames;
  }

  /**
   * Retorna os nomes completos das categorias em que o algoritmo do Pacote de
   * Algoritmos faz parte.
   * 
   * @return os nomes completos das categorias em que o algoritmo do PA faz
   *         parte
   */
  public List<String> getAlgoPackCategoryFullNames() {
    return categoryFullNames;
  }

}
