/*
 * AlgoServiceInterface.java
 * 
 * $Author$ $Revision$ - $Date: 2006-10-30 18:40:34 -0300
 * (Mon, 30 Oct 2006) $
 */
package csbase.remote;

import java.rmi.RemoteException;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import csbase.exception.PermissionException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.CategoriesFileNotSavedException;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmOutline;
import csbase.logic.algorithms.AlgorithmProperty;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.AlgorithmsPack;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.HistoryRecord;
import csbase.logic.algorithms.ImportAlgorithmsPackTransferInfo;
import csbase.logic.algorithms.PAImportOperation;
import csbase.logic.algorithms.parsers.ParameterFactory;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Interface remota para acesso ao Servio de Gerncia de Algoritmos
 */
public interface AlgorithmServiceInterface extends TransactionInterface,
  ServiceInterface, RemoteObservable {
  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "AlgorithmService";

  /**
   * Altera as propriedades de um algoritmo.
   * 
   * @param name Nome do algoritmo
   * @param newValues Mapa das propriedades
   * 
   * @return As informaes alteradas do algoritmo
   * 
   * @throws RemoteException Erro de RMI.
   * @throws AlgorithmNotFoundException caso o algoritmo no seja encontrado
   */
  public AlgorithmInfo changeAlgorithmProperties(String name,
    Hashtable<String, String> newValues) throws RemoteException,
    AlgorithmNotFoundException;

  /**
   * Altera as propriedades de uma verso de um algoritmo.
   * 
   * @param name Nome do algoritmo.
   * @param versionId Identificador da verso
   * @param newValues Mapa das propriedades
   * 
   * @return As informaes alteradas do algoritmo
   * 
   * @throws RemoteException Erro de RMI.
   * @throws AlgorithmNotFoundException caso o algoritmo ou no seja encontrado
   */
  public AlgorithmInfo changeVersionProperties(String name,
    AlgorithmVersionId versionId, Hashtable<String, String> newValues)
    throws RemoteException, AlgorithmNotFoundException;

  /**
   * Copia arquivos de uma pasta para a outra com a possibilidade de dar
   * permisso de execuo s cpias.
   * 
   * @param sourceVersion Informaes da verso de origem.
   * @param files Arquivos a serem transferidos.
   * @param sourcePathFactory Fbrica para criar o caminho dos arquivos a serem
   *        transferidos.
   * @param targetVersion Informaes da verso de destino.
   * @param targetDir diretrio de destino.
   * @param targetPathFactory Fbrica para criar o caminho dos arquivos de
   *        destino. Este caminho  relativo ao valor do parmetro targetDir.
   * @param setExecutables <tt>true</tt> se as cpias devero ter permisso de
   *        execuo ou no.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public void copyFiles(final AlgorithmVersionInfo sourceVersion,
    final List<FileInfo> files, final IPathFactory sourcePathFactory,
    final AlgorithmVersionInfo targetVersion, final FileInfo targetDir,
    final IPathFactory targetPathFactory, final boolean setExecutables)
    throws RemoteException;

  /**
   * Cria uma categoria de algoritmos como filha de uma categoria ou do primeiro
   * nvel. Nesse caso, deve-se passar null para o identificador da categoria
   * pai.
   * 
   * @param parentCategoryId identificador da categoria pai. Se for null, ento
   *        a categoria ser criada na raiz, caso contrrio, como sub-categoria
   * @param name nome da categoria
   * 
   * @return retorna a categoria criada, ou null, caso a categoria no tenha
   *         sido encontrada
   * @throws RemoteException erro de RMI
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public Category createCategory(String parentCategoryId, String name)
    throws RemoteException, CategoriesFileNotSavedException;

  /**
   * Remove uma categoria de algoritmos, a partir do seu identificador nico.
   * 
   * A categoria a ser removida pode estar na raiz de categorias ou ser uma
   * sub-categoria.
   * 
   * @param id identificador da categoria a ser removida
   * 
   * @return retorna a categoria removida, ou null, caso a categoria no tenha
   *         sido encontrada
   * 
   * @throws RemoteException erro de RMI
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public Category removeCategory(String id) throws RemoteException,
    CategoriesFileNotSavedException;

  /**
   * Associa um conjunto de algoritmos a um conjunto de categorias.
   * 
   * Nesse caso, para cada categoria da lista de categorias sero adicionados
   * todos os algoritmos da lista de algoritmos. Essa informao ser persistida
   * em xml.
   * 
   * @param categoryIds lista de identificadores das categorias
   * @param algoIds lista de identificadores dos algoritmos
   * 
   * @return retorna o conjunto de categorias modificadas
   * 
   * @throws RemoteException erro de RMI
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public CategorySet bindAlgorithmsToCategories(List<Object> algoIds,
    List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException;

  /**
   * Desassocia um conjunto de algoritmos de um conjunto de categorias.
   * 
   * Nesse caso, para cada categoria da lista de categorias sero removidos
   * todos os algoritmos da lista de algoritmos. Essa informao ser persistida
   * em xml.
   * 
   * @param categoryIds lista de identificadores das categorias
   * @param algoIds lista de identificadores dos algoritmos
   * 
   * @return retorna o conjunto de categorias modificadas
   * 
   * @throws RemoteException erro de RMI
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public CategorySet unbindAlgorithmsFromCategories(List<Object> algoIds,
    List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException;

  /**
   * Atribui um conjunto de algoritmos de um conjunto de categorias,
   * substituindo o conjunto anterior.
   * 
   * @param categoryIds lista de identificadores das categorias
   * @param algoIds lista de identificadores dos algoritmos
   * 
   * @return retorna o conjunto de categorias modificadas
   * 
   * @throws RemoteException erro de RMI
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public CategorySet setAlgorithmsToCategories(List<Object> algoIds,
    List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException;

  /**
   * Cria um algoritmo a partir do nome, do identificador e das propriedades
   * informadas.
   * 
   * @param name Nome do algoritmo
   * @param id Identificador do algoritmo
   * @param properties Propriedades do algoritmo
   * 
   * @return As informaes do algoritmo criado
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo createAlgorithm(String name, String id,
    Hashtable<String, String> properties) throws RemoteException;

  /**
   * Renomeia o algoritmo.
   *
   * @param id Identificador do algoritmo
   * @param name novo nome do algoritmo
   *
   * @return As informaes do algoritmo atualizadas
   *
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo renameAlgorithm(Object id, String name)
    throws RemoteException;

  /**
   * Cria um configurador para uma verso especfica de um algoritmo.
   * 
   * @param algorithmName .
   * @param algorithmVersionId .
   * 
   * @return o configurador ou null caso o algoritmo ou verso no estejam
   *         cadastrados
   * 
   * @throws RemoteException Erro de RMI.
   * @throws AlgorithmNotFoundException quando o algoritmo ou verso no so
   *         encontrados
   */
  public AlgorithmConfigurator createAlgorithmConfigurator(String algorithmName,
    AlgorithmVersionId algorithmVersionId) throws RemoteException,
    AlgorithmNotFoundException;

  /**
   * Cria no repositrio a estrutura para uma nova verso de algoritmo.
   * 
   * @param algoId identificador do algoritmo
   * @param major numero da verso
   * @param minor numero da reviso
   * @param patch numero da correo
   * @param properties Mapa com as propriedades da verso.
   * 
   * @return informaes atualizadas do algoritmo
   * 
   * @throws PermissionException caso o usurio no tenha permisso para acessar
   *         o servio de algoritmos.
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo createVersion(Object algoId, int major, int minor,
    int patch, Hashtable<String, String> properties) throws RemoteException;

  /**
   * Cria uma nova verso do algoritmo com o mesmos dados que a verso dada. Ou
   * seja, cria a verso, copia os arquivos de documentao e configurao,
   * insere as plataformas e copia os executveis.
   * 
   * @param algoId id do algoritmo
   * @param versionIdFrom id da verso de origem
   * @param majorTo nmero da verso
   * @param minorTo nmero da reviso
   * @param patchTo nmero da correo
   * 
   * @throws RemoteException Erro de RMI.
   */
  public void duplicateVersion(Object algoId, Object versionIdFrom, int majorTo,
    int minorTo, int patchTo) throws RemoteException;

  /**
   * Obtm a Retorna a lista de <code>Property</code> dos algoritmos. Cada
   * <code>Property</code> mapeia a chave da propriedade ao Label (String de
   * apresentao).
   * 
   * @return Lista com as propriedades dos algoritmos. Caso no haja
   *         propriedades, retorna uma lista vazia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public List<AlgorithmProperty> getAlgorithmProperties()
    throws RemoteException;

  /**
   * Retorna de todos os identificadores dos algoritmos cadastrados.
   * 
   * @return um array de strings com os identificadores.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public Object[] getAllIds() throws RemoteException;

  /**
   * Retorna todos os dados de todos os algoritmos cadastrados.
   * 
   * @return um array com as informaes de todos os algoritmos cadastrados
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo[] getAllInfo() throws RemoteException;

  /**
   * Retorna todos os resumos dos algoritmos cadastrados.
   * 
   * @return um array com os resumos ou <code>null</code> se no houver
   *         algoritmos instalados.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmOutline[] getAllOutlines() throws RemoteException;

  /**
   * Retorna as fbricas dos parmetros disponveis no sistema.
   *
   * @return um mapa de fbricas de parmetros indexado pela tag xml
   *         correspondente.
   *
   * @throws RemoteException Erro de RMI.
   */
  public Map<String, ParameterFactory> getParameterFactories()
    throws RemoteException;

  /**
   * Retorna informaes de um algoritmo.
   * 
   * @param id identificador do algoritmo
   * 
   * @return as informaes solicitadas
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo getInfo(Object id) throws RemoteException;

  /**
   * Retorna informaes de um algoritmo a partir do nome
   * 
   * @param name O nome do algoritmo
   * 
   * @return as informaes solicitadas
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo getInfo(String name) throws RemoteException;

  /**
   * Obtm o conjunto de categorias disponveis no servidor.
   * 
   * @return o conjunto de categorias
   * 
   * @throws RemoteException Erro de RMI.
   */
  public CategorySet getAllCategories() throws RemoteException;

  /**
   * Obtm uma categoria que possui o identificador especificado.
   * 
   * @param id identificador da categoria
   * 
   * @return a categoria encontrada, caso contrrio, retorna null
   * @throws RemoteException erro de rmi
   */
  public Category getCategory(String id) throws RemoteException;

  /**
   * Obtm a Retorna a lista de <code>Property</code> dos algoritmos. Cada
   * <code>Property</code> mapeia a chave da propriedade ao Label (String de
   * apresentao).
   * 
   * @return Lista com as propriedades das verses dos algoritmos. Caso no haja
   *         propriedades, retorna uma lista vazia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public List<AlgorithmProperty> getVersionProperties() throws RemoteException;

  /**
   * Inclui uma plataforma para uma verso de algoritmo.
   * 
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @param platform plataforma a incluir
   * 
   * @return informaes atualizadas do algoritmo
   * 
   * @throws PermissionException caso o usurio no tenha permisso para acessar
   *         o servio de algoritmos.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo includePlatform(Object algoId, Object versionId,
    String platform) throws RemoteException;

  /**
   * Pedido de recarga dos algoritmos.
   * 
   * @return true caso a recarga tenha sido efetuada com sucesso.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public boolean reloadAlgorithms() throws RemoteException;

  /**
   * Remove um algoritmo do repositrio.
   * 
   * @param id identificador do algoritmo a ser removido
   * 
   * @return <code>true</code> se a remoo teve sucesso, <code>false</code>
   *         caso contrrio
   * 
   * @throws PermissionException caso o usurio no tenha permisso para acessar
   *         o servio de algoritmos.
   * @throws RemoteException Erro de RMI.
   */
  public boolean removeAlgorithm(Object id) throws RemoteException;

  /**
   * Remove arquivos executveis de uma verso de algoritmo.
   * 
   * @param version Verso que detm os arquivos.
   * @param platform Plataforma que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * 
   * @throws RemoteException Erro de RMI
   */
  public void removeExecutableFiles(AlgorithmVersionInfo version,
    String platform, FileInfo[] files) throws RemoteException;

  /**
   * Remove arquivos de configurao de uma verso de algoritmo.
   * 
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * 
   * @throws RemoteException Erro de RMI
   */
  public void removeConfigurationFiles(AlgorithmVersionInfo version,
    FileInfo[] files) throws RemoteException;

  /**
   * Remove arquivos de documentao de uma verso de algoritmo.
   * 
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * 
   * @throws RemoteException Erro de RMI
   */
  public void removeDocumentationFiles(AlgorithmVersionInfo version,
    FileInfo[] files) throws RemoteException;

  /**
   * Remove arquivos de release notes de uma verso de algoritmo.
   * 
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * 
   * @throws RemoteException Erro de RMI
   */
  public void removeReleaseNotesFiles(AlgorithmVersionInfo version,
    FileInfo[] files) throws RemoteException;

  /**
   * Remove uma plataforma de uma verso de algoritmo.
   * 
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @param platform plataforma a remover
   * 
   * @return informaes atualizadas do algoritmo
   * 
   * @throws PermissionException caso o usurio no tenha permisso para acessar
   *         o servio de algoritmos.
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo removePlatform(Object algoId, Object versionId,
    String platform) throws RemoteException;

  /**
   * Remove uma verso de algoritmo do repositrio.
   * 
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * 
   * @return informaes atualizadas do algoritmo
   * 
   * @throws PermissionException caso o usurio no tenha permisso para acessar
   *         o servio de algoritmos.
   * @throws RemoteException Erro de RMI.
   */
  public AlgorithmInfo removeVersion(Object algoId, Object versionId)
    throws RemoteException;

  // #TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Obtm a URL para upload de um configurador.
   * 
   * @param algoId identificador do algoritmo
   * @param versionId Identificador da verso do algoritmo
   * 
   * @return url para upload
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveConfigUploadURL(Object algoId, Object versionId)
    throws RemoteException;

  /**
   * Obtm uma URL para permitir o upload de um arquivo de documentao, a
   * partir do disco rgido do usurio para o servidor.
   * 
   * @param algoId identificador do algoritmo que ir receber o arquivo.
   * @param versionId identificador da verso que ir receber o arquivo.
   * 
   * @return URL para upload de arquivo de documentao.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveDocUploadURL(Object algoId, Object versionId)
    throws RemoteException;

  /**
   * Obtm uma URL para permitir o upload de um arquivo de release notes, a
   * partir do disco rgido do usurio para o servidor.
   * 
   * @param algoId identificador do algoritmo que ir receber o arquivo.
   * @param versionId identificador da verso que ir receber o arquivo.
   * 
   * @return URL para upload de arquivo de release notes.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveReleaseNotesUploadURL(Object algoId, Object versionId)
    throws RemoteException;

  // #TODO } A remover quando todos sistemas CSBase forem assinados

  /**
   * Obtem a url para a visualizao de um arquivo do algoritmo.
   * 
   * @param algoId identificador do algoritmo ao qual o arquivo pertence.
   * @param filePath caminho para o arquivo a partir da raiz do algoritmo.
   * 
   * @return url para visualizao de um arquivo do algoritmo em um navegador.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveDownloadURL(Object algoId, String[] filePath)
    throws RemoteException;

  // #TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Obtm a URL para upload de um executvel.
   * 
   * @param algoId identificador do algoritmo
   * @param versionId Identificador da verso do algoritmo
   * @param platform Plataforma do algoritmo
   * 
   * @return url para upload
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveExecUploadURL(Object algoId, Object versionId,
    String platform) throws RemoteException;

  /**
   * Obtm a URL para upload de pacote de verso.
   * 
   * @param algoId identificador do algoritmo
   * 
   * @return url para upload
   * 
   * @throws RemoteException Erro de RMI.
   */
  public String retrieveVersionUploadURL(Object algoId) throws RemoteException;

  // #TODO } A remover quando todos sistemas CSBase forem assinados

  /**
   * Recupera o histrico do caminho especificado.
   * 
   * @param path identificador do caminho interno  verso.
   * 
   * @return <code>List</code> contendo instncias de <code>HistoryRecord</code>
   *         , representando o histrico do caminho especificado ou
   *         <code>null</code> se no for possvel recuperar o histrico.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public List<HistoryRecord> retrieveHistory(String[] path)
    throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um arquivo binrio, dentro do algoritmo especificado.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param platformName nome da plataforma.
   * @param fileName nome do arquivo de destino.
   * @param expandIfZip flag indicando que ao final do upload, caso o arquivo
   *        seja um zip o servidor deve descompactar ele.
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareUploadExecFile(Object algoId,
    Object versionId, String platformName, String fileName, boolean expandIfZip)
    throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um arquivo de configurao, dentro do algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de destino.
   * @param expandIfZip flag indicando que ao final do upload, caso o arquivo
   *        seja um zip o servidor deve descompactar ele.
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareUploadConfigFile(Object algoId,
    Object versionId, String fileName, boolean expandIfZip)
    throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um arquivo de documentao, dentro do algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de destino.
   * @param expandIfZip flag indicando que ao final do upload, caso o arquivo
   *        seja um zip o servidor deve descompactar ele.
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareUploadDocFile(Object algoId,
    Object versionId, String fileName, boolean expandIfZip)
    throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um arquivo de release notes, dentro do algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de destino.
   * @param expandIfZip flag indicando que ao final do upload, caso o arquivo
   *        seja um zip o servidor deve descompactar ele.
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareUploadReleaseNotesFile(Object algoId,
    Object versionId, String fileName, boolean expandIfZip)
    throws RemoteException;

  /**
   * <p>
   * Prepara o download de um arquivo binrio, pertencente ao algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o download  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam ler no sistema de arquivos do servidor. Dessa forma, o cliente 
   * obrigado primeiro a passar por este mtodo, autenticar-se, obter uma chave
   * que guarda as informaes necessrias para a transferncia (sem as quais
   * no consegue transmitir), para ento repass-la via socket. De posse da
   * chave, o servidor j sabe que o cliente est autenticado e pode ento
   * prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param platformName nome da plataforma.
   * @param fileName nome do arquivo de origem.
   * 
   * @return objeto contendo as informaes necessrias para solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareDownloadExecFile(Object algoId,
    Object versionId, String platformName, String fileName)
    throws RemoteException;

  /**
   * <p>
   * Prepara o download de um arquivo de configurao, pertencente ao algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o download  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam ler no sistema de arquivos do servidor. Dessa forma, o cliente 
   * obrigado primeiro a passar por este mtodo, autenticar-se, obter uma chave
   * que guarda as informaes necessrias para a transferncia (sem as quais
   * no consegue transmitir), para ento repass-la via socket. De posse da
   * chave, o servidor j sabe que o cliente est autenticado e pode ento
   * prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de origem.
   * 
   * @return objeto contendo as informaes necessrias para solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareDownloadConfigFile(Object algoId,
    Object versionId, String fileName) throws RemoteException;

  /**
   * <p>
   * Prepara o download de um arquivo de documentao, pertencente ao algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o download  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam ler no sistema de arquivos do servidor. Dessa forma, o cliente 
   * obrigado primeiro a passar por este mtodo, autenticar-se, obter uma chave
   * que guarda as informaes necessrias para a transferncia (sem as quais
   * no consegue transmitir), para ento repass-la via socket. De posse da
   * chave, o servidor j sabe que o cliente est autenticado e pode ento
   * prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de origem.
   * 
   * @return objeto contendo as informaes necessrias para solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareDownloadDocFile(Object algoId,
    Object versionId, String fileName) throws RemoteException;

  /**
   * <p>
   * Prepara o download de um arquivo de release notes, pertencente ao algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o download  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam ler no sistema de arquivos do servidor. Dessa forma, o cliente 
   * obrigado primeiro a passar por este mtodo, autenticar-se, obter uma chave
   * que guarda as informaes necessrias para a transferncia (sem as quais
   * no consegue transmitir), para ento repass-la via socket. De posse da
   * chave, o servidor j sabe que o cliente est autenticado e pode ento
   * prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de origem.
   * 
   * @return objeto contendo as informaes necessrias para solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareDownloadNotesFile(Object algoId,
    Object versionId, String fileName) throws RemoteException;

  /**
   * Verifica se um determinado arquivo de configurao existe no servidor.
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de configurao.
   * 
   * @return <code>true</code> se o arquivo j existir no servidor.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public boolean configFileExists(Object algoId, Object versionId,
    String fileName) throws RemoteException;

  /**
   * Verifica se um determinado arquivo de documentao existe no servidor.
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de documentao.
   * 
   * @return <code>true</code> se o arquivo j existir no servidor.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public boolean docFileExists(Object algoId, Object versionId, String fileName)
    throws RemoteException;

  /**
   * Verifica se um determinado arquivo de release notes existe no servidor.
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo de documentao.
   * 
   * @return <code>true</code> se o arquivo j existir no servidor.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public boolean releaseNotesFileExists(Object algoId, Object versionId,
    String fileName) throws RemoteException;

  /**
   * Verifica se um determinado arquivo executvel existe no servidor.
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param platformName nome da plataforma.
   * @param fileName nome do arquivo executvel.
   * 
   * @return <code>true</code> se o arquivo j existir no servidor.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public boolean execFileExists(Object algoId, Object versionId,
    String platformName, String fileName) throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um pacote de verso (arquivo zip), dentro do algoritmo
   * especificado.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param fileName nome do arquivo de destino.
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public RemoteFileChannelInfo prepareUploadVersionPack(Object algoId,
    String fileName) throws RemoteException;

  /**
   * <p>
   * Prepara o download de um arquivo de pacote de verso, pertencente ao
   * algoritmo especificado.
   * </p>
   * <p>
   * Como o download  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam ler no sistema de arquivos do servidor. Dessa forma, o cliente 
   * obrigado primeiro a passar por este mtodo, autenticar-se, obter uma chave
   * que guarda as informaes necessrias para a transferncia (sem as quais
   * no consegue transmitir), para ento repass-la via socket. De posse da
   * chave, o servidor j sabe que o cliente est autenticado e pode ento
   * prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param versionPackFileName nome do arquivo de pacote de verso origem
   * @return objeto contendo as informaes necessrias para solicitar a
   *         transferncia
   * @throws RemoteException RemoteException Erro de RMI
   */
  public RemoteFileChannelInfo prepareDownloadVersionPackFile(Object algoId,
    AlgorithmVersionId versionId, String versionPackFileName)
    throws RemoteException;

  /**
   * <p>
   * Prepara o upload de um pacote de algoritmos (arquivo zip), dentro do
   * repositrio de algoritmos.
   * </p>
   * <p>
   * Como o upload  feito por uma conexo entre sockets, o uso deste mtodo
   * preparatrio  necessrio para evitar que clientes no-autenticados
   * consigam escrever no sistema de arquivos do servidor. Dessa forma, o
   * cliente  obrigado primeiro a passar por este mtodo, autenticar-se, obter
   * uma chave que guarda as informaes necessrias para a transferncia (sem
   * as quais no consegue transmitir), para ento repass-la via socket. De
   * posse da chave, o servidor j sabe que o cliente est autenticado e pode
   * ento prosseguir com a transferncia.
   * </p>
   * <p>
   * O mtodo retorna um objeto contendo todas as informaes necessrias para
   * solicitar a transferncia, a ser realizada posteriormente.
   * </p>
   * 
   * @return objeto contendo as informaes necessrias para a solicitar a
   *         transferncia.
   * 
   * @throws RemoteException Erro de RMI.
   */
  public ImportAlgorithmsPackTransferInfo prepareImportAlgorithmsPack()
    throws RemoteException;

  /**
   * Obtm os algoritmos que fazem parte do Pacote de Algoritmos, sem ainda
   * terem passado por uma validao.
   * 
   * @param importDataToken identificador nico do dado (PA) a ser importado
   * 
   * @return o pacote de algoritmos sem validao dos algoritmos
   * 
   * @throws RemoteException erro de rmi.
   * @throws PermissionException caso o usurio que solicite a consulta no seja
   *         o mesmo que requisitou a preparao da importao.
   */
  public AlgorithmsPack getAlgorithmsPackInfo(String importDataToken)
    throws RemoteException;

  /**
   * Realiza a importao do Pacote de Algoritmos, de acordo com as operaes
   * especificadas.
   * 
   * Essas operaes vo determinar o critrio para resoluo de conflitos a ser
   * adotado, ou seja, se vai substituir ou manter um algoritmo que j exista no
   * repositrio, se vai manter uma verso especfica, se uma nova categoria
   * deve ser criada, e assim por diante.
   * 
   * @param importDataToken identificador nico do dado (PA) a ser importado
   * @param operations operaes a serem realizadas na importao do PA
   * 
   * @return retorna true, se a importao foi realizada com sucesso, caso
   *         contrrio, retorna false
   * 
   * @throws RemoteException erro de rmi.
   * @throws PermissionException caso o usurio que solicite a consulta no seja
   *         o mesmo que requisitou a preparao da importao.
   */
  public boolean importAlgorithmsPack(String importDataToken,
    PAImportOperation... operations) throws RemoteException;

  /**
   * Finaliza o processo de importao do Pacote de Algoritmos identificado pelo
   * token, removendo os diretrios e arquivos criados temporariamente.
   * 
   * @param importDataToken identificador nico do dado (PA) a ser importado
   * 
   * @throws RemoteException erro de rmi.
   * @throws PermissionException caso o usurio que solicite a consulta no seja
   *         o mesmo que requisitou a preparao da importao.
   */
  public void finishImportAlgorithmsPack(String importDataToken)
    throws RemoteException;

  /**
   * Valida o pacote de algoritmos identificado pelo token antes da sua
   * importao.
   * 
   * @param importDataToken identificador nico do dado (PA) a ser importado
   * @return retorna um texto com a informao sobre os algoritmos vlidos e
   *         sobre os invlidos
   * @throws RemoteException erro de rmi.
   */
  public String validateAlgorithmsPack(String importDataToken)
    throws RemoteException;
}
