/**
 * $Id$
 */
package csbase.logic;

import java.io.Serializable;

import tecgraf.javautils.core.io.FileUtils;

/**
 * Modela os dados de um arquivo que participa da sincronizao entre dois
 * projetos.
 */
public class ProjectSynchronizationFileInfo implements Serializable {
  private ClientProjectFileInfo clientInfo;
  private String[] originalPath;
  private boolean transfer;
  private boolean newFile;

  /**
   * Constri um objeto de dados de um arquivo da sincronizao entre projetos
   * contendo as informaes do arquivo (como obtidas do projeto). O parmetro
   * de transferncia quando <code>false</code>, indica que o arquivo no deve
   * ser copiado. Caso seja <code>true</code>, indica que o arquivo deve ser
   * copiado para o outro servidor. O Parmetro <code>newFile</code> indica se o
   * arquivo  o mais recente. O path deve ser o caminho absoluto do arquivo no
   * projeto, sem incluir o nome do projeto.
   * 
   * @param clientInfo .
   * @param path .
   * @param transfer .
   */
  public ProjectSynchronizationFileInfo(ClientProjectFileInfo clientInfo,
    String[] path, boolean transfer) {
    this.clientInfo = clientInfo;
    this.originalPath = path;
    this.transfer = transfer;
    this.newFile = false;
  }

  /**
   * Retorna as informaes sobre o arquivo.
   * 
   * @return Informaes sobre o arquivo.
   */
  public ClientProjectFileInfo getClientFileInfo() {
    return clientInfo;
  }

  /**
   * Retorna o indicativo de cpia do arquivo para o outro servidor.
   * 
   * @return indicativo de cpia.
   */
  public boolean getTransfer() {
    return transfer;
  }

  /**
   * Altera o indicativo de cpia do arquivo para o outro servidor.
   * 
   * @param transfer novo indicativo de cpia.
   */
  public void setTransfer(boolean transfer) {
    this.transfer = transfer;
  }

  /**
   * Retorna o nome do arquivo.
   * 
   * @return o nome do arquivo.
   */
  public final String getFilename() {
    return originalPath[originalPath.length - 1];
  }

  /**
   * Retorna o caminho absoluto do diretrio em que se encontra o arquivo.
   * 
   * @return o caminho absoluto do diretrio em que se encontra o arquivo.
   */
  public final String[] getAbsoluteDirPath() {
    String[] absolutePath = new String[originalPath.length - 1];
    System.arraycopy(originalPath, 0, absolutePath, 0, absolutePath.length);

    return absolutePath;
  }

  /**
   * Retorna o caminho absoluto do arquivo.
   * 
   * @return o caminho absoluto do arquivo.
   */
  public final String[] getAbsoluteFilename() {
    return originalPath.clone();
  }

  /**
   * Gera uma String com o caminho absoluto do arquivo sem incluir o nome do
   * projeto. O caracter separador  '/'.
   * 
   * @return o caminho absoluto do arquivo.
   */
  public String generateAbsoluteFilename() {
    return FileUtils.joinPath('/', originalPath);
  }

  /**
   * Indica se o arquivo  mais recente, logo, deveria ser transferido.
   * 
   * @return boolean
   */
  public boolean isNewFile() {
    return newFile;
  }

  /**
   * Seta se o arquivo  o mais recente para indicar que deve ser transferido.
   * 
   * @param newFile .
   */
  public void setNewFile(boolean newFile) {
    this.newFile = newFile;
  }
}
