/*
 * $Id$
 */

package csbase.logic;

import java.io.IOException;
import java.io.OutputStream;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Implementa um escritor (stream) de um arquivo (client-side).
 * 
 * @author Tecgraf/PUC-Rio
 */
public class RemoteFileOutputStream extends OutputStream {
  /**
   * Identificador do arquivo remoto.
   */
  private byte[] id = null;

  /**
   * Canal de acesso ao arquivo a ler lido.
   */
  private SyncRemoteFileChannel channel = null;

  /**
   * Posio corrente no arquivo
   */
  private long position = 0;

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    if (channel == null) {
      return;
    }
    try {
      channel.setSize(position);
      channel.close();
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
    id = null;
    channel = null;
  }

  /**
   * {@inheritDoc}
   * 
   * No faz nada.
   */
  @Override
  public void flush() throws IOException {
    // vazio
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(int b) throws IOException {
    byte[] src = new byte[1];
    src[0] = (byte) (b & 0xFF);
    write(src);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] b) throws IOException {
    write(b, 0, b.length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] b, int off, int len)
    throws IOException {
    try {
      channel.syncWrite(b, off, len, position);
      position += len;
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Cria um <code>OutputStream</code> a partir de um canal de acesso a um
   * arquivo remoto.
   * 
   * @param info Informaes para conexo do canal.
   * 
   * @throws IOException Em caso de falha na operao.
   */
  public RemoteFileOutputStream(RemoteFileChannelInfo info) throws IOException {
    this.id = info.getIdentifier();
    this.channel = new SyncRemoteFileChannel(info);
    try {
      channel.open(true);
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Cria um <code>OutputStream</code> a partir de um canal de acesso a um
   * arquivo remoto.
   * 
   * @param id Indentificador do arquivo remoto.
   * @param channel Canal de acesso ao arquivo remoto. Deve ter sido aberto
   *        antes de ser passado a esse construtor.
   */
  public RemoteFileOutputStream(byte[] id, SyncRemoteFileChannel channel) {
    this.id = id;
    this.channel = channel;
  }

  /**
   * {@inheritDoc}
   */
  public String toString() {
    return "RemoteFileOutputStream para " + new String(id);
  }
}
