/**
 * $Id$
 */
package csbase.logic.algorithms.parameters;

import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.Validation;

/**
 * Grupo de parmetros.
 *
 * @author lmoreira
 */
public final class ParameterGroup extends Parameter<Object> {
  /**
   * Os grupos filhos deste grupo.
   */
  private List<ParameterGroup> groups;
  /**
   * Os parmetros do tipo lista de arquivos de entrada que so filhos deste
   * grupo.
   */
  private List<InputFileListParameter> inputFileListParameters;
  /**
   * Os parmetros do tipo arquivos de entrada que so filhos deste grupo.
   */
  private List<InputFileParameter> inputParameters;
  /**
   * Os parmetros do tipo lista de arquivos de sada que so filhos deste
   * grupo.
   */
  private List<OutputFileListParameter> outputFileListParameters;
  /**
   * Os parmetros do arquivos de sada que so filhos deste grupo.
   */
  private List<OutputFileParameter> outputParameters;

  /**
   * Os parmetros do tipo lista de URLs de entrada que so filhos deste grupo.
   */
  private List<InputURLListParameter> inputURLListParameters;
  /**
   * Os parmetros do tipo URLs de entrada que so filhos deste grupo.
   */
  private List<InputURLParameter> inputURLParameters;
  /**
   * Os parmetros do tipo lista de URLs de sada que so filhos deste grupo.
   */
  private List<OutputURLListParameter> outputURLListParameters;
  /**
   * Os parmetros do URLs de sada que so filhos deste grupo.
   */
  private List<OutputURLParameter> outputURLParameters;

  /**
   * Os parmetros filhos deste grupo.
   */
  private List<Parameter<?>> parameters;
  /**
   * Carregador de parmetros.
   */
  private ParameterLoader parameterLoader;
  /**
   * Os parmetros simples filhos deste grupo indexados pelo nome.
   */
  private Map<String, SimpleParameter<?>> simpleParametersByName;
  /**
   * Rtulo.
   */
  private String label;
  /**
   * Indica se o grupo  minimizvel (se o grupo pode ser "fechado" e "aberto"
   * na interface grfica).
   */
  private boolean isCollapsible;

  /**
   * Cria um grupo.
   *
   * @param id O identificador do grupo (No aceita {@code null}).
   * @param label rtulo do grupo.
   * @param isCollapsible determina se o grupo  minimizvel.
   */
  public ParameterGroup(String id, String label, boolean isCollapsible) {
    super(id);
    setLabel(label);
    setCollapsible(isCollapsible);
    this.groups = new LinkedList<ParameterGroup>();
    this.inputParameters = new LinkedList<InputFileParameter>();
    this.inputFileListParameters = new LinkedList<InputFileListParameter>();
    this.outputParameters = new LinkedList<OutputFileParameter>();
    this.outputFileListParameters = new LinkedList<OutputFileListParameter>();
    this.inputURLParameters = new LinkedList<InputURLParameter>();
    this.outputURLParameters = new LinkedList<OutputURLParameter>();
    this.parameters = new LinkedList<Parameter<?>>();
    this.simpleParametersByName = new HashMap<String, SimpleParameter<?>>();
    this.parameterLoader = null;
  }

  /**
   * Cria um grupo com rtulo idntico ao nome.
   *
   * @param id O identificador do grupo (No aceita {@code null}).
   */
  public ParameterGroup(String id) {
    this(id, id, false);
  }

  /**
   * Adiciona um parmetro filho.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se j houver um
   *         filho com o mesmo nome do parmetro informado.
   */
  public boolean addParameter(Parameter<?> parameter) {
    if (parameter == null) {
      throw new IllegalArgumentException("O parmetro parameter est nulo.");
    }
    if (this.parameters.contains(parameter)) {
      return false;
    }
    if (parameter instanceof SimpleParameter<?>) {
      if (parameter instanceof InputFileParameter) {
        this.inputParameters.add((InputFileParameter) parameter);
      }
      else if (parameter instanceof OutputFileParameter) {
        this.outputParameters.add((OutputFileParameter) parameter);
      }
      else if (parameter instanceof InputFileListParameter) {
        this.inputFileListParameters.add((InputFileListParameter) parameter);
      }
      else if (parameter instanceof OutputFileListParameter) {
        this.outputFileListParameters.add((OutputFileListParameter) parameter);
      }
      else if (parameter instanceof InputURLParameter) {
        this.inputURLParameters.add((InputURLParameter) parameter);
      }
      else if (parameter instanceof OutputURLParameter) {
        this.outputURLParameters.add((OutputURLParameter) parameter);
      }
      SimpleParameter<?> simpleParameter = (SimpleParameter<?>) parameter;
      this.simpleParametersByName.put(simpleParameter.getName(),
        simpleParameter);
    }
    else if (parameter instanceof ParameterGroup) {
      ParameterGroup group = (ParameterGroup) parameter;
      this.groups.add(group);
    }
    this.parameters.add(parameter);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> parameterValues = new HashMap<String, Object>();
    for (Parameter<?> parameter : this.parameters) {
      Map<String, Object> values = parameter.exportValue();
      for (String key : values.keySet()) {
        parameterValues.put(key, values.get(key));
      }
    }
    return Collections.unmodifiableMap(parameterValues);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandLine(CommandLineContext context) {
    String commandLine = "";
    String separator = "";
    for (Parameter<?> parameter : this.parameters) {
      String parameterCommandLine = parameter.getCommandLine(context);
      if (parameterCommandLine != null) {
        commandLine += separator;
        commandLine += parameter.getCommandLine(context);
        separator = " ";
      }
    }
    if (commandLine.length() == 0) {
      return null;
    }
    return commandLine;
  }

  /**
   * Obtm os grupos-filhos deste grupo.
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return Os grupos-filhos (Se no houver grupos-filhos, o conjunto estar
   *         vazio).
   */
  public Set<ParameterGroup> getGroups() {
    Set<ParameterGroup> childGroups = new HashSet<ParameterGroup>();
    childGroups.addAll(this.groups);
    for (ParameterGroup group : childGroups) {
      childGroups.addAll(group.getGroups());
    }
    return childGroups;
  }

  /**
   * Obtm os parmetros do tipo lista de arquivo de entrada que so filhos
   * deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<InputFileListParameter> getInputFileListParameters() {
    List<InputFileListParameter> params =
      new LinkedList<InputFileListParameter>();
    params.addAll(this.inputFileListParameters);
    for (ParameterGroup group : getGroups()) {
      params.addAll(group.getInputFileListParameters());
    }
    return Collections.unmodifiableList(params);
  }

  /**
   * Obtm os parmetros do tipo lista de arquivo de sada que so filhos deste
   * grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<OutputFileListParameter> getOutputFileListParameters() {
    List<OutputFileListParameter> params =
      new LinkedList<OutputFileListParameter>();
    params.addAll(this.outputFileListParameters);
    for (ParameterGroup group : getGroups()) {
      params.addAll(group.getOutputFileListParameters());
    }
    return Collections.unmodifiableList(params);
  }

  /**
   * Obtm os parmetros do tipo arquivo de entrada que so filhos deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<InputFileParameter> getInputFileParameters() {
    List<InputFileParameter> inputFileParameters =
      new LinkedList<InputFileParameter>();
    inputFileParameters.addAll(this.inputParameters);
    for (ParameterGroup group : getGroups()) {
      inputFileParameters.addAll(group.getInputFileParameters());
    }
    return Collections.unmodifiableList(inputFileParameters);
  }

  /**
   * Obtm os parmetros do tipo arquivo de sada que so filhos deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<OutputFileParameter> getOutputFileParameters() {
    List<OutputFileParameter> outputFileParameters =
      new LinkedList<OutputFileParameter>();
    outputFileParameters.addAll(this.outputParameters);
    for (ParameterGroup group : getGroups()) {
      outputFileParameters.addAll(group.getOutputFileParameters());
    }
    return Collections.unmodifiableList(outputFileParameters);
  }

  /**
   * Obtm os parmetros do tipo lista de URL de entrada que so filhos deste
   * grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<InputURLListParameter> getInputURLListParameters() {
    List<InputURLListParameter> params =
      new LinkedList<InputURLListParameter>();
    params.addAll(this.inputURLListParameters);
    for (ParameterGroup group : getGroups()) {
      params.addAll(group.getInputURLListParameters());
    }
    return Collections.unmodifiableList(params);
  }

  /**
   * Obtm os parmetros do tipo lista de URLs de sada que so filhos deste
   * grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<OutputURLListParameter> getOutputURLListParameters() {
    List<OutputURLListParameter> params =
      new LinkedList<OutputURLListParameter>();
    params.addAll(this.outputURLListParameters);
    for (ParameterGroup group : getGroups()) {
      params.addAll(group.getOutputURLListParameters());
    }
    return Collections.unmodifiableList(params);
  }

  /**
   * Obtm os parmetros do tipo URL de entrada que so filhos deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<InputURLParameter> getInputURLParameters() {
    List<InputURLParameter> inputURLParameters =
      new LinkedList<InputURLParameter>();
    inputURLParameters.addAll(this.inputURLParameters);
    for (ParameterGroup group : getGroups()) {
      inputURLParameters.addAll(group.getInputURLParameters());
    }
    return Collections.unmodifiableList(inputURLParameters);
  }

  /**
   * Obtm os parmetros do tipo URL de sada que so filhos deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<OutputURLParameter> getOutputURLParameters() {
    List<OutputURLParameter> outputURLParameters =
      new LinkedList<OutputURLParameter>();
    outputURLParameters.addAll(this.outputURLParameters);
    for (ParameterGroup group : getGroups()) {
      outputURLParameters.addAll(group.getOutputURLParameters());
    }
    return Collections.unmodifiableList(outputURLParameters);
  }

  /**
   * Obtm o carregador de parmtros.
   *
   * @return o carregador de parmtros, ou {@code null} caso no exista.
   */
  public ParameterLoader getParameterLoader() {
    return parameterLoader;
  }

  /**
   * Obtm os parmetros que so filhos deste grupo.
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public List<Parameter<?>> getParameters() {
    return Collections.unmodifiableList(this.parameters);
  }

  /**
   * Obtm um parmetro simples que  filho deste grupo atravs de seu nome.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O parmetro-filho ou {@code null} se ele no existir.
   */
  public SimpleParameter<?> getSimpleParameter(String parameterName) {
    SimpleParameter<?> parameter = this.simpleParametersByName.get(
      parameterName);
    if (parameter == null) {
      for (ParameterGroup group : getGroups()) {
        parameter = group.getSimpleParameter(parameterName);
      }
    }
    return parameter;
  }

  /**
   * Obtm os parmetros simples que so filhos deste grupo.
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return Os parmetros-filhos (Se no houver grupos-filhos, a lista estar
   *         vazia).
   */
  public Set<SimpleParameter<?>> getSimpleParameters() {
    Set<SimpleParameter<?>> simpleParameters =
      new HashSet<SimpleParameter<?>>();
    simpleParameters.addAll(this.simpleParametersByName.values());
    for (ParameterGroup group : getGroups()) {
      simpleParameters.addAll(group.getSimpleParameters());
    }
    return Collections.unmodifiableSet(simpleParameters);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings({ "rawtypes", "unchecked" })
  @Override
  public void importValue(Map parameterValue) {
    if (parameterValue == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterValues est nulo.");
    }
    for (Parameter<?> parameter : this.parameters) {
      parameter.importValue(parameterValue);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSetDefaultValue() {
    for (Parameter<?> parameter : getParameters()) {
      if (!parameter.isSetDefaultValue()) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isVisible() {
    for (Parameter<?> parameter : this.parameters) {
      if (parameter.isVisible()) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetValue() {
    for (Parameter<?> parameter : this.parameters) {
      parameter.resetValue();
    }
  }

  /**
   * Atualiza o carregador de parmtros.
   *
   * @param parameterLoader carregador de parmtros. No aceita {@code null}.
   * @return {@code true} se no havia sido definido anteriormente nenhum
   *         carregador de parmetros.
   */
  public boolean setParameterLoader(ParameterLoader parameterLoader) {
    if (this.parameterLoader != null) {
      return false;
    }
    if (parameterLoader == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterLoader est nulo.");
    }
    this.parameterLoader = parameterLoader;
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validate(ValidationContext context)
    throws RemoteException {

    Validation result = new Validation();
    for (Parameter<?> parameter : parameters) {
      result.addChild(parameter.validate(context));
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setVisible(boolean isVisible) {
    boolean changedVisibility = false;
    for (Parameter<?> parameter : this.parameters) {
      if (isVisible != parameter.isVisible()) {
        if (parameter.setVisible(isVisible)) {
          changedVisibility = true;
        }
      }
    }
    return changedVisibility;
  }

  /**
   * <p>
   * Modifica o rtulo de um grupo.
   * </p>
   *
   * @param label O rtulo (No aceita {@code null}).
   *
   * @return {@code true} se ele for modificado ou {@code false} se o rtulo
   *         corrente for igual ao rtulo fornecido.
   */
  public boolean setLabel(String label) {
    if (label == null) {
      throw new IllegalArgumentException("label == null");
    }
    if (label.equals(this.label)) {
      return false;
    }
    this.label = label;
    return true;
  }

  /**
   * Obtm o rtulo.
   *
   * @return O rtulo.
   */
  public final String getLabel() {
    return this.label;
  }

  /**
   * Indica se o grupo  minimizvel.
   *
   * @return verdadeiro se o grupo  minimizvel ou falso, caso cotrrio.
   */
  public boolean isCollapsible() {
    return isCollapsible;
  }

  /**
   * Determina se o grupo deve ser minimizvel.
   *
   * @param isCollapsible verdadeiro se o grupo  minimizvel ou falso, caso
   *        cotrrio.
   */
  public void setCollapsible(boolean isCollapsible) {
    this.isCollapsible = isCollapsible;
  }
}
