/*
 * $Id$
 */
package csbase.logic.applicationservice;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.regex.Pattern;

import csbase.logic.FileInfo;
import csbase.logic.RemoteFileInputStream;
import csbase.remote.ApplicationServiceInterface;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNG.TranslationListener;

/**
 * Classe que representa um registro de aplicao.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class ApplicationRegistry implements Serializable {

  /**
   * Lista com os padres de nome que determina que uma classe pode ser
   * carregada no classloader da aplicao (alm do classpath da aplicao).
   */
  private final List<Pattern> classLoaderWhiteList;
  /**
   * Lista com os padres de nome que determina que uma classe *no* pode ser
   * carregada no classloader da aplicao (alm do classpath da aplicao).
   */
  private final List<Pattern> classLoaderBlackList;

  /**
   * cone configurado para a aplicao.
   */
  private byte[] iconDefinition;

  /**
   * Imagem configurada para a aplicao.
   */
  private byte[] imageDefinition;

  /**
   * Identificador da aplicao.
   */
  final private String id;

  /**
   * Resource bundle da aplicao para strings de texto
   */
  transient private AppPropertyResourceBundle propResBundle = null;

  /**
   * Tipos de arquivos que abrem esta aplicacao com o duplo clique
   */
  final private List<String> fileTypes = new ArrayList<String>();

  /**
   * Propriedades especficas da aplicao.
   */
  private Properties properties = null;

  /** O callback corrente */
  private TranslationListener callback;

  /**
   * Flag indicativo de registry fechado (no pode mais ser alterado).
   */
  private Boolean isClosed = false;

  /**
   * Timestamp da ltima alterao do registro no servidor.
   */
  private final long timestamp;

  /**
   * Lista da bibliotecas da aplicao no servidor.
   */
  private FileInfo[] applicationLibs;

  /**
   * Diretrio base de onde buscar os arquivos declarados no classpath.
   */
  private File classpathBaseDir;

  /**
   * Identificador da chave de propriedade: nome do programa
   */
  private static final String APP_NAME_PROPERTY = "name";

  /**
   * Identificador da chave de propriedade: descrio do programa
   */
  private static final String DESCRIPTION_PROPERTY = "description";

  /**
   * Identificador da chave de propriedade: classe (Java) do programa
   */
  private static final String CLASS_NAME_PROPERTY = "class.name";

  /**
   * Identificador da chave de propriedade: Habilitao
   */
  private static final String IS_ENABLED_PROPERTY = "enabled";

  /**
   * Identificador da chave de propriedade: Diretrio de bibliotecas da
   * aplicao a serem carregadas dinamicamente.
   */
  private static final String CLASSPATH_PROPERTY = "classpath";

  /**
   * Identificador da chave de propriedade: Flag de uso de prjetos.
   */
  private static final String REQUIRE_PROJECT_PROPERTY = "require.project";

  /**
   * Identificador da chave de propriedade: Visibilidade do dilogo
   */
  private static final String MAIN_FRAME_IS_VISIBLE_PROPERTY =
    "main.frame.visible";

  /**
   * Nome da propriedade que indica se a aplicao ser exibida com group frame.
   */
  private static final String SHOWN_AT_APP_PANEL_PROPERTY =
    "shown.at.application.panel";

  /**
   * Nome da propridade que indica se a aplicao ser exibida no menu de
   * aplicaes.
   */
  private static final String SHOWN_AT_MENU_PROPERTY =
    "shown.at.application.menu";

  /**
   * Identificador da chave de propriedade: Flag de necessidade de bundle.
   */
  private static final String NEED_BUNDLE_PROPERTY = "need.bundle";

  /**
   * Identificador da chave de propriedade: Comando
   */
  private static final String APP_COMMAND_PROPERTY = "application.command";

  /**
   * Identificador da chave de propriedade: Nome do autor
   */
  private static final String AUTHOR_NAME_PROPERTY = "author.name";

  /**
   * Identificador da chave de propriedade: Email de contato do autor
   */
  private static final String AUTHOR_MAIL_PROPERTY = "author.mail";

  /**
   * Identificador da chave de propriedade: Cdigo da verso
   */
  private static final String VERSION_PROPERTY = "version.number";

  /**
   * Identificador da chave de propriedade: tipos de arquivos tratados
   */
  private static final String FILE_TYPES_PROPERTY = "file.types";

  /**
   * Identificador da chave de propriedade: singleton
   */
  private static final String IS_SINGLETON_PROPERTY = "singleton";

  /**
   * Identificador dos arquivos adicionais de idiomas. Esta propriedade deve ser
   * definida no arquivo de propriedades da prpria aplicao, usando o mesmo
   * prefixo que os demais atributos, e seu valor deve ser o caminho at o nome
   * do arquivo de propriedades com '/' ao invs de '.', sem incluir o sufixo.
   * Por se tratar de uma propriedade do tipo "lista de strings", o nome da
   * propriedade deve ter um sufixo numrico (".1", ".2" etc.)
   * <p>
   * P.ex.: para definirmos um bundle adicional para a aplicao
   * <p>
   * <code>csbase.client.applications.notepad.Notepad</code>
   * <p>
   * a definio da propriedade ficaria no arquivo que define as propriedades da
   * aplicao e teria p.ex. o seguinte valor (o nome do arquivo  arbitrrio):
   * <p>
   * <code>/csbase/client/applications/notepad/resources/NotepadExtras</code>
   * <p>
   * Isto faria com que, para pt_BR o arquivo
   * <p>
   * <code>/csbase/client/applications/notepad/resources/NotepadExtras_pt_BR.properties</code>
   * <p>
   * fosse carregado.
   * <p>
   * <b>IMPORTANTE:</b> o caminho deve comear com '/' .
   */
  private static final String ADDITIONAL_LANGUAGE_FILE_PROPERTY =
    "additional.language.file";

  /**
   * Identificador de nome base para os arquivos externos de idiomas. Esta
   * propriedade deve ser definida no arquivo de propriedades da prpria
   * aplicao. Representa o sufixo que ser concatenado a uma URL-base. A
   * URL-base  o caminho (incluindo protocolo) que ir localizar os arquivos
   * externos. Seu valor deve ser o caminho at o nome do arquivo de
   * propriedades com '/' ao invs de '.', sem incluir o sufixo (associado ao
   * locale). Assumimos Por se tratar de uma propriedade do tipo "lista de
   * strings", o nome da propriedade deve ter um sufixo numrico (".1", ".2"
   * etc.)
   * <p>
   * P.ex.: para definirmos o nome base para um bundle externo para a aplicao
   * <p>
   * <code>csbase.client.applications.notepad.Notepad</code>
   * <p>
   * a definio da propriedade ficaria no arquivo que define as propriedades da
   * aplicao e poderia ter p.ex. o valor (o nome do arquivo  arbitrrio):
   * <p>
   * <code>/language/applications/notepad/Notepad</code>
   * <p>
   * Tomando uma URL-Base qualquer, para pt_BR poderamos carregar o arquivo
   * <p>
   * <code>http://app_server_host:app_server_port/system/language/applications/notepad/Notepad_pt_BR.properties</code>
   * <p>
   */
  private static final String EXTERNAL_LANGUAGE_FILE_BASENAME_PROPERTY =
    "external.language.file.basename";

  /**
   * Define o callback usado para chaves no encontradas.
   *
   * @param callback o objeto que implementa a interface
   *        <code>NotTranslationInterface</code> para tratar chaves no
   *        encontradas nos bundles carregados.
   */
  public void setTranslationListener(TranslationListener callback) {
    this.callback = callback;
  }

  /**
   * Obtm o callback usado para chaves no encontradas.
   *
   * @return callback o objeto que implementa a interface
   *         <code>NotTranslationInterface</code> para tratar chaves no
   *         encontradas nos bundles carregados.
   */
  public TranslationListener getTranslationListener() {
    return callback;
  }

  /**
   * Carrega os arquivos de idioma. O primeiro bundle carregado  o arquivo da
   * prpria aplicao (p.ex. <code>AppClassName_pt_BR.properties</code>).
   * Arquivos adicionais definidos via
   * {@link #ADDITIONAL_LANGUAGE_FILE_PROPERTY} so carregados em seguida, na
   * ordem em que foram definidos. A busca por tradues acontece do ltimo
   * arquivo definido para o primeiro; o primeiro que possuir a chave desejada
   * ser usado.
   *
   * @param mainClass Classe principal da aplicao.
   * @param locale O locale do usurio logado.
   * @return indicativo se algume arquivo foi encontrado e carregado.
   */
  final public boolean loadClientBundles(Class<?> mainClass,
    final Locale locale) {
    final List<String> allLanguageFilePathPrefixList = new LinkedList<String>();

    final String defaultLanguageFilePathPrefix =
      getDefaultLanguageFilePathPrefix();
    allLanguageFilePathPrefixList.add(defaultLanguageFilePathPrefix);

    final List<String> additionalLanguageFilePathPrefixes =
      getStringListSpecificProperty(ADDITIONAL_LANGUAGE_FILE_PROPERTY);
    allLanguageFilePathPrefixList.addAll(additionalLanguageFilePathPrefixes);

    final String language = locale.getLanguage();
    final String country = locale.getCountry();

    boolean loaded = false;
    AppPropertyResourceBundle resourceBundle = null;
    for (final String filePathPrefix : allLanguageFilePathPrefixList) {
      final String idiomFileName = filePathPrefix + "_" + language + "_"
        + country + ".properties";
      final InputStream in = mainClass.getResourceAsStream(idiomFileName);
      if (in == null) {
        final String fmt =
          "(%s) [loadClientBundles] No detectou-se arquivo de idiomas: %s";
        final String err = String.format(fmt, getId(), idiomFileName);
        logBundleLoading(err);
        continue;
      }

      try {
        final String fmt =
          "(%s) [loadClientBundles] Carregando arquivo de idiomas: %s";
        final String err = String.format(fmt, getId(), idiomFileName);
        logBundleLoading(err);
        resourceBundle = new AppPropertyResourceBundle(in, locale, this);
      }
      catch (final IOException e) {
        final String fmt =
          "(%s) [loadClientBundles] Falha de carga de bundle cliente da aplicao: %s - %s";
        final String err = String.format(fmt, getId(), idiomFileName, e
          .getMessage());
        logBundleLoading(err);
        continue;
      }
      final AppPropertyResourceBundle oldBundle = propResBundle;
      propResBundle = resourceBundle;
      if (oldBundle != null) {
        propResBundle.setParent(oldBundle);
      }
      try {
        in.close();
      }
      catch (final IOException e) {
        final String fmt =
          "(%s) [loadClientBundles] Falha de close de bundle cliente da aplicao: %s - %s";
        final String err = String.format(fmt, getId(), idiomFileName, e
          .getMessage());
        logBundleLoading(err);
      }
      loaded = true;
    }
    return loaded;
  }

  /**
   * Mtodo interno de depurao que pode estar eventualmente comentado ou usar
   * algum mecanismo de log no cliente (quando existir).
   *
   * @param message mensagem
   */
  private void logBundleLoading(final String message) {
    //System.err.println(message);
  }

  /**
   * Carga de bundle de servidor.
   *
   * @param appService servio
   * @param locale locale
   * @return indicativo se o arquivo foi encontrado e carregado.
   * @throws RemoteException em caso de erro
   */
  public boolean loadServerBundles(ApplicationServiceInterface appService,
    Locale locale) throws RemoteException {
    final String appId = getId();
    final String fileName = getDefaultLanguageFileName(locale);
    final String[] path = new String[] { "bundles", fileName };
    final RemoteFileChannelInfo info = appService.getApplicationResource(appId,
      path);
    if (info == null) {
      return false;
    }
    try {
      InputStream stream = new RemoteFileInputStream(info);
      final AppPropertyResourceBundle srvBundle = new AppPropertyResourceBundle(
        stream, locale, this);
      final AppPropertyResourceBundle oldBundle = propResBundle;
      propResBundle = srvBundle;
      if (oldBundle != null) {
        propResBundle.setParent(oldBundle);
      }

      final String fmt =
        "(%s) [loadServerBundles] Carga de bundle remoto da aplicao: %s";
      final String err = String.format(fmt, getId(), fileName);
      logBundleLoading(err);

      return true;
    }
    catch (final IOException e) {
      final String fmt =
        "(%s) [loadServerBundles] Falha de carga de bundle remoto da aplicao: %s = %s ";
      final String err = String.format(fmt, getId(), fileName, e.getMessage());
      logBundleLoading(err);
      return false;
    }
  }

  /**
   * Obtm o nome simples da classe da aplicao.
   *
   * @return o nome
   */
  private String getClassSimpleName() {
    final String className = getClassName();
    final int idx = className.lastIndexOf(".");
    final int length = className.length();
    final String classSimpleName = className.substring(idx + 1, length);
    return classSimpleName;
  }

  /**
   * Obtm o nome do arquivo de idioma padro da aplicao.
   *
   * @param locale locale.
   * @return o nome.
   */
  private String getDefaultLanguageFileName(Locale locale) {
    final String language = locale.getLanguage();
    final String country = locale.getCountry();
    final String classSimpleName = getClassSimpleName();
    final String fileName = classSimpleName + "_" + language + "_" + country
      + ".properties";
    return fileName;
  }

  /**
   * Carrega os arquivos de idiomas a partir de URLs. Os arquivos so carregados
   * na ordem em que foram definidos. A busca por tradues acontece do ltimo
   * arquivo definido para o primeiro; o primeiro que possuir a chave desejada
   * ser usado.
   * <p>
   * <br>
   * Assumimos que os bundles da aplicao j foram lidos anteriormente.
   * Portanto, se os tipos de bundles forem utilizados por uma aplicao, isso
   * significa que os bundles externos (obtidos de URLs) sero colocados no fim
   * da hierarquia.
   *
   * @param pathPrefixURL URL-base (prefixo) para obter a URl dos bundles
   *        externos de cada aplicao. Deve conter o protocolo usado (http://,
   *        file://, etc.)
   *
   * @param locale O locale do usurio logado.
   */
  final public void loadURLBundles(final String pathPrefixURL,
    final Locale locale) {
    try {

      final List<String> allLanguageURLPrefixList =
        getStringListSpecificProperty(EXTERNAL_LANGUAGE_FILE_BASENAME_PROPERTY);

      AppPropertyResourceBundle resourceBundle = propResBundle;
      for (final String filePathPrefix : allLanguageURLPrefixList) {

        final String idiomFileName = pathPrefixURL + filePathPrefix + "_"
          + locale + ".properties";
        URL url = new URL(idiomFileName);
        final InputStream in = url.openStream();
        if (in == null) {
          throw new IllegalStateException(
            "Falha na carga do arquivo de idiomas: " + filePathPrefix);
        }
        resourceBundle = new AppPropertyResourceBundle(in, locale, this);
        final String fmt =
          "(%s) [loadURLBundles] Carregando arquivo de idiomas: %s";
        final String err = String.format(fmt, getId(), idiomFileName);
        logBundleLoading(err);

        final AppPropertyResourceBundle oldBundle = propResBundle;
        propResBundle = resourceBundle;
        if (oldBundle != null) {
          propResBundle.setParent(oldBundle);
        }
        in.close();
      }
    }
    catch (final IOException e) {
      final String err =
        "[loadURLBundles] Falha de carga de resource bundle da aplicao: ";
      final String className = getClassName();
      throw new IllegalStateException(err + className + " - " + e.getMessage());
    }
  }

  /**
   * Obtm o nome do arquivo de idioma padro da aplicao. O arquivo padro
   * fica no diretrio resources no mesmo pacote que a classe principal da
   * aplicaAo,
   *
   * @return o nome do arquivo de idioma padro da aplicao
   */
  private String getDefaultLanguageFilePathPrefix() {
    final String className = getClassName();
    final int idx = className.lastIndexOf(".");
    final String path = className.substring(0, idx);
    final int length = className.length();
    final String classLastName = className.substring(idx + 1, length);
    final String sep = "/";
    final String replaced = path.replace(".", sep);
    final String resPath = sep + replaced + sep + "resources" + sep
      + classLastName;
    return resPath;
  }

  /**
   * Obtem todos os tipos de arquivos que abrem diretamente esta aplicacao.
   */
  private void loadFileTypes() {
    if (isPropertyNull(FILE_TYPES_PROPERTY)) {
      return;
    }
    final String prop = getStringSpecificProperty(FILE_TYPES_PROPERTY);
    final String[] aux = prop.split(",");
    for (final String element : aux) {
      fileTypes.add(element.trim().toUpperCase());
    }
  }

  /**
   * Obtem uma uma de internacionalizao com base em um objeto.
   *
   * @param clazz classe
   * @param tag tag
   * @return chave (key)
   */
  private String getClassKeyForInternacionalization(final Class<?> clazz,
    final String tag) {
    final String prefix = clazz.getSimpleName();
    final String key = prefix + "." + tag;
    return key;
  }

  /**
   * Consulta da existncia de uma string de idioma, com base em uma classe.
   *
   * @param clazz classe.
   * @param key a chave de busca.
   * @return indicativo
   * @see #hasString(String)
   */
  final public boolean hasClassString(final Class<?> clazz, final String key) {
    final String k = getClassKeyForInternacionalization(clazz, key);
    return hasString(k);
  }

  /**
   * Retorna uma string de idioma, com base em uma classe.
   *
   * @param clazz classe.
   * @param key a chave de busca.
   * @return indicativo
   * @see #getString(String)
   */
  final public String getClassString(final Class<?> clazz, final String key) {
    final String k = getClassKeyForInternacionalization(clazz, key);
    return getString(k);
  }

  /**
   * Retorna uma string de idioma, com base em uma classe.
   *
   * @param clazz a classe.
   * @param key a chave de busca.
   * @param args argumentos
   * @return indicativo
   * @see #getString(String, Object[])
   */
  final public String getClassString(final Class<?> clazz, final String key,
    final Object... args) {
    final String k = getClassKeyForInternacionalization(clazz, key);
    return getString(k, args);
  }

  /**
   * Consulta da existncia de uma string de idioma, com base em um objeto.
   *
   * @param object o objeto.
   * @param key a chave de busca.
   * @return indicativo
   * @see #hasString(String)
   */
  final public boolean hasObjectString(final Object object, final String key) {
    final Class<? extends Object> clazz = object.getClass();
    final String k = getClassKeyForInternacionalization(clazz, key);
    return hasString(k);
  }

  /**
   * Retorna uma string de idioma, com base em um objeto.
   *
   * @param object o objeto.
   * @param key a chave de busca.
   * @return indicativo
   * @see #getString(String)
   */
  final public String getObjectString(final Object object, final String key) {
    final Class<? extends Object> clazz = object.getClass();
    final String k = getClassKeyForInternacionalization(clazz, key);
    return getString(k);
  }

  /**
   * Retorna uma string de idioma, com base em um objeto.
   *
   * @param object o objeto.
   * @param key a chave de busca.
   * @param args argumentos
   * @return indicativo
   * @see #getString(String, Object[])
   */
  final public String getObjectString(final Object object, final String key,
    final Object... args) {
    final Class<? extends Object> clazz = object.getClass();
    final String k = getClassKeyForInternacionalization(clazz, key);
    return getString(k, args);
  }

  /**
   * Consulta da existncia de uma string de idioma.
   *
   * @param key a chave de busca.
   * @return o texto no idioma correto.
   */
  final public boolean hasString(final String key) {
    try {
      if (this.propResBundle == null) {
        return false;
      }
      return this.propResBundle.containsKey(key);
    }
    catch (final Exception e) {
      return false;
    }
  }

  /**
   * Obtem uma string de idioma.
   *
   * @param key a chave de busca.
   * @return o texto no idioma correto.
   * @see #propResBundle
   */
  final public String getString(final String key) {
    return this.propResBundle.getString(key);
  }

  /**
   * Obtem uma string de idioma.
   *
   * @param key a chave de busca.
   * @param objs os objetos de formatao
   * @return o texto no idioma correto.
   * @see #propResBundle
   */
  final public String getString(final String key, final Object[] objs) {
    try {
      final String pattern = getString(key);
      return MessageFormat.format(pattern, objs);
    }
    catch (final Exception e) {
      return (key == null) ? "<<<null>>>" : ("<<<" + key + ">>>");
    }
  }

  /**
   * Consulta se aplicao aparece em menu.
   *
   * @return um flag indicativo.
   */
  final public boolean isShownAtApplicationMenu() {
    if (isPropertyNull(SHOWN_AT_MENU_PROPERTY)) {
      return true;
    }
    return getBooleanSpecificProperty(SHOWN_AT_MENU_PROPERTY);
  }

  /**
   * Consulta se aplicao aparece no app-panel
   *
   * @return um flag indicativo.
   */
  final public boolean isShownAtApplicationPanel() {
    if (isPropertyNull(SHOWN_AT_APP_PANEL_PROPERTY)) {
      return true;
    }
    return getBooleanSpecificProperty(SHOWN_AT_APP_PANEL_PROPERTY);

  }

  /**
   * Consulta se aplicao necessita de projeto.
   *
   * @return um flag indicativo.
   */
  final public boolean requireProject() {
    return getBooleanSpecificProperty(REQUIRE_PROJECT_PROPERTY);
  }

  /**
   * Consulta se a frame da aplicao deve ficar visvel.
   *
   * @return um flag indicativo.
   */
  final public boolean mainFrameIsVisible() {
    if (isPropertyNull(MAIN_FRAME_IS_VISIBLE_PROPERTY)) {
      return true;
    }
    return getBooleanSpecificProperty(MAIN_FRAME_IS_VISIBLE_PROPERTY);
  }

  /**
   * Obtm o id da aplicao.
   *
   * @return o id com uma string.
   * @see #id
   */
  final public String getId() {
    return this.id;
  }

  /**
   * Obtm o resource bundle da aplicao
   *
   * @return o bundle
   * @see #propResBundle
   */
  final public AppPropertyResourceBundle getResourceBundle() {
    return this.propResBundle;
  }

  /**
   * Obtm a classe da aplicao.
   *
   * @return o nome da classe.
   */
  final public String getClassName() {
    return getStringSpecificProperty(CLASS_NAME_PROPERTY);
  }

  /**
   * Obtm os tipos de arquivo que abrem diretamente esta aplicao.
   *
   * @return os tipos de arquivo.
   */
  final public List<String> getFileTypes() {
    return this.fileTypes;
  }

  /**
   * Consulta ao comando da aplicao.
   *
   * @return um texto com a string.
   */
  final public String getApplicationCommand() {
    if (isPropertyNull(APP_COMMAND_PROPERTY)) {
      return null;
    }
    return getStringSpecificProperty(APP_COMMAND_PROPERTY);
  }

  /**
   * Consulta ao diretrio onde est o comando da aplicao.
   *
   * @return o diretrio onde est o comando da aplicao
   */
  final public String getApplicationCommandDir() {
    if (isPropertyNull(APP_COMMAND_PROPERTY)) {
      return null;
    }
    final String appCommand = getStringSpecificProperty(APP_COMMAND_PROPERTY);
    return ApplicationRegistryUtilities.getCommandPath(appCommand);
  }

  /**
   * Consulta a habilitao do aplicativo.
   *
   * @return um flag booleano indicativo.
   * @see #isEnabled
   */
  final public boolean isEnabled() {
    return getBooleanSpecificProperty(IS_ENABLED_PROPERTY);
  }

  /**
   * Consulta a lista de bibliotecas do classpath da aplicao.
   *
   * @return a lista de bibliotecas (jars, zips) da aplicao.
   */
  final public List<String> getClasspath() {
    if (!isPropertyNull(CLASSPATH_PROPERTY)) {
      String classpathAsString = getStringSpecificProperty(CLASSPATH_PROPERTY);
      List<String> classpathList = new ArrayList<String>();
      String[] classpathEntries = classpathAsString.split(",");
      for (String classpathEntry : classpathEntries) {
        String entry = classpathEntry.trim();
        if (!classpathEntry.isEmpty()) {
          classpathList.add(entry);
        }
      }
      return classpathList;
    }
    else {
      return getStringListSpecificProperty(CLASSPATH_PROPERTY);
    }
  }

  /**
   * Indica se a aplicao possui uma linha de comando e seu o comando foi
   * encotrado no <code>PATH</code> do cliente.
   *
   * @return true se a aplicao pode ser executada
   */
  final public boolean isRunnable() {
    final String appCommand = getApplicationCommandDir();
    if (appCommand == null) {
      return true;
    }
    final String appCommandDir = ApplicationRegistryUtilities.getCommandPath(
      appCommand);
    return appCommandDir != null;
  }

  /**
   * Consulta se uma aplicao  "singleton"
   *
   * @return um indicativo
   */
  final public boolean isSingleton() {
    return getBooleanSpecificProperty(IS_SINGLETON_PROPERTY);
  }

  /**
   * Indicativo de propriedade sem valor.
   *
   * @param propName nome da propriedade.
   * @return indicativo
   */
  final public boolean isPropertyNull(final String propName) {
    final String tag = id + "." + propName;
    final String value = properties.getProperty(tag);
    if (value == null) {
      return true;
    }
    if (value.isEmpty()) {
      return true;
    }
    return false;
  }

  /**
   * Fecha as definies de um registry.
   *
   * @throws ApplicationRegistryException em caso de m configurao do
   *         servidor.
   */
  final public void closeConfiguration() throws ApplicationRegistryException {
    checkProperties();
    loadFileTypes();
    loadClasspath();
    synchronized (isClosed) {
      if (isClosed) {
        throw new IllegalStateException("Registry j fechado!");
      }
      isClosed = true;
    }
  }

  /**
   * Construtor.
   *
   * @param id identificador da aplicao.
   * @param timestamp data da ltima modificao do diretrio da aplicao no
   *        servidor.
   */
  public ApplicationRegistry(final String id, long timestamp) {
    this.id = id;
    this.properties = new Properties();
    this.timestamp = timestamp;
    this.classLoaderWhiteList = new ArrayList<Pattern>();
    this.classLoaderBlackList = new ArrayList<Pattern>();
  }

  /**
   * Faz checagem de existncia de propriedades no registry criado no servidor
   * para no impedir que o servidor entre mal configurado.
   *
   * @throws ApplicationRegistryException em caso de falta de configurao.
   */
  private void checkProperties() throws ApplicationRegistryException {
    final String[] mustProperties = new String[] { CLASS_NAME_PROPERTY,
        IS_ENABLED_PROPERTY, IS_SINGLETON_PROPERTY, NEED_BUNDLE_PROPERTY,
        REQUIRE_PROJECT_PROPERTY };
    for (String prp : mustProperties) {
      if (isPropertyNull(prp)) {
        final String fmt = "Falta propriedade da aplicao %s (%s).";
        final String err = String.format(fmt, id, prp);
        throw new ApplicationRegistryException(err);
      }
    }
  }

  /**
   * Ajusta o diretrio base de onde buscar os arquivos declarados no classpath.
   *
   * @param baseDir o diretrio base.
   */
  public void setClasspathBaseDir(File baseDir) {
    this.classpathBaseDir = baseDir;
  }

  /**
   * Carrega as bibliotecas declaradas no classpath da aplicao.
   *
   * @throws ApplicationRegistryException caso haja alguma declarao invlida
   *         no classpath.
   */
  private void loadClasspath() throws ApplicationRegistryException {
    List<String> classpath = getClasspath();
    if (classpath == null || classpath.isEmpty()) {
      return;
    }
    List<FileInfo> fileInfos = new ArrayList<FileInfo>();
    /*
     * Forando o uso do separador UNIX, para que um mesmo repositrio de
     * aplicaes possa ser usado por um servidor Windows ou Unix.
     */
    String separator = "/";
    for (String libPath : classpath) {
      checkLibrary(classpathBaseDir, libPath);
      String filePath = FileUtils.getFilePath(libPath, separator);
      String[] splitPath = FileUtils.splitPath(filePath, separator);
      String fileName = FileUtils.getFileName(libPath, separator);
      FileInfo fileInfo;
      if (splitPath.length > 0) {
        fileInfo = new FileInfo(splitPath, fileName, false);
      }
      else {
        fileInfo = new FileInfo(fileName, false);
      }
      fileInfos.add(fileInfo);
    }
    this.applicationLibs = fileInfos.toArray(new FileInfo[fileInfos.size()]);
  }

  /**
   * Valida as bibliotecas declaradas no classpath da aplicao.
   *
   * @param baseDir diretrio base do classpath.
   * @param libPath caminho para a bilioteca.
   *
   * @throws ApplicationRegistryException caso haja alguma declarao invlida
   *         no classpath.
   */
  private void checkLibrary(File baseDir, String libPath)
    throws ApplicationRegistryException {
    final File path = new File(baseDir, libPath);
    if (!path.exists()) {
      final String fmt =
        "O arquivo %s do classpath da aplicao %s no foi encontrado";
      final String err = String.format(fmt, path, id);
      throw new ApplicationRegistryException(err);
    }
    if (path.isDirectory()) {
      final String fmt =
        "O arquivo %s do classpath da aplicao %s  um diretrio";
      final String err = String.format(fmt, path, id);
      throw new ApplicationRegistryException(err);
    }
    boolean isInsideBaseDir;
    try {
      isInsideBaseDir = org.apache.commons.io.FileUtils.directoryContains(
        baseDir, path);
    }
    catch (IOException e) {
      final String fmt =
        "Erro de ao verificar se o arquivo %s do classpath da aplicao %s se encontra dentro do diretrio da aplicao %s";
      final String err = String.format(fmt, path, id, baseDir);
      throw new ApplicationRegistryException(err);
    }
    if (!isInsideBaseDir) {
      final String fmt =
        "O arquivo %s do classpath da aplicao %s no se encontra dentro do diretrio da aplicao %s";
      final String err = String.format(fmt, path, id, baseDir);
      throw new ApplicationRegistryException(err);
    }
  }

  /**
   * Consulta de bundle  necessrio.
   *
   * @return indicativo.
   */
  public final boolean isBundleRequired() {
    return getBooleanSpecificProperty(NEED_BUNDLE_PROPERTY);
  }

  /**
   * <p>
   * Obtm uma lista de valores de propriedades que seguem o padro para
   * propriedades multi-valoradas usado pelo JDK da Sun.
   * </p>
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   *
   * @param name O nome da propriedade (nos exemplos acima, seriam "propriedade"
   *        e "propriedade.nome" respectivamente).
   *
   * @return Uma lista com todas as propriedades (caso no existam propriedades,
   *         a lista estar vazia).
   */
  final public List<String> getStringListSpecificProperty(final String name) {
    final List<String> list = new LinkedList<String>();
    for (int i = 1; true; i++) {
      final String pName = name + "." + i;
      if (isPropertyNull(pName)) {
        break;
      }
      final String value = getStringSpecificProperty(pName);
      list.add(value);
    }
    return Collections.unmodifiableList(list);
  }

  /**
   * Busca da imagem configurada para a aplicao.
   *
   * @return a imagem
   */
  final public byte[] getImageDefinition() {
    return imageDefinition;
  }

  /**
   * Busca do cone configurada para a aplicao.
   *
   * @return a imagem
   */
  final public byte[] getIconDefinition() {
    return iconDefinition;
  }

  /**
   * Consulta de propriedade formatada.
   *
   * @param propertyName o nome da tag desejada.
   * @param locale locale
   * @return o texto com o valor da propriedade desejada.
   */
  private String getLocaledProperty(final String propertyName,
    final Locale locale) {
    final String language = locale.getLanguage();
    final String country = locale.getCountry();
    final Object[] args = new Object[] { language, country };
    final String key = propertyName + ".{0}.{1}";
    final String fmtKey = MessageFormat.format(key, args);
    if (isPropertyNull(fmtKey)) {
      Locale nativeLocale = LNG.getNativeLocale();
      if (nativeLocale != null && !locale.equals(LNG.getNativeLocale())) {
        return getLocaledProperty(propertyName, nativeLocale);
      }
      else {
        return "<<" + fmtKey + ">>>";
      }
    }
    final String value = getStringSpecificProperty(fmtKey);
    return value;
  }

  /**
   * Consulta ao email de contato.
   *
   * @return o email como string
   */
  final public String getAuthorMail() {
    if (isPropertyNull(AUTHOR_MAIL_PROPERTY)) {
      return "-";
    }
    return getStringSpecificProperty(AUTHOR_MAIL_PROPERTY);
  }

  /**
   * Consulta ao nome do autor do programa.
   *
   * @return o nome como string
   */
  final public String getAuthorName() {
    if (isPropertyNull(AUTHOR_NAME_PROPERTY)) {
      return "-";
    }
    return getStringSpecificProperty(AUTHOR_NAME_PROPERTY);
  }

  /**
   * Montagem de um nmero (convencionado) para uma verso de aplicativo.
   *
   * @param vTxt o texto da verso.
   * @return um nmero de verso.
   */
  private int getVersionNumber(final String vTxt) {
    if (vTxt == null) {
      return 0;
    }
    final String txt = vTxt.trim();
    if (txt.equals("")) {
      return 0;
    }
    final String[] elems = txt.split("\\.");
    if (elems == null || elems.length != 3) {
      return 0;
    }

    int mj, mn, pt;
    try {
      mj = Integer.parseInt(elems[0]);
    }
    catch (final Throwable t) {
      mj = 0;
    }
    try {
      mn = Integer.parseInt(elems[1]);
    }
    catch (final Throwable t) {
      mn = 0;
    }
    try {
      pt = Integer.parseInt(elems[2]);
    }
    catch (final Throwable t) {
      pt = 0;
    }

    return (mj * 1000000) + (mn * 1000) + (pt);
  }

  /**
   * Consulta  verso do programa
   *
   * @return a verso como nmero
   * @see #getVersionNumber(String)
   */
  final public int getVersionNumber() {
    return getVersionNumber(getVersion());
  }

  /**
   * Consulta  verso do programa
   *
   * @return a verso como string
   */
  final public String getVersion() {
    if (isPropertyNull(VERSION_PROPERTY)) {
      return "0.0.0";
    }
    return getStringSpecificProperty(VERSION_PROPERTY);
  }

  /**
   * Obtm uma propriedade especfica do tipo inteiro.
   *
   * @param propName nome da propriedade.
   *
   * @return propriedade especfica do tipo inteiro.
   *
   * @throws IllegalStateException se a propriedade no foi definida ou se o
   *         valor no pode ser convertido para um nmero inteiro
   */
  public final int getIntSpecificProperty(final String propName) {
    final String str = getStringSpecificProperty(propName);
    try {
      final int value = Integer.parseInt(str);
      return value;
    }
    catch (final Exception e) {
      final String err = "Propriedade " + propName + " inteira mal formatada";
      throw new IllegalStateException(err + " - applicao:" + getId());
    }
  }

  /**
   * Obtm uma propriedade especfica do tipo long.
   *
   * @param propName nome da propriedade.
   *
   * @return propriedade especfica do tipo long.
   *
   * @throws IllegalStateException se a propriedade no foi definida ou se o
   *         valor no pode ser convertido para um nmero long
   */
  public final long getLongSpecificProperty(final String propName) {
    final String str = getStringSpecificProperty(propName);
    try {
      final long value = Long.parseLong(str);
      return value;
    }
    catch (final Exception e) {
      final String err = "Propriedade " + propName + " long mal formatada";
      throw new IllegalStateException(err + " - applicao:" + getId());
    }
  }

  /**
   * Obtm uma propriedade especfica do tipo booleana.
   *
   * @param propName nome da propriedade.
   * @return propriedade especfica do tipo booleana.
   */
  public final boolean getBooleanSpecificProperty(final String propName) {
    final String str = getStringSpecificProperty(propName);
    final String[] accepts = new String[] { "true", "false" };
    for (final String acc : accepts) {
      if (acc.equals(str.trim())) {
        final boolean value = Boolean.parseBoolean(str);
        return value;
      }
    }
    final String err = "Propriedade " + propName + " booleana mal formatada";
    throw new IllegalArgumentException(err + " - aplicao:" + getId());
  }

  /**
   * Obtm uma propriedade especfica.
   *
   * @param propName nome da propriedade.
   *
   * @return propriedade especfica.
   *
   * @throws IllegalStateException se a propriedade no foi definida
   */
  final public String getStringSpecificProperty(final String propName) {
    if (isPropertyNull(propName)) {
      final String err = "Prop. indefinida [" + propName + "] para apl. " + id;
      throw new IllegalStateException(err);
    }

    final String tag = id + "." + propName;
    final String value = properties.getProperty(tag);
    if (value == null) {
      return null;
    }
    if (value.isEmpty()) {
      return null;
    }
    return value.trim();
  }

  /**
   * Obtm uma propriedade especfica do tipo double.
   *
   * @param propName nome da propriedade.
   *
   * @return propriedade especfica do tipo double.
   *
   * @throws IllegalStateException se a propriedade no foi definida ou se o
   *         valor no pode ser convertido para um nmero double.
   */
  public final double getDoubleSpecificProperty(final String propName) {
    final String str = getStringSpecificProperty(propName);
    try {
      final double value = Double.parseDouble(str);
      return value;
    }
    catch (final Exception e) {
      final String err = "Propriedade " + propName + " double mal formatada";
      throw new IllegalStateException(err + " - applicao:" + getId());
    }
  }

  /**
   * Consulta ao conunto propriedades especficas.
   *
   * @return conjunto.
   */
  final public Properties getSpecificProperties() {
    return properties;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public String toString() {
    return getId();
  }

  /**
   * Consulta o nome de uma aplicao.
   *
   * @param locale locale.
   * @return o nome
   */
  public final String getApplicationName(final Locale locale) {
    return getLocaledProperty(APP_NAME_PROPERTY, locale);
  }

  /**
   * Consulta o nome de uma aplicao.
   *
   * @param locale locale.
   * @return o nome
   */
  public final String getApplicationDescription(final Locale locale) {
    return getLocaledProperty(DESCRIPTION_PROPERTY, locale);
  }

  /**
   * Ajuste de icone de aplicao.
   *
   * @param img16 imagem
   */
  final public void setIconDefinition(byte[] img16) {
    checkClosed();
    this.iconDefinition = img16;
  }

  /**
   * Ajuste de imagem de aplicao.
   *
   * @param img32 imagem
   */
  final public void setImageDefinition(byte[] img32) {
    checkClosed();
    this.imageDefinition = img32;
  }

  /**
   * Ajusta properties da aplicao.
   *
   * @param prop propriedade a ser ajustada.
   */
  final public void setSpecificProperties(final Properties prop) {
    checkClosed();
    this.properties = prop;
  }

  /**
   * Define a lista com os padres de nome que determina que uma classe pode ser
   * carregada no classloader da aplicao (alm das que so declaradas no
   * classpath da aplicao).
   *
   * @param whiteList a lista.
   */
  final public void setClassLoaderWhiteList(List<Pattern> whiteList) {
    checkClosed();
    this.classLoaderWhiteList.addAll(whiteList);
  }

  /**
   * Define a lista com os padres de nome que determina que uma classe *no*
   * pode ser carregada no classloader da aplicao (alm das que so declaradas
   * no classpath da aplicao).
   *
   * @param blackList a lista.
   */
  final public void setClassLoaderBlackList(List<Pattern> blackList) {
    checkClosed();
    this.classLoaderBlackList.addAll(blackList);
  }

  /**
   * Faz checagem de lock do registry (i.e. j est fechado).
   *
   * @throws IllegalStateException em caso de erro.
   */
  private void checkClosed() throws IllegalStateException {
    synchronized (isClosed) {
      if (isClosed) {
        final String err = "ApplicationRegistry bloqueado!";
        throw new IllegalStateException(err);
      }
    }
  }

  /**
   * Obtm a lista com os padres de nome que determina que uma classe pode ser
   * carregada no classloader da aplicao (alm das que so declaradas no
   * classpath da aplicao).
   *
   * @return a lista.
   */
  public List<Pattern> getClassLoaderWhiteList() {
    return Collections.unmodifiableList(classLoaderWhiteList);
  }

  /**
   * Obtm a lista com os padres de nome que determina que uma classe *no*
   * pode ser carregada no classloader da aplicao (alm das que so declaradas
   * no classpath da aplicao).
   *
   * @return a lista.
   */
  public List<Pattern> getClassLoaderBlackList() {
    return Collections.unmodifiableList(classLoaderBlackList);
  }

  /**
   * Retorna o timestamp desse registro no servidor. Essa informao pode ser
   * utilizada para verificar se a aplicao foi atualizada no servidor.
   *
   * @return o timestamp.
   */
  final public long getTimestamp() {
    return timestamp;
  }

  /**
   * Obtm as bibliotecas da aplicao.
   *
   * @return Um conjunto de descritores de arquivo {@link FileInfo} das
   *         bibliotecas.
   */
  public FileInfo[] getApplicationLibs() {
    return this.applicationLibs;
  }

}