package csbase.logic.algorithms.flows.configurator;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;

public final class Node implements Serializable {

  private AlgorithmConfigurator configurator;
  private int id;
  private Map<String, Output> outputsByParameterName;

  public Node(int id, AlgorithmConfigurator configurator) {
    this.id = id;
    setConfigurator(configurator);
    createOutputs();
  }

  @Override
  public boolean equals(Object object) {
    if (object == null) {
      return false;
    }

    if (!getClass().equals(object.getClass())) {
      return false;
    }
    Node node = (Node) object;
    return getId() == node.getId();
  }

  public AlgorithmConfigurator getConfigurator() {
    return this.configurator;
  }

  public int getId() {
    return this.id;
  }

  public Output getOutput(String parameterName) {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    return this.outputsByParameterName.get(parameterName);
  }

  public Collection<Output> getOutputs() {
    return Collections.unmodifiableCollection(this.outputsByParameterName
      .values());
  }

  @Override
  public int hashCode() {
    return getId();
  }

  private void createOutputs() {
    this.outputsByParameterName = new HashMap<String, Output>();
    /* FileParameter */
    for (OutputFileParameter parameter : getConfigurator()
      .getOutputFileParameters()) {
      if (parameter.getMode() == FileParameterMode.REGULAR_FILE) {
        this.outputsByParameterName.put(parameter.getName(), new Output(
          getId(), parameter.getName(), false));
      }
      else if (parameter.getMode() == FileParameterMode.DIRECTORY) {
        this.outputsByParameterName.put(parameter.getName(), new Output(
          getId(), parameter.getName(), true));
      }
    }
    /* URLParameter */
    for (OutputURLParameter parameter : getConfigurator()
      .getOutputURLParameters()) {
      if (parameter.getMode() == FileParameterMode.REGULAR_FILE) {
        this.outputsByParameterName.put(parameter.getName(), new Output(
          getId(), parameter.getName(), false));
      }
      else if (parameter.getMode() == FileParameterMode.DIRECTORY) {
        this.outputsByParameterName.put(parameter.getName(), new Output(
          getId(), parameter.getName(), true));
      }
    }
  }

  private void setConfigurator(AlgorithmConfigurator configurator) {
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    this.configurator = configurator;
  }
}
