package csbase.remote;

import java.rmi.RemoteException;
import java.security.cert.Certificate;
import java.util.Map;
import java.util.SortedSet;

import csbase.logic.diagnosticservice.DeploymentInfo;
import csbase.logic.server.ServerInfo;
import csbase.logic.server.ServerInfoData;

/**
 * Interface que define o comportamento do servio de servidores.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface ServerServiceInterface extends ServiceInterface,
  RemoteObservable {
  /** Nome do servio */
  public static final String SERVICE_NAME = "ServerService";

  /**
   * Obtm um conjunto ordenado (por nome) com as informaes de todos os
   * servidores pertencentes ao repositrio.
   * 
   * @return As informaes dos servidores.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public SortedSet<ServerInfo> getServersInfos() throws RemoteException;

  /**
   * Adiciona informaes de um servidor ao repositrio.
   * 
   * @param serverInfoData Os dados mnimo necessrios para adicionar ao
   *        repositrio.
   * 
   * @return true, caso o servidor tenha sido adicionado, ou false, caso j
   *         exista um outro servidor com o mesmo nome.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public boolean addServerInfo(ServerInfoData serverInfoData)
    throws RemoteException;

  /**
   * Remove as informaes de um servidor do repositrio.
   * 
   * @param serverInfo Informaes do servidor a ser removido.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public void removeServerInfo(ServerInfo serverInfo) throws RemoteException;

  /**
   * Modifica as informaes de um servidor no repositrio.
   * 
   * @param serverInfo Informaes do servidor no seu estado antigo.
   * @param serverInfoData Os dados a serem alterados.
   * 
   * @return true, caso as informaes do servidor tenham sido modificadas, ou
   *         false, caso j exista um outro servidor com o mesmo nome.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio..
   */
  public boolean modifyServerInfo(ServerInfo serverInfo,
    ServerInfoData serverInfoData) throws RemoteException;

  /**
   * Adiciona um certificado ao servidor
   * 
   * @param serverName O nome do servidor
   * @param c A instncia do certificado
   * @return true se o certificado foi adicionado com sucesso, false caso
   *         contrrio
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public boolean addCert(String serverName, Certificate c)
    throws RemoteException;

  /**
   * Remove o certificado do servidor
   * 
   * @param serverName O nome do servidor
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public void removeCert(String serverName) throws RemoteException;

  /**
   * Obtm todas as propriedades runtime do servidor na forma de uma string.
   * 
   * @return propriedades runtime do servidor, na forma de uma string
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   * 
   * @see #getRuntimeProperties()
   */
  public String listRuntimeProperties() throws RemoteException;

  /**
   * Obtm as propriedades runtime do servidor sob a forma de um mapa
   * &lt;chave,valor&gt;.
   * 
   * @return propriedades runtime do servidor, sob a forma de um mapa
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   * 
   * @see #listRuntimeProperties()
   */
  public Map<String, String> getRuntimeProperties() throws RemoteException;

  /**
   * Obtm informaes a respeito do ambiente de instalao do servidor.
   * 
   * @return informaes a respeito do ambiente de instalao do servidor
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public DeploymentInfo getDeploymentInfo() throws RemoteException;

  /**
   * Exibe o nmero de usurios cadastrados no sistema (inclui o admin).
   * 
   * @return nmero de usurios cadastrados no sistema
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public int getNumRegisteredUsers() throws RemoteException;

  /**
   * Obtm o timestamp (em ms) referente  hora de inicializao do servidor.
   * 
   * @return timestamp (em ms) referente  hora de inicializao do servidor
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public long getStartupTime() throws RemoteException;

  /**
   * Obtm estatsticas de logins.
   * 
   * @param succeeded <code>true</code> para receber as estatsticas de logins
   *        bem-sucedidos, <code>false</code> para receber as estatsticas de
   *        falhas de autenticao
   * @return estatsticas de logins bem-sucedidos ou de falhas de autenticao
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public Map<String, Integer> getLoginStats(boolean succeeded)
    throws RemoteException;
}
