package csbase.logic;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.FileLockedException;
import tecgraf.ftc.common.exception.MaxClientsReachedException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.common.logic.RemoteFileChannel;
import tecgraf.ftc.common.logic.RemoteFileChannelImpl;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Abstrao de um canal que exporta os mtodos padro de RemoteFileChannel e
 * acrescenta variantes de mtodos com prefixo "sync" que tentam realizar a
 * transferncia completa dos dados em questo, ou lanam uma exceo caso no
 * consigam.
 * 
 * @author Tecgraf
 */
public class SyncRemoteFileChannel implements RemoteFileChannel {
  /**
   * Tempo de espera, em ms, entre falhas de transferncia.
   */
  private static final long FAILURE_WAIT = 500;

  /**
   * Nmero mximo de falhas de transferncia antes de lanar exceo.
   */
  private static final int MAX_FAILURES = 20;

  /**
   * O identificador do arquivo remoto, usado nas excees.
   */
  private byte[] id = null;

  /**
   * O canal implementado pelo FTC para transferncia.
   */
  private RemoteFileChannel channel = null;

  /**
   * Constri um canal para acesso a um arquivo remoto.
   * 
   * @param info Informaes que descrevem o canal para um arquivo remoto.
   */
  public SyncRemoteFileChannel(RemoteFileChannelInfo info) {
    id = info.getIdentifier();
    channel = new RemoteFileChannelImpl(info);
  }

  /**
   * Constri um canal para acesso a um arquivo remoto.
   * 
   * @param id O identificador do arquivo.
   * @param writable Indica se a escrita  permitida no arquivo.
   * @param host A mquina do servidor de arquivos.
   * @param port A porta do servidor de arquivos.
   * @param key A chave de acesso ao arquivo.
   */
  public SyncRemoteFileChannel(byte[] id, boolean writable, String host,
    int port, byte[] key) {
    this.id = id;
    channel = new RemoteFileChannelImpl(id, writable, host, port, key);
  }

  /**
   * Constri um canal para acesso a um arquivo remoto.
   * 
   * @param channel O canal sobre o qual este ir atuar.
   */
  public SyncRemoteFileChannel(RemoteFileChannel channel) {
    this.id = channel.toString().getBytes();
    this.channel = channel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean open(boolean readOnly) throws PermissionException,
    FileNotFoundException, FailureException, MaxClientsReachedException {
    //DEBUG
    User user = User.getLoggedUser();
    String realUser = (String) user.getAttribute("realUser");
    realUser = realUser != null ? ", realUser: " + realUser : "";
    String dmsg = String.format("[SRFC] op: OPEN, ulogin: %s, uname: %s%s, channelId: %s", user.getLogin(), user
      .getName(), realUser, javax.xml.bind.DatatypeConverter.printHexBinary(this.id));
    StackTraceElement[] trace = Thread.currentThread().getStackTrace();
    System.out.println("--------------------------------------------------------------------------------");
    System.out.println(dmsg + "\n\t" + java.util.Arrays.toString(trace));
    //END DEBUG
    return channel.open(readOnly);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOpen() {
    return channel.isOpen();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws FailureException {
    channel.close();
    //DEBUG
    System.out.println("[SRFC] op: CLOSE, channelId: " + javax.xml.bind.DatatypeConverter.printHexBinary(this.id));
    //END DEBUG
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getPosition() {
    return channel.getPosition();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPosition(long position) throws FailureException {
    channel.setPosition(position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getSize() {
    return channel.getSize();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSize(long size) throws PermissionException, FailureException {
    channel.setSize(size);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] target) throws FailureException {
    // Seria:
    //   return channel.read(target);
    // Mas estamos tratando aqui o erro [FTC-13].
    return this.read(target, 0, target.length, 0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] target, long position) throws FailureException {
    // Seria:
    //   return channel.read(target, position);
    // Mas estamos tratando aqui o erro [FTC-13].
    return this.read(target, 0, target.length, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] target, int offset, int length)
    throws FailureException {
    // Seria:
    //   return channel.read(target, offset, length);
    // Mas estamos tratando aqui o erro [FTC-13].
    return this.read(target, 0, target.length, 0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] target, int offset, int length, long position)
    throws FailureException {
    // S  necessrio limitar em avail por causa do erro [FTC-13].
    int avail = (int) Math.min(length, channel.getSize() - position);
    return channel.read(target, offset, avail, position);
  }

  /**
   * L uma sequncia de bytes do arquivo. Esse mtodo tenta fazer leituras
   * consecutivas at conseguir a quantidade de bytes solicitada, ou atingir o
   * final do arquivo. Caso no consiga recuperar a quantidade solicitada sem
   * ter atingido o final do arquivo (isto , por problemas de comunicao ou
   * outros), o mtodo lana uma exceo.
   * 
   * @param target O array a ser preenchido.
   * @param offset O deslocamento no array a partir do qual os bytes sero
   *        armazenados.
   * @param length A quantidade de bytes a serem lidos do arquivo.
   * @param position A posio do arquivo a partir da qual a leitura deve ser
   *        iniciada.
   * @return A quantidade de bytes lidos. Se for menor que o nmero de bytes
   *         solicitados, o final do arquivo foi atingido.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  public int syncRead(byte[] target, int offset, int length, long position)
    throws FailureException {
    long size = channel.getSize();
    if (size > -1) {
      length = (int) Math.min(length, size - position);
    }
    int transfered = 0;
    int numFailures = 0;
    while (transfered < length) {
      int part =
        channel.read(target, offset + transfered, length - transfered, position
          + transfered);
      transfered += part;
      if (part == 0) {
        if (++numFailures > MAX_FAILURES) {
          throw new FailureException(
            "Sem retorno do servidor ao solicitar dados do arquivo "
              + new String(id));
        }
        try {
          Thread.sleep(FAILURE_WAIT);
        }
        catch (InterruptedException e) {
          // do nothing
        }
      }
    }
    return transfered;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(byte[] source) throws PermissionException, FailureException,
    FileLockedException {
    return channel.write(source);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(byte[] source, long position) throws PermissionException,
    FailureException, FileLockedException {
    return channel.write(source, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(byte[] source, int offset, int length)
    throws PermissionException, FailureException, FileLockedException {
    return channel.write(source, offset, length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(byte[] source, int offset, int length, long position)
    throws PermissionException, FailureException, FileLockedException {
    return channel.write(source, offset, length, position);
  }

  /**
   * Escreve uma sequncia de bytes do arquivo. Esse mtodo tenta fazer escritas
   * consecutivas at conseguir enviar a quantidade de bytes solicitada. Caso
   * no consiga, o mtodo lana uma exceo.
   * 
   * @param source O array contendo os bytes a serem escritos no arquivo.
   * @param offset O deslocamento no array a partir do qual os bytes sero
   *        lidos.
   * @param length A quantidade de bytes a serem escritos do arquivo.
   * @param position A posio do arquivo a partir da qual a escrita deve ser
   *        iniciada.
   * 
   * @throws PermissionException em caso de erro de permisso
   * @throws FailureException em caso de falha no procedimento.
   * @throws FileLockedException no caso do arquivo estar bloqueado.
   */
  public void syncWrite(byte[] source, int offset, int length, long position)
    throws PermissionException, FailureException, FileLockedException {
    int transfered = 0;
    int numFailures = 0;
    while (transfered < length) {
      int part =
        channel.write(source, offset + transfered, length - transfered,
          position + transfered);
      transfered += part;
      if (part == 0) {
        if (++numFailures > MAX_FAILURES) {
          throw new FailureException(
            "Sem retorno do servidor ao enviar dados para o arquivo "
              + new String(id));
        }
        try {
          Thread.sleep(FAILURE_WAIT);
        }
        catch (InterruptedException e) {
          // do nothing
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferTo(long position, long count, OutputStream target)
    throws FailureException {
    return channel.transferTo(position, count, target);
  }

  /**
   * Transfere os dados do arquivo para um fluxo de sada. Esse mtodo tenta
   * fazer leituras consecutivas do arquivo at conseguir a quantidade de bytes
   * solicitada, ou atingir o final do arquivo. Caso no consiga recuperar a
   * quantidade solicitada sem ter atingido o final do arquivo (isto , por
   * problemas de comunicao ou outros), o mtodo lana uma exceo.
   * 
   * @param position A posio inicial a partir da qual o arquivo ser lido.
   * @param count A quantidade de bytes que sero transferidos.
   * @param target O fluxo de sada que receber os dados do arquivo.
   * @return O nmero de bytes lidos do arquivo e enviados para o fluxo de
   *         sada. S ser menor que count caso o final do arquivo seja
   *         atingido antes.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  public long syncTransferTo(long position, long count, OutputStream target)
    throws FailureException {
    count = Math.min(count, getSize() - position);
    long transfered = 0;
    int numFailures = 0;
    while (transfered < count) {
      long part =
        channel.transferTo(position + transfered, count - transfered, target);
      transfered += part;
      if (part == 0) {
        if (++numFailures > MAX_FAILURES) {
          throw new FailureException(
            "Sem retorno do servidor ao enviar dados do arquivo "
              + new String(id));
        }
        try {
          Thread.sleep(FAILURE_WAIT);
        }
        catch (InterruptedException e) {
          // do nothing
        }
      }
    }
    return transfered;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferTo(long position, long count, RemoteFileChannel target)
    throws PermissionException, FailureException, FileLockedException {
    return channel.transferTo(position, count, target);
  }

  /**
   * Transfere os dados do arquivo para um outro arquivo remoto. Esse mtodo
   * tenta fazer leituras consecutivas do arquivo at conseguir a quantidade de
   * bytes solicitada, ou atingir o final do arquivo. Caso no consiga recuperar
   * a quantidade solicitada sem ter atingido o final do arquivo (isto , por
   * problemas de comunicao ou outros), o mtodo lana uma exceo.
   * 
   * @param position A posio inicial a partir da qual o arquivo ser lido.
   * @param count A quantidade de bytes que sero transferidos.
   * @param target O canal do arquivo que receber os dados.
   * @return O nmero de bytes lidos do arquivo e enviados para o arquivo de
   *         sada remoto. S ser menor que count caso o final do arquivo seja
   *         atingido antes.
   * 
   * @throws PermissionException Caso o outro arquivo remoto tenha sido aberto
   *         somente para leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Caso o outro arquivo remoto esteja reservado
   *         para outro usurio.
   */
  public long syncTransferTo(long position, long count, RemoteFileChannel target)
    throws PermissionException, FailureException, FileLockedException {
    count = Math.min(count, getSize() - position);
    long transfered = 0;
    int numFailures = 0;
    while (transfered < count) {
      long part =
        channel.transferTo(position + transfered, count - transfered, target);
      transfered += part;
      if (part == 0) {
        if (++numFailures > MAX_FAILURES) {
          throw new FailureException(
            "Sem retorno do servidor ao enviar dados do arquivo "
              + new String(id));
        }
        try {
          Thread.sleep(FAILURE_WAIT);
        }
        catch (InterruptedException e) {
          // do nothing
        }
      }
    }
    return transfered;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferFrom(InputStream source, long position, long count)
    throws PermissionException, FailureException, FileLockedException {
    return channel.transferFrom(source, position, count);
  }

  /**
   * Transfere os dados de um fluxo de entrada para o arquivo. Esse mtodo tenta
   * fazer escritas consecutivas no arquivo at conseguir enviar a quantidade de
   * bytes solicitada. Caso no consiga enviar a quantidade solicitada sem ter
   * atingido o final do fluxo de entrada (isto , por problemas de comunicao
   * ou outros), o mtodo lana uma exceo.
   * 
   * @param source O fluxo de entrada.
   * @param position A posio inicial a partir da qual o arquivo ser escrito.
   * @param count A quantidade de bytes que sero transferidos.
   * @return O nmero de bytes lidos do fluxo de entrada e escritos no arquivo.
   *         S ser menor que count caso o final do fluxo de entrada seja
   *         atingido antes.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  public long syncTransferFrom(InputStream source, long position, long count)
    throws PermissionException, FailureException, FileLockedException {
    long transfered = 0;
    int numFailures = 0;
    while (transfered < count) {
      long part =
        channel.transferFrom(source, position + transfered, count - transfered);
      transfered += part;
      if (part == 0) {
        int b = 0;
        try {
          b = source.read();
        }
        catch (IOException e) {
          throw new FailureException(e);
        }
        if (b == -1) {
          return transfered;
        }
        else {
          byte[] barray = new byte[1];
          barray[0] = (byte) (b & 0xFF);
          this.syncWrite(barray, 0, 1, position + transfered);
          transfered++;
        }
        if (part == 0) {
          if (++numFailures > MAX_FAILURES) {
            throw new FailureException(
              "Sem retorno do servidor ao enviar para o arquivo "
                + new String(id));
          }
          try {
            Thread.sleep(FAILURE_WAIT);
          }
          catch (InterruptedException e) {
            // do nothing
          }
        }
      }
    }
    return transfered;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferFrom(RemoteFileChannel source, long position, long count)
    throws PermissionException, FailureException, FileLockedException {
    return channel.transferFrom(source, position, count);
  }

  /**
   * Transfere os dados de um outro arquivo remoto para o arquivo. Esse mtodo
   * tenta fazer escritas consecutivas no arquivo at conseguir enviar a
   * quantidade de bytes solicitada. Caso no consiga enviar a quantidade
   * solicitada sem ter atingido o final do arquivo remoto de entrada (isto ,
   * por problemas de comunicao ou outros), o mtodo lana uma exceo.
   * 
   * @param source O fluxo de entrada.
   * @param position A posio inicial a partir da qual o arquivo ser escrito.
   * @param count A quantidade de bytes que sero transferidos.
   * @return O nmero de bytes lidos do fluxo de entrada e escritos no arquivo.
   *         S ser menor que count caso o final do fluxo de entrada seja
   *         atingido antes.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  public long syncTransferFrom(RemoteFileChannel source, long position,
    long count) throws PermissionException, FailureException,
    FileLockedException {
    count = Math.max(count, source.getSize()-source.getPosition());
    long transfered = 0;
    int numFailures = 0;
    while (transfered < count) {
      long part =
        channel.transferFrom(source, position + transfered, count - transfered);
      transfered += part;
      if (part == 0) {
        if (++numFailures > MAX_FAILURES) {
          throw new FailureException(
            "Sem retorno do servidor ao enviar para o arquivo "
              + new String(id));
        }
        try {
          Thread.sleep(FAILURE_WAIT);
        }
        catch (InterruptedException e) {
          // do nothing
        }
      }
    }
    return transfered;
  }
}
