package csbase.logic.algorithms.parsers;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.HiddenParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parsers.elements.IElementStructure;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedElement;
import csbase.logic.algorithms.parsers.elements.ParsedParameter;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.StringAttribute;

/**
 * <p>
 * Analisador de {@link HiddenParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo oculto. O elemento corrente
 * do {@link XmlParser analisador de XML} precisa ser um elemento
 * {@link HiddenParameter}.
 * </p>
 */
public class HiddenParameterParser implements ParameterFactory {

  /**
   * <p>
   * O elemento {@value #HIDDEN_PARAMETER_ELEMENT}: descreve as propriedades de
   * um {@link HiddenParameter parmetro do tipo oculto}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  static final String HIDDEN_PARAMETER_ELEMENT = "oculto";

  /**
   * <p>
   * O atributo {@value #HIDDEN_PARAMETER_ELEMENT_VALUE_ATTRIBUTE} do elemento
   * {@link #HIDDEN_PARAMETER_ELEMENT}. Indica o valor do parmetro. 
   * obrigatrio e tipo  string.
   * </p>
   */
  private static final String HIDDEN_PARAMETER_ELEMENT_VALUE_ATTRIBUTE =
    "valor";

  /**
   * {@inheritDoc}
   */
  @Override
  public HiddenParameter createParameter(XmlParser parser,
    String defaultCommandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    ParameterStructure<HiddenParameter> structure = getHiddenStructure();
    Map<IElementAttribute<?>, Object> values = new HashMap<>();
    for (IElementAttribute<?> attribute : structure.getAttributes()) {
      values.put(attribute, parser.extractAttribute(attribute));
    }
    parser.checkAttributes();
    Map<IElementAttribute<?>, Object> attributeValues = values;
    Map<IElementStructure<?>, List<ParsedElement>> children = parser
      .extractChildren(structure.getChildElements());
    ParsedParameter definition = new ParsedParameter(attributeValues, children,
      defaultCommandLinePattern);
    String name = definition.getName();
    String attributeValue = definition.getAttributeValue(
      HIDDEN_PARAMETER_ELEMENT_VALUE_ATTRIBUTE);
    String commandLinePattern = definition.getCommandLinePattern();
    if (commandLinePattern == null) {
      throw new ParseException(
        "No  possvel criar um parmetro oculto com formato no comando "
          + "vazio.\n" + "Parmetro: {0}\nValor: {1}.", name, attributeValue);
    }
    return new HiddenParameter(name, attributeValue, commandLinePattern);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ParameterStructure<?>> getParameterStructures() {
    ParameterStructure<?> structure = getHiddenStructure();
    return Collections.singletonList(structure);
  }

  /**
   * Obtm a estrutura do elemento oculto.
   *
   * @return a estrutura do elemento.
   */
  private ParameterStructure<HiddenParameter> getHiddenStructure() {
    ParameterStructure<HiddenParameter> structure = new ParameterStructure<>(
      HIDDEN_PARAMETER_ELEMENT, HiddenParameter.class);
    structure.addAttribute(new StringAttribute(
      HIDDEN_PARAMETER_ELEMENT_VALUE_ATTRIBUTE));
    return structure;

  }

}
