package csbase.logic.algorithms.xml.algorithmspack;

import java.io.IOException;
import java.io.Writer;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLElementInterface;

/**
 * Esta classe implementa o elemento XML que representa uma verso do algoritmo.
 * 
 */
public class XMLVersionElement extends XMLBasicElement {

  /** Mantm a descrio da verso corrente do algoritmo */
  private String description;

  /** Mantm os valores das propriedades da verso corrente do algoritmo */
  private Hashtable<String, String> propertiesValues;

  /**
   * Constri o elemento xml que descreve uma verso do algoritmo.
   */
  public XMLVersionElement() {
    super();
    setTag(XmlAlgorithmsPackUtils.VERSION_TAG);
  }

  /**
   * Construtor utilizado no processo de escrita do arquivo xml com as verses
   * dos algoritmos.
   * 
   * @param description descrio da verso do algoritmo
   * @param propertiesValues valores das propriedades da verso do algoritmo
   * 
   */
  public XMLVersionElement(String description,
    Hashtable<String, String> propertiesValues) {
    this();
    this.description = description;
    this.propertiesValues = propertiesValues;
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  @Override
  public void endTag(List<XMLElementInterface> childrenList) {
    ImprovedXMLListIterator iterator =
      new ImprovedXMLListIterator(childrenList);
    XMLElementInterface nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    while (nextElem != null) {
      //L a descrio da verso do algoritmo
      if (isDescElement(nextElem)) {
        description = nextElem.getStrValue();
      }

      //L as propriedades do algoritmo
      if (isVersionPropertiesElement(nextElem)) {
        propertiesValues =
          ((XMLPropertySetElement) nextElem).getPropertyValues();
      }

      nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(Writer arg0, String arg1) throws IOException {
    // TODO Auto-generated method stub

  }

  /**
   * Verifica se  o elemento que representa a descrio da verso do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento  a descrio da verso do algoritmo,
   *         caso contrrio, retorna false
   */
  private boolean isDescElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.DESCRIPTION_TAG);
  }

  /**
   * Verifica se  o elemento que representa as propriedades do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento representa as propriedades do algoritmo,
   *         caso contrrio, retorna false
   */
  private boolean isVersionPropertiesElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.PROPERTIES_TAG);
  }

  /**
   * Obtm os valores das propriedades da verso do algoritmo.
   * 
   * @return os valores das propriedades da verso do algoritmo
   */
  public Map<String, String> getProperties() {
    return propertiesValues;
  }

  /**
   * Obtm a descrio da verso do algoritmo.
   * 
   * @return a descrio da verso do algoritmo
   */
  public String getDescription() {
    return description;
  }

}
