package csbase.logic.algorithms.parsers.elements.attributes;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.elements.ParsedElement;

/**
 * Atributo do tipo inteiro.
 */
public class IntegerAttribute extends AbstractElementAttribute<Integer> {

  /**
   * Atributo que define o valor mnimo deste atributo.
   */
  private IntegerAttribute minimumValueAttribute;
  /**
   * Atributo que define o valor mximo deste atributo.
   */
  private IntegerAttribute maximumValueAttribute;
  /**
   * Valor mnimo.
   */
  private Integer minimumValue;
  /**
   * Valor mximo.
   */
  private Integer maximumValue;

  /**
   * Construtor.
   *
   * @param name nome do atributo.
   * @param isOptional indicativo de opcionalidade.
   * @param defaultValue valor padro do atributo.
   * @param defaultValueAttribute atributo com valor padro.
   * @param minimumValue valor mnimo do atributo.
   * @param minimumValueAttribute atributo que define o valor mnimo deste
   * atributo.
   * @param maximumValue valor mximo do atributo.
   * @param maximumValueAttribute atributo que define o valor mximo deste
   * atributo.
   */
  protected IntegerAttribute(String name, boolean isOptional, Integer defaultValue,
    IntegerAttribute defaultValueAttribute, Integer minimumValue,
    IntegerAttribute minimumValueAttribute, Integer maximumValue,
    IntegerAttribute maximumValueAttribute) {
    super(name, Integer.class, isOptional, defaultValue, defaultValueAttribute);
    this.minimumValue = minimumValue;
    this.maximumValue = maximumValue;
    this.minimumValueAttribute = minimumValueAttribute;
    this.maximumValueAttribute = maximumValueAttribute;
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name nome do atributo.
   * @param defaultValue valor padro do atributo.
   * @param minimumValue valor mnimo do atributo.
   * @param maximumValue valor mximo do atributo.
   */
  public IntegerAttribute(String name, Integer defaultValue,
    Integer minimumValue, Integer maximumValue) {
    this(name, true, defaultValue, null, minimumValue, null, maximumValue,
      null);
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name nome do atributo.
   * @param defaultValue valor padro do atributo.
   */
  public IntegerAttribute(String name, Integer defaultValue) {
    this(name, defaultValue, null, null);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   * @param minimumValue valor mnimo do atributo.
   * @param maximumValue valor mximo do atributo.
   */
  public IntegerAttribute(String name, Integer minimumValue,
    Integer maximumValue) {
    this(name, false, null, null, minimumValue, null, maximumValue, null);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   */
  public IntegerAttribute(String name) {
    this(name, null, null);
  }

  /**
   * Atribui o valor mnimo do atributo.
   *
   * @param minimumValue o valor mnimo.
   */
  public void setMinimumValue(Integer minimumValue) {
    this.minimumValue = minimumValue;
  }

  /**
   * Atribui o valor mximo do atributo.
   *
   * @param maximumValue o valor mximo.
   */
  public void setMaximumValue(Integer maximumValue) {
    this.maximumValue = maximumValue;
  }

  /**
   * Define o atributo que determina o valor mximo deste atributo.
   *
   * @param maximumValueAttribute o valor mximo.
   */
  public void setMaximumValueAttribute(IntegerAttribute maximumValueAttribute) {
    this.maximumValueAttribute = maximumValueAttribute;
  }

  /**
   * Define o atributo que determina o valor mnimo deste atributo.
   *
   * @param minimumValueAttribute o valor mnimo.
   */
  public void setMinimumValueAttribute(IntegerAttribute minimumValueAttribute) {
    this.minimumValueAttribute = minimumValueAttribute;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean validate(
    ParsedElement definition) throws ParseException {
    return super.validate(definition) && validateMaximum(
      definition) && validateMinimum(definition);
  }

  /**
   * Valida o valor mnimo do atributo, seja por um valor definido ou seja por
   * um valor que seja determinado por outro atributo.
   *
   * @param definition o elemento lido.
   * @return verdadeiro se a validao for bem sucedida ou falso, caso
   * contrrio.
   * @throws ParseException em caso de erro.
   */
  private boolean validateMinimum(
    ParsedElement definition) throws ParseException {
    Integer integerValue = definition.getAttributeValue(getName());
    if (integerValue != null) {
      if (minimumValueAttribute != null) {
        this.minimumValue =
          definition.getAttributeValue(minimumValueAttribute.getName());
      }
      if (minimumValue != null) {
        if (integerValue < minimumValue) {
          throw new ParseException(
            "O valor do atributo {1} est abaixo do valor mnimo permitido" +
              ".\nValor " + "encontrado: ({2}).\nValor mnimo: {3}.\n",
            getName(), integerValue, minimumValue);
        }
      }
    }
    return true;
  }

  /**
   * Valida o valor mximo do atributo, seja por um valor definido ou seja por
   * um valor que seja determinado por outro atributo.
   *
   * @param definition o elemento lido.
   * @return verdadeiro se a validao for bem sucedida ou falso, caso
   * contrrio.
   * @throws ParseException em caso de erro.
   */
  private boolean validateMaximum(
    ParsedElement definition) throws ParseException {
    Integer integerValue = definition.getAttributeValue(getName());
    if (integerValue != null) {
      if (maximumValueAttribute != null) {
        this.maximumValue =
          definition.getAttributeValue(maximumValueAttribute.getName());
      }
      if (maximumValue != null) {
        if (integerValue > maximumValue) {
          throw new ParseException(
            "O valor do atributo {1} est acima do valor mximo permitido" +
              ".\nValor " + "encontrado: ({2}).\nValor mximo: {3}.\n",
            getName(), integerValue, maximumValue);
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StringToValueConverter<Integer> getValueConverter() {
    return new StringToIntegerConverter();
  }

  /**
   * Conversor de valores inteiros para string.
   */
  private class StringToIntegerConverter implements StringToValueConverter<Integer> {

    /**
     * {@inheritDoc}
     */
    @Override
    public Integer valueOf(String value) throws ParseException {
      if (value == null) {
        return null;
      }
      Integer integerValue;
      try {
        integerValue = new Integer(value);
      }
      catch (final NumberFormatException e) {
        throw new ParseException(e,
          "O valor do atributo {1} deveria ser um nmero inteiro.\nValor " +
            "encontrado: ({2}).",
          getName(), value);
      }
      return integerValue;
    }
  }
}