/*
 * Created on Feb 25, 2005
 * 
 * $Id: Utilities.java 140981 2013-05-15 14:47:23Z oikawa $
 */
package csbase.logic;

import java.io.File;
import java.lang.reflect.Array;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Classe de mtodos utilitrios que podem ser usados tanto no cliente quanto no
 * servidor.
 * 
 * @author Leonardo Barros
 */
public class Utilities {

  private static final String DEFAULT_DATE_FORMAT = "dd/MM/yyyy - HH:mm:ss";

  /**
   * Transforma a representao de um caminho, a partir de um array de String
   * para uma nica String contendo separadores de diretrio.
   * 
   * @param pathArray o caminho do arquivo.
   * 
   * @return uma string com o path completo.
   */
  public static String makeRelativePath(String[] pathArray) {
    if ((pathArray == null)) {
      return "";
    }
    StringBuilder path = new StringBuilder();
    for (int i = 0; i < (pathArray.length - 1); i++) {
      path.append(pathArray[i]);
      path.append(File.separatorChar);
    }
    if (pathArray.length > 0) {
      path.append(pathArray[pathArray.length - 1]);
    }
    return path.toString();
  }

  /**
   * Idntico ao mtodo <code>makeRelativePath</code>, adicionando-se um
   * separador de diretrios no incio do caminho.
   * 
   * @param pathArray o caminho do arquivo.
   * 
   * @return uma string com o path completo.
   */
  public static String makePath(String[] pathArray) {
    if ((pathArray == null)) {
      return "";
    }
    return File.separator + Utilities.makeRelativePath(pathArray);
  }

  /**
   * Quebra um caminho da rvore de projetos no array de seus componentes.
   * 
   * @param pathText caminho a ser quebrado.
   * 
   * @return caminho quebrado em um array de seus componentes.
   */
  public static String[] splitProjectPath(String pathText) {
    return splitPath("/", pathText);
  }

  /**
   * Quebra um caminho no array de seus componentes.
   * 
   * @param sepPattern separador a ser utilizado.
   * @param pathText caminho a ser quebrado.
   * 
   * @return caminho quebrado em um array de seus componentes.
   */
  private static String[] splitPath(String sepPattern, String pathText) {
    sepPattern = sepPattern.replaceAll("\\\\", "\\\\\\\\");
    String[] path = pathText.split(sepPattern);
    int i = 0;
    while ((i < path.length) && path[i].equals("")) {
      i++;
    }
    if (i > 0) {
      String[] newPath = new String[path.length - i];
      System.arraycopy(path, i, newPath, 0, newPath.length);
      path = newPath;
    }
    return path;
  }

  /**
   * Retorna uma data formatada a partir de um long.
   * 
   * @param date data
   * 
   * @return String com a data representada por dd/MM/yyyy hh:mm:ss
   */
  public static String getFormattedDate(Date date) {
    return getFormattedDate(date, DEFAULT_DATE_FORMAT);
  }

  /**
   * Retorna uma data formatada a partir de um long.
   * 
   * @param timeInMillis data em milesegundos
   * 
   * @return String com a data representada por dd/MM/yyyy hh:mm:ss
   */
  public static String getFormattedDate(long timeInMillis) {
    Date date = new Date(timeInMillis);
    return getFormattedDate(date);
  }

  /**
   * Tenta converter uma <code>String</code> representando uma data no estilo
   * padro para uma <code>Date</code>.
   * 
   * @param formattedDate <code>String</code> a ser processada.
   * 
   * @return <code>Date</code> representando a <code>String</code> convertida.
   * 
   * @throws ParseException se o parmetro passado no for reconhecido como
   *         data.
   */
  public static Date parseDate(String formattedDate) throws ParseException {
    if (formattedDate == null) {
      throw new IllegalArgumentException("formattedDate == null");
    }
    String format = DEFAULT_DATE_FORMAT;
    DateFormat formatter = new SimpleDateFormat(format);
    return formatter.parse(formattedDate);
  }

  /**
   * Retorna uma data no formato desejado a partir de um long.
   * 
   * @param timeInMillis data em milesegundos
   * @param format formato desejado
   * 
   * @return String com a data representada pelo formato desejado
   */
  public static String getFormattedDate(long timeInMillis, String format) {
    Date date = new Date(timeInMillis);
    return getFormattedDate(date, format);
  }

  /**
   * Retorna uma data no formato desejado a partir de um long.
   * 
   * @param date data
   * @param format formato desejado
   * 
   * @return String com a data representada pelo formato desejado
   */
  public static String getFormattedDate(Date date, String format) {
    SimpleDateFormat sdf = new SimpleDateFormat(format);
    return sdf.format(date);
  }

  /**
   * Remove o elemento de ndice especificado de um array.
   * 
   * @param array array do qual ser removido o elemento
   * @param index ndice do elemento a ser removido
   * 
   * @return array sem o elemento especificado
   */
  public static Object[] removeItemFromArray(Object[] array, int index) {
    if (array == null || array.length == 0) {
      return null;
    }
    if (index < 0 || index > array.length - 1) {
      return null;
    }
    Class<?> arrayClass = array[0].getClass();
    Object[] newArray =
      (Object[]) Array.newInstance(arrayClass, array.length - 1);
    System.arraycopy(array, 0, newArray, 0, index);
    if (index < array.length - 1) {
      System.arraycopy(array, index + 1, newArray, index, array.length - 1
        - index);
    }
    return newArray;
  }
}
