/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Map;

import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.Validation;

/**
 * Parmetro de um configurador de algoritmos.
 * 
 * @param <V> O tipo do valor.
 * 
 */
public abstract class Parameter<V> implements Comparable<Parameter<V>>,
  Serializable {

  /**
   * Nome do parmetro.
   */
  private String name;

  /**
   * Construtor.
   * 
   * @param name nome do parmetro. (No aceita <code>null</code>)
   */
  protected Parameter(String name) {
    setName(name);
  }

  /**
   * Exporta o valor do parmetro como um mapa "nome{@code ->}valor".
   * 
   * @return O valor do parmetro como um mapa.
   */
  public abstract Map<String, Object> exportValue();

  /**
   * Monta a linha de comando referente a este parmetro.
   * 
   * @param context Contexto para gerao da linha de comando.
   * 
   * @return a linha de comando.
   */
  public abstract String getCommandLine(CommandLineContext context);

  /**
   * Importa o valor do parmetro a partir de um mapa "nome{@code ->}valor".
   * 
   * @param values O mapa com o valor do parmetro.
   */
  public abstract void importValue(Map<String, Object> values);

  /**
   * Determina se o parmetro est com o valor padro.
   * 
   * @return verdadeiro se o parmetro est com o valor padro ou falso, caso
   *         contrrio.
   */
  public abstract boolean isSetDefaultValue();

  /**
   * <p>
   * Exibe/oculta este parmetro.
   * </p>
   * 
   * @param isVisible {@code true} para exibir ou {@code false} para ocultar.
   * 
   * @return {@code true} se houver mudana de estado ou {@code false} se o
   *         estado atual foi igual ao estado esperado (exemplo: exibir um
   *         parmetro j visvel).
   */
  public abstract boolean setVisible(boolean isVisible);

  /**
   * Determina se o parmetro est visvel no configurador.
   * 
   * @return verdadeiro se o parmetro est visvel ou falso, caso contrrio.
   */
  public abstract boolean isVisible();

  /**
   * Atribui o valor padro ao parmetro.
   */
  public abstract void resetValue();

  /**
   * {@inheritDoc}
   */
  @Override
  public final int compareTo(Parameter<V> parameter) {
    return getName().compareTo(parameter.getName());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    Parameter<?> parameter = (Parameter<?>) obj;
    return getName().equals(parameter.getName());
  }

  /**
   * Obtm o nome do parmetro.
   * 
   * @return o nome.
   */
  public final String getName() {
    return this.name;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final int hashCode() {
    return getName().hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String toString() {
    return getName();
  }

  /**
   * Atribui o nome do parmetro. (No aceita <code>null</code>)
   * 
   * @param name o nome.
   */
  private void setName(String name) {
    if (name == null) {
      throw new IllegalArgumentException("name == null");
    }
    this.name = name;
  }

  /**
   * Verifica se os valores armazenados no parmetro so vlidos.
   * 
   * @param context contexto usado na validao.
   * 
   * @return O resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  public abstract Validation validate(ValidationContext context)
    throws RemoteException;

}
