/*
 * CommandPersistenceServiceInterface.java
 *
 * $Author:$ $Date:$ $Revision$
 */

package csbase.remote;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import csbase.exception.ServiceFailureException;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.commands.CommandPersistenceNotification;
import csbase.util.messages.MessageBroker;

/**
 * <p>
 * Interface do servio de persistncia de comandos.
 * </p>
 * Persiste os comandos no projeto corrente.
 *
 * @author TecGraf / PUC-Rio
 */
public interface CommandPersistenceServiceInterface extends ServiceInterface {

  /**
   * Nome do servio
   */
  String SERVICE_NAME = "CommandPersistenceService";

  /**
   * <p>
   * Obtm todas as informaes dos comandos salvas no projeto especificado.<br>
   * </p>
   * <p>
   * Diferente do {@link #getCommandInfos(Object)}, este mtodo retorna assim
   * que  chamado e os comandos so entregues, aos poucos, como de mensagens
   * disparadas atravs {@link MessageBroker} do servidor, a cada
   * <i>desiredResponseTime</i>. <br>
   * Os comandos so encapsulados em {@link CommandInfosRetrived} que chegam ao
   * usurio como corpo da mensagem. <br>
   * As mensagens so enviadas com tempo de vida igual a zero. Isso quer dizer
   * que para receber as mensagens, o usurio deve antes cadastrar um ouvinte
   * que filtre as mensagens com corpo do tipo {@link CommandInfosRetrived}
   * antes de chamar este mtodo.
   * </p>
   *
   * @param projectId O identificador do projeto (No pode ser {@code null}).
   * @param desiredResponseTime O tempo, em milisegundos, que o cliente gostaria
   *        de esperar para cada pacote de comandos.
   * @return total de comandos que sero carregados.
   *
   * @throws IllegalArgumentException Caso projectId seja nulo, ou o
   *         desiredResponseTime seja menor que 1.
   * @throws ServiceFailureException Caso no seja possvel recuperar o projeto.
   * @throws RemoteException Em caso de erro de RMI.
   */
  long requestCommandInfos(Object projectId, final long desiredResponseTime)
    throws RemoteException;

  /**
   * Obtm todas as informaes dos comandos salvas no projeto especificado.
   *
   * @param projectId O identificador do projeto (No pode ser {@code null}).
   *
   * @return O conjunto de informaes no-modificvel (
   *         {@link Collections#unmodifiableSet(Set)}). Se no houver
   *         informaes, o conjunto retornado estar vazio. Caso ocorra algum
   *         erro na tentativa de ler as informaes de um comando, este ser
   *         representado por um comando com o status
   *         {@link CommandStatus#SYSTEM_FAILURE}. Neste as nicas propriedades
   *         garantidas so o identificador do comando e o identificador do
   *         projeto, todas as demais devem ser desconsideradas.
   *
   * @throws IllegalArgumentException Caso o rojectId seja nulo.
   * @throws ServiceFailureException Caso no seja possvel recuperar o projeto.
   * @throws RemoteException Em caso de erro de RMI.
   */
  Set<CommandInfo> getCommandInfos(Object projectId) throws RemoteException;

  /**
   * Obtm as informaes de comando especfico dado o identificador do projeto
   * e o identificador do comando.
   *
   * @param projectId O identificador do projeto (No pode ser {@code null}).
   * @param commandId O identificador do comando (No pode ser {@code null}).
   *
   * @return As informaes do comando, {@code null} se ele no existir ou, caso
   *         no seja possvel carregar as informaes do comando por algum
   *         erro, ser retornado um comando com o status
   *         {@link CommandStatus#SYSTEM_FAILURE}. Neste as nicas propriedades
   *         garantidas so o identificador do comando e o identificador do
   *         projeto, todas as demais devem ser desconsideradas.
   *
   * @throws IllegalArgumentException Caso algum dos parmetros seja nulo.
   * @throws RemoteException Em caso de erro de RMI.
   */
  CommandInfo getCommandInfo(Object projectId, String commandId)
    throws RemoteException;

  /**
   * <p>
   * Remove o comando especfico.
   * </p>
   * <p>
   * Caso a remoo tenha sido feita com sucesso, gera uma notificao
   * {@link CommandPersistenceNotification} do tipo
   * {@link csbase.logic.algorithms.commands.CommandPersistenceNotification.Type#REMOVED}
   * .
   * </p>
   *
   * @param projectId Identificador do projeto aonde o comando foi persistido
   *        (No pode ser {@code null}).
   * @param commandId Identificador do comando (No pode ser {@code null}).
   *
   * @return {@code false} e o comando no existir ou {@code true} em caso de
   *         sucesso.
   *
   * @throws ServiceFailureException Em caso de erro ao tentar ler as
   *         propriedades. Por ex.: erro de IO ao ler do dispositivo de
   *         persistncia.
   * @throws RemoteException Em caso de erro de RMI.
   */
  boolean removeCommandInfo(Object projectId, String commandId)
    throws RemoteException;

  /**
   * <p>
   * Remove comandos.
   * </p>
   * <p>
   * Para cada remoo feita com sucesso, gera uma notificao
   * {@link CommandPersistenceNotification} do tipo
   * {@link csbase.logic.algorithms.commands.CommandPersistenceNotification.Type#REMOVED}
   * .
   * </p>
   *
   * @param projectIds lista de identificador dos projetos respectivos a lista
   *        de identificadores de comandos (No pode ser {@code null}).
   * @param commandIds lista de identificador dos comandos a serem removidos
   *        (No pode ser {@code null}).
   *
   * @return uma lista de booleanos indicando o sucesso {@code true} da remoo
   *         de cada comando.
   *
   * @throws ServiceFailureException Em caso de erro ao tentar ler as
   *         propriedades. Por ex.: erro de IO ao ler do dispositivo de
   *         persistncia.
   * @throws RemoteException Em caso de erro de RMI.
   */
  boolean[] removeCommandInfos(List<Object> projectIds, List<String> commandIds)
    throws RemoteException;

  /**
   * Obtm todas as informaes dos comandos que esto em determinado estado,
   * salvas no projeto especificado.
   *
   * @param projectId O identificador do projeto (No pode ser {@code null}).
   * @param statusFilter indica os estados que se deseja obter.
   * @param include Quando verdadeiro indica que os estados informados devem ser
   *        usados para incluso dos comandos. Quando falso, os estados
   *        informados so usados para excluso dos mesmos.
   *
   * @return O conjunto de informaes no-modificvel (
   *         {@link Collections#unmodifiableSet(Set)}). Se no houver
   *         informaes, o conjunto retornado estar vazio. Caso ocorra algum
   *         erro na tentativa de ler as informaes de um comando, este ser
   *         representado por um comando com o status
   *         {@link CommandStatus#SYSTEM_FAILURE}. Neste as nicas propriedades
   *         garantidas so o identificador do comando e o identificador do
   *         projeto, todas as demais devem ser desconsideradas.
   *
   * @throws IllegalArgumentException Caso o rojectId seja nulo.
   * @throws ServiceFailureException Caso no seja possvel recuperar o projeto.
   * @throws RemoteException Em caso de erro de RMI.
   */
  Set<CommandInfo> getStatusCommandInfos(Object projectId,
    List<CommandStatus> statusFilter, boolean include) throws RemoteException;

  /**
   * <p>
   * Atualiza a decrio de um comando.
   * </p>
   *
   * <p>
   * Caso a atualizao tenha sido feita com sucesso, gera uma notificao
   * {@link CommandPersistenceNotification} do tipo
   * {@link csbase.logic.algorithms.commands.CommandPersistenceNotification.Type#UPDATED}
   * .
   * </p>
   *
   * @param projectId O identificador do projeto no qual o comando foi
   *        persistido (No aceita {@code null}).
   * @param commandId O identificador do comando (No aceita {@code null}).
   * @param description A nova descrio do comando (No aceita {@code null}).
   *
   * @throws ServiceFailureException Em caso de erro ao tentar escrever as
   *         informaes. Por ex.: erro de IO ao ler do dispositivo de
   *         persistncia.
   * @throws RemoteException Em caso de erro de RMI.
   */
  void updateCommandDescription(Object projectId, String commandId,
    String description) throws RemoteException;

  /**
   * Obtm o configurador de comando especfico dado o identificador do projeto
   * e o identificador do comando.
   *
   * @param projectId O identificador do projeto (No pode ser {@code null}).
   * @param commandId O identificador do comando (No pode ser {@code null}).
   *
   * @return O configurador do comando ou {@code null} se ele no existir.
   *
   * @throws ServiceFailureException Em caso de erro ao tentar ler o
   *         configurador. Por ex.: erro de IO ao ler do dispositivo de
   *         persistncia.
   * @throws RemoteException Em caso de erro de RMI.
   */
  AlgorithmConfigurator readAlgorithmConfigurator(Object projectId,
    String commandId) throws RemoteException;

  /**
   * Estrutura contendo informaes de leitura de comandos.
   *
   * @see CommandPersistenceServiceInterface#requestCommandInfos(Object, long)
   *
   * @author Tecgraf / PUC-Rio
   */
  final class CommandInfosRetrived implements Serializable {

    /** ltimos comandos lidos. */
    private final Set<CommandInfo> commands;
    /** Quantos comandos ainda falta ler. */
    private final long missing;
    /**
     * O total de comandos que sero lidos ao final da chamada ao mtodo
     * {@link CommandPersistenceServiceInterface#requestCommandInfos(Object, long)}
     * .
     */
    private final long total;

    /**
     * Construtor.
     *
     * @param commands ltimos comandos lidos.
     * @param missing Quantos comandos ainda falta ler.
     * @param total O total de comandos que sero lidos ao final da chamada ao
     *        mtodo
     *        {@link CommandPersistenceServiceInterface#requestCommandInfos(Object, long)}
     *        .
     */
    public CommandInfosRetrived(Set<CommandInfo> commands, long missing,
      long total) {
      this.commands = Collections.unmodifiableSet(commands);
      this.missing = missing;
      this.total = total;
    }

    /**
     * Obtm os ltimos comandos lidos.
     * 
     * @return ltimos comandos lidos.
     */
    public Set<CommandInfo> getCommandInfos() {
      return commands;
    }

    /**
     * Obtm quantos comandos ainda falta ler.
     * 
     * @return Quantos comandos ainda falta ler.
     */
    public long getMissing() {
      return missing;
    }

    /**
     * Obtm o total de comandos que sero lidos ao final da chamada ao mtodo
     * {@link CommandPersistenceServiceInterface#requestCommandInfos(Object, long)}
     * .
     *
     * @return O total de comandos que sero lidos ao final da chamada ao mtodo
     *         {@link CommandPersistenceServiceInterface#requestCommandInfos(Object, long)}
     *         .
     */
    public long getTotal() {
      return total;
    }
  }
}
