package csbase.remote;

import java.util.Date;
import java.util.Map;

import csbase.exception.ParseException;

/**
 * interface para controle da inicializacao dos servicos Rest
 *
 * @author rodneicouto
 *
 */
public interface RestServiceInterface extends ServiceInterface {
  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "RestService";


  /**
   * Retorna a URL externa do servidor.
   *
   * @return URL
   */
  public String getExternalURL();

  /**
   * Adiciona no log um registro de evento severo em decorrncia de um erro no
   * servidor causado por alguma requisio proveniente da API REST.
   *
   * @param msg a mensagem para o registro no log
   * @param t a exceo que causou o erro
   */
  public void logSevereMessage(String msg, Throwable t);

  /**
   * Adiciona no log um registro de evento de informao relacionada a alguma
   * requisio proveniente da API REST.
   *
   * @param msg a mensagem para o registro no log
   */
  public void logInfoMessage(String msg);

  /**
   * Recupera a informao associado ao token. Caso o token no esteja vlido,
   * retorna null.
   *
   * @param token o token
   * @param outAttributes Mapa usado para retorno dos atributos via parmetro.
   *        Se for diferentes de nulo, os atributos adiconais sero preenchidos
   *        nesse mapa.Se for nulo, mesmo havendo atributos adicionais, nao ser
   *        possvel l-los.
   * @return a Informao do Subject associada ao token ou null caso no exista.
   * @throws ParseException Caso o token seja invlido
   */
  public String parserToken(String token, Map<String, Object> outAttributes)
    throws ParseException;

  /**
   * Cria um token assinado
   *
   * @param subject Informacao do subject. Exemplo de uso  o login do usuario
   * @param attributes Atributos adicionais.
   * @param expirationDate Uma data de expirao
   * @param issuedDate Uma data de emisso
   * @return o token criado e assinado
   */
  public String createToken(String subject, Map<String, Object> attributes,
    Date expirationDate, Date issuedDate);

  /**
   * Obtm a URL do servidor HTTP do cliente com o identificador especificado.
   *
   * @param id identificador do cliente.
   * @return a URL do servidor HTTP do cliente.
   */
  public String getClientHttpServer(String id);

  /**
   * Registra uma URL de servidor HTTP do cliente com o identificador
   * especificado.
   *
   * @param id identificador do cliente.
   * @param httpServerURL a URL do servidor HTTP do cliente.
   */
  public void registerClientHttpServer(String id, String httpServerURL);

  /**
   * Desregistra uma URL de servidor HTTP do cliente com o identificador
   * especificado.
   *
   * @param id identificador do cliente.
   */
  public void unregisterClientHttpServer(String id);

  /**
   * Recupera uma flag da configurao de Client Developer Mode.
   * @return true caso o cliente esteja em modo desenvolvedor, ou false caso contrario
   */
  public boolean isClientDeveloperMode();
}
