package csbase.console;

import java.rmi.RemoteException;
import java.util.Map;
import java.util.Set;

import csbase.logic.BusInfo;
import csbase.logic.User;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diagnosticservice.ValidationStatus;
import csbase.remote.ClientRemoteLocator;
import scs.core.ComponentId;

/**
 * Ferramenta para diagnsticos de um servidor CSBase. A conexo com o servidor
 *  feita como admin, mediante senha obtida interativamente.
 * 
 * @author Tecgraf
 */
public class Diagnostics extends AbstractConsoleApp {
  /**
   * Popriedades runtime do servidor.
   */
  private Map<String, String> properties;

  /**
   * Construtor.
   * 
   * @param args parmetros fornecidos na linha de comando
   */
  public Diagnostics(String[] args) {
    super(args);
  }

  /**
   * Executa todos os diagnsticos em uma sequncia arbitrria.
   * 
   * @return <code>true</code> se todos os testes foram bem-sucedidos,
   *         <code>false</code> se algum deles falhou
   * @throws RemoteException erro na comunicao RMI
   */
  protected boolean runDiagnostics() throws RemoteException {
    boolean status = true;
    /*
     * informaes bsicas
     */
    showBasicInfo();
    /*
     * ambiente de instalao
     */
    println("\n" + getDeploymentInfo());
    /*
     * agora faremos os testes propriamente ditos
     */
    status &= runBasicNetworkDiagnostics();
    status &= runOpenBusDiagnostics();
    return status;
  }

  /**
   * Executa diagnsticos da conexo com o barramento.
   * 
   * @return <code>true</code> se todos os testes foram bem-sucedidos,
   *         <code>false</code> se algum deles falhou
   * @throws RemoteException erro na comunicao RMI
   */
  private boolean runOpenBusDiagnostics() throws RemoteException {
    Status status = ClientRemoteLocator.diagnosticService.getStatus("openbus");
    System.out.println("*** Status da conexo com o barramento Openbus");
    for (ValidationStatus validationStatus : status.getValidationStatusList()) {
      println(validationStatus.message);
    }
    // TODO obter o estado diretamente do OpenBusService?
    boolean enabled = getBooleanProperty("OpenBusService.enabled");
    println();
    /*
     * seo com informaes bsicas
     */
    InfoTable section = new InfoTable("Conexo com o barramento", false);
    if (!enabled) {
      section.add("status", "desabilitada");
      println(section.toString());
      return true;
    }
    boolean isActive = ClientRemoteLocator.openBusService.isActive();
    if (isActive) {
      section.add("status", "habilitada e ativa");
    }
    else {
      section.add("status", "habilitada e INATIVA", false);
    }
    BusInfo busInfo = ClientRemoteLocator.openBusService.getBusInfo();
    section.add("host", busInfo.getHost());
    section.add("porta", busInfo.getPort());

    println(section.toString());

    if (!isActive) {
      return false;
    }

    /*
     * seo com lista dos componentes registrados
     */
    section = new InfoTable("Componentes registrados no barramento", true);
    Set<ComponentId> components =
      ClientRemoteLocator.openBusService.getRegisteredComponents();
    if (components != null) {
      for (ComponentId componentId : components) {
        String version =
          String.format("v%d.%d.%d", componentId.major_version,
            componentId.minor_version, componentId.patch_version);
        section.add(componentId.name, version);
      }
      section.setOK(validateOpenBusComponents(components));
    }
    else {
      section.setOK(false);
    }

    println(section.toString());
    return section.isOK();
  }

  /**
   * Valida os componentes registrados. Este mtodo deve ser redefinido por
   * subclasses que precisem fazer validaes especficas (p.ex. garantir que
   * componentes especficos tenham sido registrados).
   * 
   * @param components conjunto dos componentes registrados
   * @return <code>true</code> se os componentes esto vlidos
   */
  protected boolean validateOpenBusComponents(Set<ComponentId> components) {
    return true;
  }

  /**
   * Obtm o valor de uma propriedade como booleano.
   * 
   * @param prop propriedade
   * @return valor da propriedade como booleano, ou <code>false</code> caso a
   *         propriedade no exista
   */
  private boolean getBooleanProperty(String prop) {
    return Boolean.valueOf(properties.get(prop));
  }

  /**
   * Executa diagnsticos bsicos da configurao de rede.
   * 
   * @return <code>true</code> se todos os testes foram bem-sucedidos
   */
  protected boolean runBasicNetworkDiagnostics() {
    return true;
  }

  //  /**
  //   * Obtm um {@link InetAddress} a partir de um IP.
  //   * 
  //   * @param host endereo IP
  //   * @return {@link InetAddress} associado ao endereo
  //   * @throws UnknownHostException
  //   */
  //  private InetAddress getHostByAddress(String host) throws UnknownHostException {
  //    byte[] ip = new byte[4];
  //    Scanner scanner = new Scanner(host).useDelimiter("\\.");
  //    ip[0] = scanner.nextByte();
  //    ip[1] = scanner.nextByte();
  //    ip[2] = scanner.nextByte();
  //    ip[3] = scanner.nextByte();
  //    scanner.close();
  //    return InetAddress.getByAddress(ip);
  //  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new BasicParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getLogin() {
    return (String) User.getAdminId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void postLoginInit() throws RemoteException {
    properties = getRuntimeProperties();
  }

  /**
   * Executa os diagnsticos a partir da linha de comando.
   * 
   * @param args argumentos provenientes da linha de comando
   * @throws RemoteException erro na comunicao RMI
   */
  public static void main(String[] args) throws RemoteException {
    /*
     * agora executamos os diagnsticos
     */
    Diagnostics diags = null;
    try {
      diags = new Diagnostics(args);
      /*
       * tentamos conexo com o servidor
       */
      if (diags.login()) {
        diags.runDiagnostics();
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      if (diags != null) {
        diags.logout();
      }
    }
  }
}
